"use client"
import React, { useState, useRef, useEffect } from "react"
import { Globe, Languages, MapPin, ChevronDown, Check } from "lucide-react"

// --- Custom Hook for clicking outside to close ---
function useOutsideClick(ref: React.RefObject<HTMLDivElement>, callback: () => void) {
  useEffect(() => {
    function handleClickOutside(event: MouseEvent) {
      if (ref.current && !ref.current.contains(event.target as Node)) {
        callback();
      }
    }
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, [ref, callback]);
}

// --- Custom Select Component ---
interface CustomDropdownProps {
  value: string;
  options: string[];
  onChange: (val: string) => void;
  icon: React.ReactNode;
}

const CustomDropdown = ({ value, options, onChange, icon }: CustomDropdownProps) => {
  const [isOpen, setIsOpen] = useState(false);
  const containerRef = useRef<HTMLDivElement>(null);
  useOutsideClick(containerRef, () => setIsOpen(false));

  return (
    <div className="relative w-full" ref={containerRef}>
      <button
        onClick={() => setIsOpen(!isOpen)}
        className="w-full flex items-center justify-between px-4 py-3 rounded-xl border transition-all duration-200 outline-none hover:bg-white/[0.02]"
        style={{ 
          backgroundColor: 'var(--input)', 
          borderColor: isOpen ? 'var(--primary)' : 'var(--border)',
          color: 'var(--foreground)'
        }}
      >
        <div className="flex items-center gap-3">
          <span className="opacity-60">{icon}</span>
          <span className="text-sm font-medium">{value}</span>
        </div>
        <ChevronDown 
          size={16} 
          className={`transition-transform duration-300 ${isOpen ? "rotate-180" : ""}`} 
        />
      </button>

      {isOpen && (
        <div 
          className="absolute z-50 w-full mt-2 rounded-xl border shadow-2xl animate-in fade-in zoom-in-95 duration-200 overflow-hidden"
          style={{ 
            backgroundColor: 'var(--popover)', 
            borderColor: 'var(--border)',
            boxShadow: '0 10px 40px -10px rgba(0,0,0,0.5)'
          }}
        >
          <div className="p-1">
            {options.map((option) => (
              <button
                key={option}
                onClick={() => {
                  onChange(option);
                  setIsOpen(false);
                }}
                className="w-full flex items-center justify-between px-3 py-2.5 rounded-lg text-sm transition-colors hover:bg-white/5 group"
              >
                <span className={value === option ? "font-bold text-white" : "opacity-70 group-hover:opacity-100"}>
                  {option}
                </span>
                {value === option && <Check size={14} className="text-white" />}
              </button>
            ))}
          </div>
        </div>
      )}
    </div>
  );
};

export default function LanguageRegion() {
  const [language, setLanguage] = useState("English");
  const [region, setRegion] = useState("Global");

  return (
    <div className="max-w-3xl mx-auto animate-in fade-in slide-in-from-bottom-4 duration-700">
      <div 
        className="rounded-2xl border transition-all duration-300"
        style={{ backgroundColor: 'var(--card)', borderColor: 'var(--border)' }}
      >
        <div className="p-6 border-b" style={{ borderColor: 'var(--border)' }}>
          <h2 className="text-xl font-bold flex items-center gap-2">
            <Globe size={20} style={{ color: 'var(--chart-2)' }} />
            Language & Region
          </h2>
        </div>

        <div className="p-6 space-y-6">
          <div className="space-y-2">
            <label className="text-xs font-bold uppercase tracking-widest opacity-40 ml-1">Display Language</label>
            <CustomDropdown 
              value={language} 
              options={["English", "Hindi", "Bengali"]} 
              onChange={setLanguage}
              icon={<Languages size={18} />}
            />
          </div>

          <div className="space-y-2">
            <label className="text-xs font-bold uppercase tracking-widest opacity-40 ml-1">Search Region</label>
            <CustomDropdown 
              value={region} 
              options={["Global", "India", "USA"]} 
              onChange={setRegion}
              icon={<MapPin size={18} />}
            />
          </div>
        </div>
      </div>
    </div>
  );
}