// YouTube API integration for video search
interface YouTubeSearchResult {
  kind: string
  etag: string
  id: {
    kind: string
    videoId: string
  }
  snippet: {
    publishedAt: string
    channelId: string
    title: string
    description: string
    thumbnails: {
      default: { url: string; width: number; height: number }
      medium: { url: string; width: number; height: number }
      high: { url: string; width: number; height: number }
    }
    channelTitle: string
    publishTime: string
  }
}

interface YouTubeApiResponse {
  kind: string
  etag: string
  nextPageToken?: string
  regionCode: string
  pageInfo: {
    totalResults: number
    resultsPerPage: number
  }
  items: YouTubeSearchResult[]
}

export interface YouTubeVideo {
  id: string
  title: string
  description: string
  thumbnail: string
  url: string
  channelTitle: string
  publishedAt: string
  duration?: string
}

export async function searchYouTubeVideos(
  query: string,
  maxResults = 12,
  safeSearch = true,
  pageToken?: string,
): Promise<{ videos: YouTubeVideo[]; nextPageToken?: string }> {
  const apiKey = process.env.YOUTUBE_API_KEY

  if (!apiKey) {
    console.warn("YouTube API key not configured, returning empty results")
    return { videos: [] }
  }

  try {
    const params = new URLSearchParams({
      part: "snippet",
      q: query,
      type: "video",
      maxResults: maxResults.toString(),
      key: apiKey,
      safeSearch: safeSearch ? "strict" : "none",
      order: "relevance",
      videoEmbeddable: "true",
      videoSyndicated: "true",
    })

    if (pageToken) {
      params.append("pageToken", pageToken)
    }

    const response = await fetch(`https://www.googleapis.com/youtube/v3/search?${params.toString()}`)

    if (!response.ok) {
      throw new Error(`YouTube API error: ${response.status} ${response.statusText}`)
    }

    const data: YouTubeApiResponse = await response.json()

    const videos: YouTubeVideo[] = data.items.map((item) => ({
      id: item.id.videoId,
      title: item.snippet.title,
      description: item.snippet.description,
      thumbnail: item.snippet.thumbnails.medium?.url || item.snippet.thumbnails.default.url,
      url: `https://www.youtube.com/watch?v=${item.id.videoId}`,
      channelTitle: item.snippet.channelTitle,
      publishedAt: item.snippet.publishedAt,
    }))

    return {
      videos,
      nextPageToken: data.nextPageToken,
    }
  } catch (error) {
    console.error("YouTube search error:", error)
    return { videos: [] }
  }
}

export async function getVideoDetails(videoIds: string[]): Promise<any[]> {
  const apiKey = process.env.YOUTUBE_API_KEY

  if (!apiKey || videoIds.length === 0) {
    return []
  }

  try {
    const params = new URLSearchParams({
      part: "contentDetails,statistics",
      id: videoIds.join(","),
      key: apiKey,
    })

    const response = await fetch(`https://www.googleapis.com/youtube/v3/videos?${params.toString()}`)

    if (!response.ok) {
      throw new Error(`YouTube API error: ${response.status} ${response.statusText}`)
    }

    const data = await response.json()
    return data.items || []
  } catch (error) {
    console.error("YouTube video details error:", error)
    return []
  }
}

// Helper function to format YouTube duration
export function formatDuration(duration: string): string {
  const match = duration.match(/PT(\d+H)?(\d+M)?(\d+S)?/)
  if (!match) return ""

  const hours = match[1] ? Number.parseInt(match[1].slice(0, -1)) : 0
  const minutes = match[2] ? Number.parseInt(match[2].slice(0, -1)) : 0
  const seconds = match[3] ? Number.parseInt(match[3].slice(0, -1)) : 0

  if (hours > 0) {
    return `${hours}:${minutes.toString().padStart(2, "0")}:${seconds.toString().padStart(2, "0")}`
  }
  return `${minutes}:${seconds.toString().padStart(2, "0")}`
}

// Helper function to format relative time
export function formatRelativeTime(dateString: string): string {
  const date = new Date(dateString)
  const now = new Date()
  const diffInSeconds = Math.floor((now.getTime() - date.getTime()) / 1000)

  if (diffInSeconds < 60) return "Just now"
  if (diffInSeconds < 3600) return `${Math.floor(diffInSeconds / 60)} minutes ago`
  if (diffInSeconds < 86400) return `${Math.floor(diffInSeconds / 3600)} hours ago`
  if (diffInSeconds < 2592000) return `${Math.floor(diffInSeconds / 86400)} days ago`
  if (diffInSeconds < 31536000) return `${Math.floor(diffInSeconds / 2592000)} months ago`
  return `${Math.floor(diffInSeconds / 31536000)} years ago`
}
