// Typesense client configuration lib/typesense.ts
import Typesense from "typesense"
import { searchYouTubeVideos, type YouTubeVideo } from "./youtube"

const typesenseConfig = {
  host: process.env.TYPESENSE_HOST || process.env.NEXT_PUBLIC_TYPESENSE_HOST || "localhost",
  port: Number.parseInt(process.env.TYPESENSE_PORT || process.env.NEXT_PUBLIC_TYPESENSE_PORT || "8108"),
  protocol: process.env.TYPESENSE_PROTOCOL || process.env.NEXT_PUBLIC_TYPESENSE_PROTOCOL || "https",
  apiKey: process.env.TYPESENSE_API_KEY || "",
}

const typesenseClient = new Typesense.Client({
  nodes: [
    {
      host: typesenseConfig.host,
      port: typesenseConfig.port,
      protocol: typesenseConfig.protocol,
    },
  ],
  apiKey: typesenseConfig.apiKey,
  connectionTimeoutSeconds: 10,
  retryIntervalSeconds: 0.1,
})

console.log("[v0] Typesense Config:", {
  host: typesenseConfig.host,
  port: typesenseConfig.port,
  protocol: typesenseConfig.protocol,
  hasApiKey: !!typesenseConfig.apiKey,
})

export interface SearchParams {
  query: string
  type: "web" | "images" | "videos" | "ads" | "websites"
  safeSearch: boolean
  dateFilter: string
  page?: number
  limit?: number
}

export interface SearchResult<T> {
  results: T[]
  total: number
  hasMore: boolean
}

export interface WebSearchResult {
  id: string
  title: string
  description: string
  url: string
  favicon?: string
  date?: string
  keywords?: string[]
  safesearch: boolean
  maindomain: boolean
}

export interface ImageSearchResult {
  id: string
  title: string
  url: string
  imgurl: string
  keywords?: string[]
  favicon?: string
  date?: string
  safesearch: boolean
}

export interface VideoSearchResult {
  id: string
  title: string
  thumbnail: string
  url: string
  date?: string
  safesearch: boolean
  channelTitle?: string
  description?: string
}

export interface AdResult {
  id: string
  title: string
  description: string
  imageUrl?: string
  url?: string
  targetUrl: string
  company?: string
  adType?: string
  keywords: string[]
  favicon?: string
  dailyLimit: number
  monthlyLimit: number
  dailyShown: number
  monthlyShown: number
  lastShownDate: string
  active: boolean
  createdAt: string
}

export interface WebsiteSubmission {
  id: string
  title: string
  url: string
  imageUrl?: string
  supportImageCrawl: boolean
  supportWebCrawl: boolean
  ownerConfirmed: boolean
  status: "pending" | "approved" | "rejected"
  webcrawl_status: "pending" | "complete" | "failed"
  imgcrawl_status: "pending" | "complete" | "failed"
  submittedAt: string
  reviewedAt?: string
  reviewedBy?: string
  notes?: string
}

export interface User {
  id: string
  username: string
  email: string
  password: string
  walletBalance: number
  createdAt: string
  lastLoginAt?: string
  isActive: boolean
  updatedAt?: string
  emailVerified: boolean
  emailVerificationToken?: string
  emailVerificationExpires?: string
  passwordResetToken?: string
  passwordResetExpires?: string
}

export interface UserAd extends AdResult {
  userId: string
  costPerClick: number
  costPerImpression: number
  totalSpent: number
  clicks: number
  impressions: number
  isActive: boolean
}

export interface WalletTransaction {
  id: string
  userId: string
  type: "credit" | "debit"
  amount: number
  description: string
  adId?: string
  razorpayPaymentId?: string
  createdAt: string
}

export async function searchWeb(params: SearchParams): Promise<SearchResult<WebSearchResult>> {
  try {
    const searchParameters = {
      q: params.query,
      query_by: "title,description,keywords",
      filter_by: params.safeSearch ? "safesearch:=true" : "",
      sort_by: "_text_match:desc",
      per_page: params.limit || 25,
      page: params.page || 1,
    }

    const searchResults = await typesenseClient.collections("web_results").documents().search(searchParameters)

    const results =
      searchResults.hits?.map((hit: any) => ({
        id: hit.document.id,
        title: hit.document.title,
        description: hit.document.description,
        url: hit.document.url,
        favicon: `https://www.google.com/s2/favicons?domain=${new URL(hit.document.url).hostname}`,
        date: hit.document.date,
        keywords: hit.document.keywords,
        safesearch: hit.document.safesearch,
        maindomain: hit.document.maindomain,
      })) || []

    const total = searchResults.found || 0
    const currentPage = params.page || 1
    const limit = params.limit || 25
    const hasMore = total > currentPage * limit

    return {
      results,
      total,
      hasMore,
    }
  } catch (error) {
    console.error("Typesense search error:", error)
    return { results: [], total: 0, hasMore: false }
  }
}

export async function searchImages(params: SearchParams): Promise<SearchResult<ImageSearchResult>> {
  try {
    const searchParameters = {
      q: params.query,
      query_by: "title,keywords",
      query_by_weights: "2,1",
      sort_by: "_text_match:desc",
      typo_tokens_threshold: 1,
      prefix: true,
      num_typos: "1,1",
      filter_by: params.safeSearch ? "safesearch:=true" : "",
      per_page: params.limit || 25,
      page: params.page || 1,
    }

    if (params.dateFilter && params.dateFilter !== "all") {
      const now = new Date()
      let filterDate: Date

      switch (params.dateFilter) {
        case "past24hours":
          filterDate = new Date(now.getTime() - 24 * 60 * 60 * 1000)
          break
        case "pastweek":
          filterDate = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000)
          break
        case "pastmonth":
          filterDate = new Date(now.getTime() - 30 * 24 * 60 * 60 * 1000)
          break
        case "pastyear":
          filterDate = new Date(now.getTime() - 365 * 24 * 60 * 60 * 1000)
          break
        default:
          filterDate = new Date(0)
      }

      const dateFilter = `date:>=${Math.floor(filterDate.getTime() / 1000)}`
      searchParameters.filter_by = searchParameters.filter_by
        ? `${searchParameters.filter_by} && ${dateFilter}`
        : dateFilter
    }

    console.log("[v0] Image search parameters:", searchParameters)

    const searchResults = await typesenseClient.collections("image_results").documents().search(searchParameters)

    console.log("[v0] Image search results count:", searchResults.hits?.length || 0)

    const results =
      searchResults.hits?.map((hit: any) => {
        const document = hit.document
        return {
          id: document.id,
          title: document.title,
          url: document.url,
          imgurl: document.imgurl,
          keywords: document.keywords,
          favicon: `https://www.google.com/s2/favicons?domain=${new URL(hit.document.url).hostname}`,
          date: document.date,
          safesearch: document.safesearch,
        }
      }) || []

    const total = searchResults.found || 0
    const currentPage = params.page || 1
    const limit = params.limit || 25
    const hasMore = total > currentPage * limit

    return {
      results,
      total,
      hasMore,
    }
  } catch (error) {
    console.error("[v0] Typesense image search error:", error)
    console.error("[v0] Error details:", {
      message: error.message,
      code: error.httpStatus,
      query: params.query,
    })
    return { results: [], total: 0, hasMore: false }
  }
}

export async function searchVideos(params: SearchParams): Promise<SearchResult<VideoSearchResult>> {
  try {
    const { videos, totalResults } = await searchYouTubeVideos(params.query, params.limit || 25, params.safeSearch)

    const results = videos.map((video: YouTubeVideo) => ({
      id: video.id,
      title: video.title,
      thumbnail: video.thumbnail,
      url: video.url,
      date: video.publishedAt,
      safesearch: params.safeSearch,
      channelTitle: video.channelTitle,
      description: video.description,
    }))

    const currentPage = params.page || 1
    const limit = params.limit || 25
    const hasMore = totalResults > currentPage * limit

    return {
      results,
      total: totalResults,
      hasMore,
    }
  } catch (error) {
    console.error("YouTube video search error:", error)
    return { results: [], total: 0, hasMore: false }
  }
}

export async function searchAds(params: SearchParams): Promise<SearchResult<AdResult>> {
  try {
    console.log(`[TYPESENSE ADS] Starting search for query: "${params.query}"`)
    console.log(`[TYPESENSE ADS] Typesense config:`, {
      host: typesenseConfig.host,
      port: typesenseConfig.port,
      hasApiKey: !!typesenseConfig.apiKey,
    })

    // Create collection if it doesn't exist
    await createAdsCollection()

    // Check if collection exists and has data
    let collectionExists = false
    let documentCount = 0

    try {
      const testSearch = await typesenseClient.collections("ads_results").documents().search({
        q: "*",
        per_page: 1,
      })
      collectionExists = true
      documentCount = testSearch.found || 0
      console.log(`[TYPESENSE ADS] Collection exists with ${documentCount} documents`)
    } catch (collectionError) {
      console.error(`[TYPESENSE ADS] Collection check failed:`, collectionError)
    }

    if (!collectionExists) {
      console.log(`[TYPESENSE ADS] Ads collection doesn't exist, creating it...`)
      await createAdsCollection()
    }

    if (documentCount === 0) {
      console.log(`[TYPESENSE ADS] No ads in collection`)
      return { results: [], total: 0, hasMore: false }
    }

    const today = new Date().toISOString().split("T")[0]
    const currentMonth = new Date().toISOString().substring(0, 7)

    // Try multiple search strategies
    let searchResults = null
    const strategies = [
      // Strategy 1: Exact keyword match
      {
        name: "keyword_match",
        params: {
          q: params.query,
          query_by: "keywords,title,description",
          query_by_weights: "4,3,2",
          filter_by: `active:=true`,
          sort_by: "_text_match:desc",
          per_page: params.limit || 10,
        },
      },
      // Strategy 2: Broader text search
      {
        name: "text_search",
        params: {
          q: params.query,
          query_by: "title,description,keywords",
          prefix: true,
          filter_by: `active:=true`,
          sort_by: "_text_match:desc",
          per_page: params.limit || 10,
        },
      },
      // Strategy 3: Wildcard (show all active ads)
      {
        name: "wildcard",
        params: {
          q: "*",
          filter_by: `active:=true`,
          per_page: params.limit || 10,
        },
      },
    ]

    for (const strategy of strategies) {
      try {
        console.log(`[TYPESENSE ADS] Trying strategy: ${strategy.name}`)
        console.log(`[TYPESENSE ADS] Search params:`, strategy.params)

        searchResults = await typesenseClient.collections("ads_results").documents().search(strategy.params)

        console.log(`[TYPESENSE ADS] Strategy ${strategy.name} found: ${searchResults.found} results`)

        if (searchResults.hits && searchResults.hits.length > 0) {
          console.log(`[TYPESENSE ADS] Success with strategy: ${strategy.name}`)
          break
        }
      } catch (error) {
        console.error(`[TYPESENSE ADS] Strategy ${strategy.name} failed:`, error)
      }
    }

    if (!searchResults || !searchResults.hits || searchResults.hits.length === 0) {
      console.log(`[TYPESENSE ADS] No results from any strategy`)
      return { results: [], total: 0, hasMore: false }
    }

    // Process results
    const results = searchResults.hits.map((hit: any) => {
      const doc = hit.document
      console.log(`[TYPESENSE ADS] Processing ad:`, { id: doc.id, title: doc.title, active: doc.active })

      return {
        id: doc.id,
        title: doc.title,
        description: doc.description,
        imageUrl: doc.imageUrl,
        url: doc.url || doc.targetUrl,
        targetUrl: doc.targetUrl,
        company: doc.company,
        adType: doc.adType,
        keywords: doc.keywords || [],
        favicon: doc.favicon,
        dailyLimit: doc.dailyLimit || 100,
        monthlyLimit: doc.monthlyLimit || 1000,
        dailyShown: doc.dailyShown || 0,
        monthlyShown: doc.monthlyShown || 0,
        lastShownDate: doc.lastShownDate || "",
        active: doc.active !== false,
        createdAt: doc.createdAt,
      }
    })

    console.log(`[TYPESENSE ADS] Processed ${results.length} ads`)

    // Apply limits filtering
    const availableResults = results.filter((ad) => {
      const canShowToday = ad.dailyShown < ad.dailyLimit || ad.lastShownDate !== today
      const canShowThisMonth = ad.monthlyShown < ad.monthlyLimit
      const meetsLimits = canShowToday && canShowThisMonth
      console.log(`[TYPESENSE ADS] Ad ${ad.id} limits check:`, { canShowToday, canShowThisMonth, meetsLimits })
      return meetsLimits
    })

    console.log(`[TYPESENSE ADS] ${availableResults.length} ads available after limits filtering`)

    // Shuffle and limit
    const finalResults = availableResults.sort(() => Math.random() - 0.5).slice(0, params.limit || 4)

    console.log(`[TYPESENSE ADS] Returning ${finalResults.length} final ads`)

    // Update shown counts
    for (const ad of finalResults) {
      try {
        await updateAdShownCount(ad.id, today, currentMonth)
      } catch (updateError) {
        console.error(`[TYPESENSE ADS] Failed to update count for ad ${ad.id}:`, updateError)
      }
    }

    return {
      results: finalResults,
      total: searchResults.found || 0,
      hasMore: false,
    }
  } catch (error) {
    console.error("[TYPESENSE ADS] Critical error:", error)
    return { results: [], total: 0, hasMore: false }
  }
}

async function updateAdShownCount(adId: string, today: string, currentMonth: string) {
  try {
    const ad = await typesenseClient.collections("ads_results").documents(adId).retrieve()

    let dailyShown = ad.dailyShown || 0
    let monthlyShown = ad.monthlyShown || 0

    // Reset daily count if it's a new day
    if (ad.lastShownDate !== today) {
      dailyShown = 0
    }

    // Reset monthly count if it's a new month
    if (!ad.lastShownDate || ad.lastShownDate.substring(0, 7) !== currentMonth) {
      monthlyShown = 0
    }

    await typesenseClient
      .collections("ads_results")
      .documents(adId)
      .update({
        dailyShown: dailyShown + 1,
        monthlyShown: monthlyShown + 1,
        lastShownDate: today,
      })

    console.log(`[TYPESENSE] Updated ad ${adId} counts: daily=${dailyShown + 1}, monthly=${monthlyShown + 1}`)
  } catch (error) {
    console.error("[TYPESENSE] Error updating ad shown count:", error)
  }
}

export async function addAd(ad: Omit<AdResult, "id" | "dailyShown" | "monthlyShown" | "lastShownDate" | "createdAt">) {
  try {
    await createAdsCollection()

    const newAd = {
      ...ad,
      id: Math.random().toString(36).substr(2, 9), // Generate a random ID
      dailyShown: 0,
      monthlyShown: 0,
      lastShownDate: "",
      createdAt: new Date().toISOString(),
    }

    const result = await typesenseClient.collections("ads_results").documents().create(newAd)
    console.log(`[TYPESENSE] Added new ad: ${newAd.id}`)
    return result
  } catch (error) {
    console.error("[TYPESENSE] Error adding ad:", error)
    throw error
  }
}

export async function migrateUsersCollection() {
  try {
    console.log("[TYPESENSE] Starting users collection migration...")

    // Check if collection exists and get current documents
    let existingUsers: any[] = []
    try {
      const searchResults = await typesenseClient.collections("users").documents().search({
        q: "*",
        per_page: 250,
      })
      existingUsers = searchResults.hits?.map((hit) => hit.document) || []
      console.log(`[TYPESENSE] Found ${existingUsers.length} existing users to migrate`)
    } catch (error) {
      console.log("[TYPESENSE] No existing users collection found")
    }

    // Drop existing collection if it exists
    try {
      await typesenseClient.collections("users").delete()
      console.log("[TYPESENSE] Dropped existing users collection")
    } catch (error) {
      console.log("[TYPESENSE] No existing collection to drop")
    }

    // Create new collection with updated schema
    await createUsersCollection()

    // Migrate existing users with new fields
    for (const user of existingUsers) {
      const migratedUser = {
        ...user,
        emailVerified: user.emailVerified || false,
        emailVerificationToken: user.emailVerificationToken || null,
        emailVerificationExpires: user.emailVerificationExpires || null,
        passwordResetToken: user.passwordResetToken || null,
        passwordResetExpires: user.passwordResetExpires || null,
        updatedAt: new Date().toISOString(),
      }

      try {
        await typesenseClient.collections("users").documents().create(migratedUser)
        console.log(`[TYPESENSE] Migrated user: ${user.id}`)
      } catch (error) {
        console.error(`[TYPESENSE] Error migrating user ${user.id}:`, error)
      }
    }

    console.log("[TYPESENSE] Users collection migration completed")
  } catch (error) {
    console.error("[TYPESENSE] Error during users collection migration:", error)
    throw error
  }
}

export async function createAdsCollection() {
  try {
    const adsSchema = {
      name: "ads_results",
      fields: [
        { name: "title", type: "string" },
        { name: "description", type: "string" },
        { name: "imageUrl", type: "string", optional: true },
        { name: "url", type: "string", optional: true },
        { name: "targetUrl", type: "string" },
        { name: "company", type: "string", optional: true },
        { name: "adType", type: "string", optional: true },
        { name: "keywords", type: "string[]" },
        { name: "favicon", type: "string", optional: true },
        { name: "dailyLimit", type: "int32" },
        { name: "monthlyLimit", type: "int32" },
        { name: "dailyShown", type: "int32", optional: true },
        { name: "monthlyShown", type: "int32", optional: true },
        { name: "lastShownDate", type: "string", optional: true },
        { name: "active", type: "bool" },
        { name: "createdAt", type: "string" },
      ],
    }

    await typesenseClient.collections().create(adsSchema)
    console.log("[TYPESENSE] Ads collection created successfully")
  } catch (error) {
    if (error.httpStatus !== 409) {
      console.error("[TYPESENSE] Error creating ads collection:", error)
    } else {
      console.log("[TYPESENSE] Ads collection already exists")
    }
  }
}

// Helper function to check if ads collection exists and has data
export async function checkAdsCollection() {
  try {
    const collection = await typesenseClient.collections("ads_results").retrieve()
    const stats = await typesenseClient.collections("ads_results").documents().search({
      q: "*",
      per_page: 1,
    })

    console.log(`[TYPESENSE] Ads collection exists with ${stats.found} documents`)
    return {
      exists: true,
      documentCount: stats.found || 0,
    }
  } catch (error) {
    console.log(`[TYPESENSE] Ads collection check failed:`, error.message)
    return {
      exists: false,
      documentCount: 0,
    }
  }
}

// Helper function to get all ads (for admin purposes)
export async function getAllAds() {
  try {
    const searchResults = await typesenseClient.collections("ads_results").documents().search({
      q: "*",
      per_page: 100,
    })

    return searchResults.hits?.map((hit: any) => hit.document) || []
  } catch (error) {
    console.error("[TYPESENSE] Error getting all ads:", error)
    return []
  }
}

// Helper function to delete an ad
export async function deleteAd(adId: string) {
  try {
    await typesenseClient.collections("ads_results").documents(adId).delete()
    console.log(`[TYPESENSE] Deleted ad: ${adId}`)
    return true
  } catch (error) {
    console.error(`[TYPESENSE] Error deleting ad ${adId}:`, error)
    return false
  }
}

// Helper function to update an ad
export async function updateAd(adId: string, updates: Partial<AdResult>) {
  try {
    const result = await typesenseClient.collections("ads_results").documents(adId).update(updates)
    console.log(`[TYPESENSE] Updated ad: ${adId}`)
    return result
  } catch (error) {
    console.error(`[TYPESENSE] Error updating ad ${adId}:`, error)
    throw error
  }
}

export async function createWebsitesCollection() {
  try {
    const websitesSchema = {
      name: "websites",
      fields: [
        { name: "title", type: "string" },
        { name: "url", type: "string" },
        { name: "imageUrl", type: "string", optional: true },
        { name: "supportImageCrawl", type: "bool" },
        { name: "supportWebCrawl", type: "bool" },
        { name: "ownerConfirmed", type: "bool" },
        { name: "status", type: "string" },
        { name: "submittedAt", type: "string" },
        { name: "reviewedAt", type: "string", optional: true },
        { name: "reviewedBy", type: "string", optional: true },
        { name: "notes", type: "string", optional: true },
      ],
    }

    await typesenseClient.collections().create(websitesSchema)
    console.log("[TYPESENSE] Websites collection created successfully")
  } catch (error) {
    if (error.httpStatus !== 409) {
      console.error("[TYPESENSE] Error creating websites collection:", error)
    } else {
      console.log("[TYPESENSE] Websites collection already exists")
    }
  }
}

export async function submitWebsite(website: Omit<WebsiteSubmission, "id" | "submittedAt" | "status">) {
  try {
    await createWebsitesCollection()

    const newWebsite = {
      ...website,
      id: Math.random().toString(36).substr(2, 9),
      status: "pending" as const,
      submittedAt: new Date().toISOString(),
    }

    const result = await typesenseClient.collections("websites").documents().create(newWebsite)
    console.log(`[TYPESENSE] Submitted new website: ${newWebsite.id}`)
    return result
  } catch (error) {
    console.error("[TYPESENSE] Error submitting website:", error)
    throw error
  }
}

export async function searchWebsites(params: { query?: string; status?: string; page?: number; limit?: number }) {
  try {
    const searchParameters = {
      q: params.query || "*",
      query_by: params.query ? "title,url" : undefined,
      filter_by: params.status ? `status:=${params.status}` : undefined,
      sort_by: params.query ? "_text_match:desc" : undefined,
      per_page: params.limit || 25,
      page: params.page || 1,
    }

    console.log("[TYPESENSE] Search websites parameters:", searchParameters)

    const searchResults = await typesenseClient.collections("websites").documents().search(searchParameters)

    const results = searchResults.hits?.map((hit: any) => hit.document) || []
    const total = searchResults.found || 0
    const currentPage = params.page || 1
    const limit = params.limit || 25
    const hasMore = total > currentPage * limit

    return {
      results,
      total,
      hasMore,
    }
  } catch (error) {
    console.error("[TYPESENSE] Error searching websites:", error)
    return { results: [], total: 0, hasMore: false }
  }
}

export async function updateWebsite(websiteId: string, updates: Partial<WebsiteSubmission>) {
  try {
    const result = await typesenseClient.collections("websites").documents(websiteId).update(updates)
    console.log(`[TYPESENSE] Updated website: ${websiteId}`)
    return result
  } catch (error) {
    console.error(`[TYPESENSE] Error updating website ${websiteId}:`, error)
    throw error
  }
}

export async function deleteWebsite(websiteId: string) {
  try {
    await typesenseClient.collections("websites").documents(websiteId).delete()
    console.log(`[TYPESENSE] Deleted website: ${websiteId}`)
    return true
  } catch (error) {
    console.error(`[TYPESENSE] Error deleting website ${websiteId}:`, error)
    return false
  }
}

export async function createUsersCollection() {
  try {
    const usersSchema = {
      name: "users",
      fields: [
        { name: "username", type: "string" },
        { name: "email", type: "string" },
        { name: "password", type: "string" },
        { name: "walletBalance", type: "float" },
        { name: "createdAt", type: "string" },
        { name: "lastLoginAt", type: "string", optional: true },
        { name: "isActive", type: "bool" },
        { name: "updatedAt", type: "string", optional: true },
        { name: "emailVerified", type: "bool" },
        { name: "emailVerificationToken", type: "string", optional: true },
        { name: "emailVerificationExpires", type: "string", optional: true },
        { name: "passwordResetToken", type: "string", optional: true },
        { name: "passwordResetExpires", type: "string", optional: true },
      ],
    }

    await typesenseClient.collections().create(usersSchema)
    console.log("[TYPESENSE] Users collection created successfully")
  } catch (error) {
    if (error.httpStatus !== 409) {
      console.error("[TYPESENSE] Error creating users collection:", error)
    } else {
      console.log("[TYPESENSE] Users collection already exists")
    }
  }
}

export async function createUserAdsCollection() {
  try {
    const userAdsSchema = {
      name: "user_ads",
      fields: [
        { name: "userId", type: "string" },
        { name: "title", type: "string" },
        { name: "description", type: "string" },
        { name: "imageUrl", type: "string", optional: true },
        { name: "targetUrl", type: "string" },
        { name: "keywords", type: "string[]" },
        { name: "dailyLimit", type: "int32" },
        { name: "monthlyLimit", type: "int32" },
        { name: "costPerClick", type: "float" },
        { name: "costPerImpression", type: "float" },
        { name: "totalSpent", type: "float" },
        { name: "clicks", type: "int32" },
        { name: "impressions", type: "int32" },
        { name: "dailyShown", type: "int32", optional: true },
        { name: "monthlyShown", type: "int32", optional: true },
        { name: "lastShownDate", type: "string", optional: true },
        { name: "isActive", type: "bool" },
        { name: "createdAt", type: "string" },
      ],
    }

    await typesenseClient.collections().create(userAdsSchema)
    console.log("[TYPESENSE] User ads collection created successfully")
  } catch (error) {
    if (error.httpStatus !== 409) {
      console.error("[TYPESENSE] Error creating user ads collection:", error)
    } else {
      console.log("[TYPESENSE] User ads collection already exists")
    }
  }
}

export async function createWalletTransactionsCollection() {
  try {
    const walletSchema = {
      name: "wallet_transactions",
      fields: [
        { name: "userId", type: "string" },
        { name: "type", type: "string" },
        { name: "amount", type: "float" },
        { name: "description", type: "string" },
        { name: "adId", type: "string", optional: true },
        { name: "razorpayPaymentId", type: "string", optional: true },
        { name: "createdAt", type: "string" },
      ],
    }

    await typesenseClient.collections().create(walletSchema)
    console.log("[TYPESENSE] Wallet transactions collection created successfully")
  } catch (error) {
    if (error.httpStatus !== 409) {
      console.error("[TYPESENSE] Error creating wallet transactions collection:", error)
    } else {
      console.log("[TYPESENSE] Wallet transactions collection already exists")
    }
  }
}

export async function checkUsernameAvailability(username: string): Promise<boolean> {
  try {
    await createUsersCollection()

    const searchResults = await typesenseClient
      .collections("users")
      .documents()
      .search({
        q: username,
        query_by: "username",
        filter_by: `username:=${username}`,
        per_page: 1,
      })

    return searchResults.found === 0
  } catch (error) {
    console.error("[TYPESENSE] Error checking username availability:", error)
    return false
  }
}


export async function createUser(
  user: Omit<
    User,
    | "id"
    | "createdAt"
    | "walletBalance"
    | "isActive"
    | "emailVerified"
    | "emailVerificationToken"
    | "emailVerificationExpires"
    | "passwordResetToken"
    | "passwordResetExpires"
  >,
) {
  try {
    await createUsersCollection();

    // 1. CHECK IF EMAIL ALREADY EXISTS
    const existingUserSearch = await typesenseClient
      .collections("users")
      .documents()
      .search({
        q: user.email,
        query_by: "email",
        filter_by: `email:=${user.email}`, 
      });

    if (existingUserSearch.hits && existingUserSearch.hits.length > 0) {
      // Return a structured error object instead of throwing
      return { success: false, error: "Email already exists" };
    }

    // 2. PREPARE NEW USER DATA
    const verificationToken = Math.random().toString(36).substr(2, 32);
    const verificationExpires = new Date(Date.now() + 24 * 60 * 60 * 1000).toISOString();

    const newUser = {
      ...user,
      id: Math.random().toString(36).substr(2, 9),
      walletBalance: 0,
      createdAt: new Date().toISOString(),
      isActive: true,
      emailVerified: false,
      emailVerificationToken: verificationToken,
      emailVerificationExpires: verificationExpires,
      passwordResetToken: null,
      passwordResetExpires: null,
    };

    // 3. CREATE DOCUMENT
    const result = await typesenseClient.collections("users").documents().create(newUser);
    console.log(`[TYPESENSE] Created new user: ${newUser.id}`);
    
    return { success: true, data: result };
    
  } catch (error: any) {
    console.error("[TYPESENSE] Internal Error:", error);
    return { success: false, error: "Internal Server Error" };
  }
}


// export async function createUser(
//   user: Omit<
//     User,
//     | "id"
//     | "createdAt"
//     | "walletBalance"
//     | "isActive"
//     | "emailVerified"
//     | "emailVerificationToken"
//     | "emailVerificationExpires"
//     | "passwordResetToken"
//     | "passwordResetExpires"
//   >,
// ) {
//   try {
//     await createUsersCollection()

//     const verificationToken = Math.random().toString(36).substr(2, 32)
//     const verificationExpires = new Date(Date.now() + 24 * 60 * 60 * 1000).toISOString() // 24 hours

//     const newUser = {
//       ...user,
//       id: Math.random().toString(36).substr(2, 9),
//       walletBalance: 0,
//       createdAt: new Date().toISOString(),
//       isActive: true,
//       emailVerified: false,
//       emailVerificationToken: verificationToken,
//       emailVerificationExpires: verificationExpires,
//       passwordResetToken: null,
//       passwordResetExpires: null,
//     }

//     const result = await typesenseClient.collections("users").documents().create(newUser)
//     console.log(`[TYPESENSE] Created new user: ${newUser.id}`)
//     return result
//   } catch (error) {
//     console.error("[TYPESENSE] Error creating user:", error)
//     throw error
//   }
// }

export async function authenticateUser(username: string, password: string): Promise<User | null> {
  try {
    const searchResults = await typesenseClient
      .collections("users")
      .documents()
      .search({
        q: username,
        query_by: "username,email",
        filter_by: `username:=${username} || email:=${username}`,
        per_page: 1,
      })

    if (searchResults.found === 0) {
      return null
    }

    const user = searchResults.hits[0].document as User

    // In a real app, you'd use bcrypt to compare hashed passwords
    if (user.password === password && user.emailVerified) {
      // Update last login
      await typesenseClient.collections("users").documents(user.id).update({
        lastLoginAt: new Date().toISOString(),
      })

      return user
    }

    return null
  } catch (error) {
    console.error("[TYPESENSE] Error authenticating user:", error)
    return null
  }
}

export async function getUserById(userId: string): Promise<User | null> {
  try {
    const user = await typesenseClient.collections("users").documents(userId).retrieve()
    return user as User
  } catch (error) {
    console.error("[TYPESENSE] Error getting user by ID:", error)
    return null
  }
}

export async function getUserByEmail(email: string): Promise<User | null> {
  try {
    const searchResults = await typesenseClient
      .collections("users")
      .documents()
      .search({
        q: email,
        query_by: "email",
        filter_by: `email:=${email}`,
        per_page: 1,
      })

    if (searchResults.found === 0) {
      return null
    }

    return searchResults.hits[0].document as User
  } catch (error) {
    console.error("[TYPESENSE] Error getting user by email:", error)
    return null
  }
}

export async function generatePasswordResetToken(userId: string): Promise<string | null> {
  try {
    const user = await getUserById(userId)
    if (!user) {
      return null
    }

    const resetToken = Math.random().toString(36).substr(2, 32)
    const resetExpires = new Date(Date.now() + 60 * 60 * 1000).toISOString() // 1 hour

    await typesenseClient.collections("users").documents(user.id).update({
      passwordResetToken: resetToken,
      passwordResetExpires: resetExpires,
      updatedAt: new Date().toISOString(),
    })

    console.log("[TYPESENSE] Generated password reset token for user:", user.id)
    return resetToken
  } catch (error) {
    console.error("[TYPESENSE] Error generating password reset token:", error)
    return null
  }
}

export async function validatePasswordResetToken(token: string): Promise<User | null> {
  try {
    const searchResults = await typesenseClient
      .collections("users")
      .documents()
      .search({
        q: token,
        query_by: "passwordResetToken",
        filter_by: `passwordResetToken:=${token}`,
        per_page: 1,
      })

    if (searchResults.found === 0) {
      return null
    }

    const user = searchResults.hits[0].document as User

    // Check if token is expired
    if (user.passwordResetExpires && new Date(user.passwordResetExpires) < new Date()) {
      return null
    }

    return user
  } catch (error) {
    console.error("[TYPESENSE] Error validating password reset token:", error)
    return null
  }
}

export async function resetPasswordWithToken(token: string, newPassword: string): Promise<boolean> {
  try {
    const user = await validatePasswordResetToken(token)
    if (!user) {
      return false
    }

    // Update password and clear reset token
    await typesenseClient.collections("users").documents(user.id).update({
      password: newPassword, // In production, hash this with bcrypt
      passwordResetToken: null,
      passwordResetExpires: null,
      updatedAt: new Date().toISOString(),
    })

    console.log("[TYPESENSE] Password reset completed for user:", user.id)
    return true
  } catch (error) {
    console.error("[TYPESENSE] Error resetting password:", error)
    return false
  }
}

export async function updateUserWallet(
  userId: string,
  amount: number,
  type: "credit" | "debit",
  description: string,
  adId?: string,
  razorpayPaymentId?: string,
) {
  try {
    await createWalletTransactionsCollection()

    const user = await getUserById(userId)
    if (!user) throw new Error("User not found")

    const newBalance = type === "credit" ? user.walletBalance + amount : user.walletBalance - amount

    if (newBalance < 0) {
      throw new Error("Insufficient wallet balance")
    }

    // Update user balance
    await typesenseClient.collections("users").documents(userId).update({
      walletBalance: newBalance,
    })

    // Create transaction record
    const transaction = {
      id: Math.random().toString(36).substr(2, 9),
      userId,
      type,
      amount,
      description,
      adId,
      razorpayPaymentId,
      createdAt: new Date().toISOString(),
    }

    await typesenseClient.collections("wallet_transactions").documents().create(transaction)

    console.log(`[TYPESENSE] Updated wallet for user ${userId}: ${type} ${amount}, new balance: ${newBalance}`)
    return newBalance
  } catch (error) {
    console.error("[TYPESENSE] Error updating user wallet:", error)
    throw error
  }
}

export async function searchUserAds(params: SearchParams): Promise<SearchResult<UserAd>> {
  try {
    console.log(`[TYPESENSE USER ADS] Starting search for query: "${params.query}"`)

    await createUserAdsCollection()

    const today = new Date().toISOString().split("T")[0]
    const currentMonth = new Date().toISOString().substring(0, 7)

    // Search for active user ads with sufficient wallet balance
    const searchParameters = {
      q: params.query || "*",
      query_by: params.query ? "keywords,title,description" : undefined,
      filter_by: `isActive:=true`,
      sort_by: params.query ? "_text_match:desc" : undefined,
      per_page: params.limit || 10,
    }

    console.log(`[TYPESENSE USER ADS] Search parameters:`, searchParameters)

    const searchResults = await typesenseClient.collections("user_ads").documents().search(searchParameters)

    console.log(`[TYPESENSE USER ADS] Found ${searchResults.hits?.length || 0} potential ads`)

    if (!searchResults.hits || searchResults.hits.length === 0) {
      return { results: [], total: 0, hasMore: false }
    }

    // Filter ads based on wallet balance and limits
    const availableAds = []

    for (const hit of searchResults.hits) {
      const ad = hit.document as UserAd

      console.log(`[TYPESENSE USER ADS] Checking ad ${ad.id} for user ${ad.userId}`)

      // Check if user has sufficient wallet balance
      const user = await getUserById(ad.userId)
      if (!user) {
        console.log(`[TYPESENSE USER ADS] User ${ad.userId} not found, skipping ad ${ad.id}`)
        continue
      }

      console.log(
        `[TYPESENSE USER ADS] User ${ad.userId} wallet balance: ${user.walletBalance}, cost per impression: ${ad.costPerImpression}`,
      )

      if (user.walletBalance < ad.costPerImpression) {
        // Auto-pause ad if insufficient balance
        await typesenseClient.collections("user_ads").documents(ad.id).update({
          isActive: false,
        })
        console.log(`[TYPESENSE USER ADS] Auto-paused ad ${ad.id} due to insufficient balance`)
        continue
      }

      // Check daily and monthly limits
      const canShowToday = (ad.dailyShown || 0) < ad.dailyLimit || ad.lastShownDate !== today
      const canShowThisMonth = (ad.monthlyShown || 0) < ad.monthlyLimit

      console.log(
        `[TYPESENSE USER ADS] Ad ${ad.id} limits - Daily: ${ad.dailyShown || 0}/${ad.dailyLimit}, Monthly: ${ad.monthlyShown || 0}/${ad.monthlyLimit}`,
      )

      if (canShowToday && canShowThisMonth) {
        availableAds.push(ad)
        console.log(`[TYPESENSE USER ADS] Ad ${ad.id} is available for display`)
      } else {
        console.log(`[TYPESENSE USER ADS] Ad ${ad.id} reached limits - skipping`)
      }
    }

    // Shuffle and limit results
    const finalResults = availableAds.sort(() => Math.random() - 0.5).slice(0, params.limit || 4)

    console.log(`[TYPESENSE USER ADS] Returning ${finalResults.length} user ads`)

    return {
      results: finalResults,
      total: searchResults.found || 0,
      hasMore: false,
    }
  } catch (error) {
    console.error("[TYPESENSE USER ADS] Error:", error)
    return { results: [], total: 0, hasMore: false }
  }
}

export async function chargeUserForAdInteraction(
  adId: string,
  interactionType: "click" | "impression",
): Promise<boolean> {
  try {
    const ad = (await typesenseClient.collections("user_ads").documents(adId).retrieve()) as UserAd
    const user = await getUserById(ad.userId)

    if (!user) {
      console.error(`[TYPESENSE] User not found for ad ${adId}`)
      return false
    }

    const cost = interactionType === "click" ? ad.costPerClick : ad.costPerImpression

    if (user.walletBalance < cost) {
      // Auto-pause ad due to insufficient balance
      await typesenseClient.collections("user_ads").documents(adId).update({
        isActive: false,
      })
      console.log(`[TYPESENSE] Auto-paused ad ${adId} due to insufficient balance`)
      return false
    }

    // Deduct cost from wallet
    await updateUserWallet(
      ad.userId,
      cost,
      "debit",
      `${interactionType === "click" ? "Click" : "Impression"} charge for ad: ${ad.title}`,
      adId,
    )

    // Update ad statistics
    const updates: any = {
      totalSpent: ad.totalSpent + cost,
    }

    if (interactionType === "click") {
      updates.clicks = (ad.clicks || 0) + 1
    } else {
      updates.impressions = (ad.impressions || 0) + 1

      // Update daily/monthly shown counts for impressions
      const today = new Date().toISOString().split("T")[0]
      const currentMonth = new Date().toISOString().substring(0, 7)

      let dailyShown = ad.dailyShown || 0
      let monthlyShown = ad.monthlyShown || 0

      if (ad.lastShownDate !== today) {
        dailyShown = 0
      }

      if (!ad.lastShownDate || ad.lastShownDate.substring(0, 7) !== currentMonth) {
        monthlyShown = 0
      }

      updates.dailyShown = dailyShown + 1
      updates.monthlyShown = monthlyShown + 1
      updates.lastShownDate = today
    }

    await typesenseClient.collections("user_ads").documents(adId).update(updates)

    console.log(`[TYPESENSE] Charged ₹${cost} for ${interactionType} on ad ${adId}`)
    return true
  } catch (error) {
    console.error(`[TYPESENSE] Error charging user for ad interaction:`, error)
    return false
  }
}

export async function searchAllAds(params: SearchParams): Promise<SearchResult<AdResult | UserAd>> {
  try {
    console.log(`[TYPESENSE ALL ADS] Starting combined search for query: "${params.query}"`)

    // Get admin ads (free)
    const adminAds = await searchAds(params)
    console.log(`[TYPESENSE ALL ADS] Admin ads found: ${adminAds.results.length}`)

    let userAds: any[] = []

    try {
      // Check if user ads collection exists
      const collectionCheck = await typesenseClient.collections("user_ads").documents().search({
        q: "*",
        per_page: 1,
      })
      console.log(`[TYPESENSE USER ADS] Collection exists with ${collectionCheck.found} documents`)
    } catch (collectionError) {
      console.error(`[TYPESENSE USER ADS] Collection check failed:`, collectionError)
      // If collection doesn't exist, create it
      try {
        await createUserAdsCollection()
        console.log(`[TYPESENSE USER ADS] Created user ads collection`)
      } catch (createError) {
        console.error(`[TYPESENSE USER ADS] Failed to create collection:`, createError)
      }
    }

    // Get user ads (paid) with correct field name
    const userAdsStrategies = [
      // Strategy 1: Keyword matching
      {
        name: "keyword_match",
        params: {
          q: params.query,
          query_by: "keywords,title,description",
          query_by_weights: "4,3,2",
          filter_by: `isActive:=true`, // Use correct field name for user ads
          sort_by: "_text_match:desc",
          per_page: params.limit || 10,
        },
      },
      // Strategy 2: Broader text search
      {
        name: "text_search",
        params: {
          q: params.query,
          query_by: "title,description,keywords",
          prefix: true,
          filter_by: `isActive:=true`, // Use correct field name for user ads
          sort_by: "_text_match:desc",
          per_page: params.limit || 10,
        },
      },
      // Strategy 3: Wildcard fallback
      {
        name: "wildcard",
        params: {
          q: "*",
          filter_by: `isActive:=true`, // Use correct field name for user ads
          per_page: params.limit || 4,
        },
      },
    ]

    // Try each strategy for user ads
    for (const strategy of userAdsStrategies) {
      try {
        console.log(`[TYPESENSE USER ADS] Trying strategy: ${strategy.name}`)
        console.log(`[TYPESENSE USER ADS] Search params:`, strategy.params)

        const searchResults = await typesenseClient.collections("user_ads").documents().search(strategy.params)

        console.log(`[TYPESENSE USER ADS] Strategy ${strategy.name} found: ${searchResults.found} results`)

        if (searchResults.found > 0) {
          const processedUserAds = []

          for (const hit of searchResults.hits || []) {
            const ad = hit.document
            console.log(`[TYPESENSE USER ADS] Processing user ad:`, {
              id: ad.id,
              title: ad.title,
              userId: ad.userId,
              isActive: ad.isActive,
              keywords: ad.keywords,
            })

            // Check user wallet balance
            try {
              const user = await getUserById(ad.userId)
              if (user && user.walletBalance >= (ad.costPerImpression || 0.1)) {
                processedUserAds.push({
                  ...ad,
                  isUserAd: true,
                })
                console.log(`[TYPESENSE USER ADS] User ad ${ad.id} approved - wallet balance: ₹${user.walletBalance}`)
              } else {
                console.log(
                  `[TYPESENSE USER ADS] User ad ${ad.id} rejected - insufficient wallet balance: ₹${user?.walletBalance || 0}`,
                )
              }
            } catch (userError) {
              console.error(`[TYPESENSE USER ADS] Error checking user ${ad.userId}:`, userError)
            }
          }

          userAds = processedUserAds
          console.log(`[TYPESENSE USER ADS] Success with strategy: ${strategy.name}, approved ${userAds.length} ads`)
          break
        }
      } catch (error) {
        console.error(`[TYPESENSE USER ADS] Strategy ${strategy.name} failed:`, error)
        continue
      }
    }

    // Combine and shuffle results
    const allAds = [...adminAds.results.map((ad) => ({ ...ad, isUserAd: false })), ...userAds]

    // Shuffle to mix admin and user ads
    const shuffledAds = allAds.sort(() => Math.random() - 0.5)

    // Limit to requested amount
    const finalResults = shuffledAds.slice(0, params.limit || 4)

    console.log(
      `[TYPESENSE ALL ADS] Returning ${finalResults.length} total ads (${adminAds.results.length} admin, ${userAds.length} user)`,
    )

    return {
      results: finalResults,
      total: adminAds.total + userAds.length,
      hasMore: false,
    }
  } catch (error) {
    console.error("[TYPESENSE ALL ADS] Error:", error)
    return { results: [], total: 0, hasMore: false }
  }
}

export async function debugSearchAllAds(query = "") {
  try {
    console.log(`[DEBUG] Testing searchAllAds with query: "${query}"`)

    const result = await searchAllAds({
      query,
      type: "ads",
      safeSearch: true,
      dateFilter: "all",
      limit: 10,
    })

    console.log(`[DEBUG] searchAllAds results:`, {
      total: result.total,
      resultsCount: result.results.length,
      adminAds: result.results.filter((ad) => !ad.isUserAd).length,
      userAds: result.results.filter((ad) => ad.isUserAd).length,
      results: result.results.map((ad) => ({
        id: ad.id,
        title: ad.title,
        isUserAd: ad.isUserAd,
        userId: ad.userId,
        keywords: ad.keywords,
      })),
    })

    return result
  } catch (error) {
    console.error("[DEBUG] Error in searchAllAds:", error)
    return { results: [], total: 0, hasMore: false }
  }
}

export async function updateUserPassword(
  userId: string,
  currentPassword: string,
  newPassword: string,
): Promise<boolean> {
  try {
    // First, get the user to verify current password
    const user = (await typesenseClient.collections("users").documents(userId).retrieve()) as User

    // Verify current password (in a real app, you'd use bcrypt)
    if (user.password !== currentPassword) {
      return false
    }

    // Update password (in a real app, you'd hash the new password with bcrypt)
    await typesenseClient.collections("users").documents(userId).update({
      password: newPassword,
      updatedAt: new Date().toISOString(),
    })

    console.log("[TYPESENSE] Password updated for user:", userId)
    return true
  } catch (error) {
    console.error("[TYPESENSE] Error updating password:", error)
    return false
  }
}

export async function verifyEmail(token: string): Promise<User | null> {
  try {
    await migrateUsersCollection()

    const searchResults = await typesenseClient
      .collections("users")
      .documents()
      .search({
        q: token,
        query_by: "emailVerificationToken",
        filter_by: `emailVerificationToken:=${token}`,
        per_page: 1,
      })

    if (searchResults.found === 0) {
      return null
    }

    const user = searchResults.hits[0].document as User

    // Check if token is expired
    if (user.emailVerificationExpires && new Date(user.emailVerificationExpires) < new Date()) {
      return null
    }

    // Update user as verified
    await typesenseClient.collections("users").documents(user.id).update({
      emailVerified: true,
      emailVerificationToken: null,
      emailVerificationExpires: null,
      updatedAt: new Date().toISOString(),
    })

    console.log("[TYPESENSE] Email verified for user:", user.id)
    return await getUserById(user.id)
  } catch (error) {
    console.error("[TYPESENSE] Error verifying email:", error)
    return null
  }
}

export async function generateNewVerificationToken(email: string): Promise<string | null> {
  try {
    // Ensure users collection exists with proper schema
    await createUsersCollection()

    const searchResults = await typesenseClient
      .collections("users")
      .documents()
      .search({
        q: email,
        query_by: "email",
        filter_by: `email:=${email}`,
        per_page: 1,
      })

    if (searchResults.found === 0) {
      return null
    }

    const user = searchResults.hits[0].document as User
    const verificationToken = Math.random().toString(36).substr(2, 32)
    const verificationExpires = new Date(Date.now() + 24 * 60 * 60 * 1000).toISOString()

    await typesenseClient.collections("users").documents(user.id).update({
      emailVerificationToken: verificationToken,
      emailVerificationExpires: verificationExpires,
      updatedAt: new Date().toISOString(),
    })

    console.log("[TYPESENSE] Generated new verification token for user:", user.id)
    return verificationToken
  } catch (error) {
    console.error("[TYPESENSE] Error generating verification token:", error)
    return null
  }
}

export async function searchUsers(searchQuery = "", statusFilter = "all"): Promise<User[]> {
  try {
    await createUsersCollection()

    let filterBy = ""
    if (statusFilter === "active") {
      filterBy = "isActive:=true"
    } else if (statusFilter === "inactive") {
      filterBy = "isActive:=false"
    } else if (statusFilter === "verified") {
      filterBy = "emailVerified:=true"
    } else if (statusFilter === "unverified") {
      filterBy = "emailVerified:=false"
    }

    const searchParams: any = {
      q: searchQuery || "*",
      query_by: searchQuery ? "username,email" : "",
      per_page: 250,
    }

    if (filterBy) {
      searchParams.filter_by = filterBy
    }

    const searchResults = await typesenseClient.collections("users").documents().search(searchParams)

    console.log(`[TYPESENSE] Found ${searchResults.found} users`)

    // Get user ads count and total spent for each user
    const usersWithStats = await Promise.all(
      searchResults.hits.map(async (hit) => {
        const user = hit.document as User
        try {
          // Get user ads count
          const userAdsResults = await typesenseClient
            .collections("user_ads")
            .documents()
            .search({
              q: "*",
              filter_by: `userId:=${user.id}`,
              per_page: 0,
            })

          return {
            ...user,
            totalAds: userAdsResults.found || 0,
            totalSpent: user.walletBalance || 0, // This would be calculated from transactions in a real app
          }
        } catch (error) {
          return {
            ...user,
            totalAds: 0,
            totalSpent: 0,
          }
        }
      }),
    )

    return usersWithStats
  } catch (error) {
    console.error("[TYPESENSE] Error searching users:", error)
    return []
  }
}

export async function getUserStats(): Promise<{
  totalUsers: number
  activeUsers: number
  verifiedUsers: number
  totalRevenue: number
}> {
  try {
    await createUsersCollection()

    const allUsersResults = await typesenseClient.collections("users").documents().search({
      q: "*",
      per_page: 0,
    })

    const activeUsersResults = await typesenseClient.collections("users").documents().search({
      q: "*",
      filter_by: "isActive:=true",
      per_page: 0,
    })

    const verifiedUsersResults = await typesenseClient.collections("users").documents().search({
      q: "*",
      filter_by: "emailVerified:=true",
      per_page: 0,
    })

    return {
      totalUsers: allUsersResults.found || 0,
      activeUsers: activeUsersResults.found || 0,
      verifiedUsers: verifiedUsersResults.found || 0,
      totalRevenue: 0, // This would be calculated from wallet transactions
    }
  } catch (error) {
    console.error("[TYPESENSE] Error getting user stats:", error)
    return {
      totalUsers: 0,
      activeUsers: 0,
      verifiedUsers: 0,
      totalRevenue: 0,
    }
  }
}

export async function updateUser(userId: string, updates: Partial<User>): Promise<boolean> {
  try {
    await typesenseClient
      .collections("users")
      .documents(userId)
      .update({
        ...updates,
        updatedAt: new Date().toISOString(),
      })

    console.log("[TYPESENSE] Updated user:", userId)
    return true
  } catch (error) {
    console.error("[TYPESENSE] Error updating user:", error)
    return false
  }
}

export async function deleteUser(userId: string): Promise<boolean> {
  try {
    // Delete user's ads first
    const userAdsResults = await typesenseClient
      .collections("user_ads")
      .documents()
      .search({
        q: "*",
        filter_by: `userId:=${userId}`,
        per_page: 250,
      })

    for (const hit of userAdsResults.hits || []) {
      await typesenseClient.collections("user_ads").documents(hit.document.id).delete()
    }

    // Delete user's wallet transactions
    const transactionsResults = await typesenseClient
      .collections("wallet_transactions")
      .documents()
      .search({
        q: "*",
        filter_by: `userId:=${userId}`,
        per_page: 250,
      })

    for (const hit of transactionsResults.hits || []) {
      await typesenseClient.collections("wallet_transactions").documents(hit.document.id).delete()
    }

    // Delete the user
    await typesenseClient.collections("users").documents(userId).delete()

    console.log("[TYPESENSE] Deleted user and related data:", userId)
    return true
  } catch (error) {
    console.error("[TYPESENSE] Error deleting user:", error)
    return false
  }
}

export async function createAdminUsersCollection() {
  try {
    const adminUsersSchema = {
      name: "admin_users",
      fields: [
        { name: "username", type: "string" },
        { name: "email", type: "string", optional: true },
        { name: "password", type: "string" },
        { name: "createdAt", type: "string" },
        { name: "lastLoginAt", type: "string", optional: true },
        { name: "updatedAt", type: "string", optional: true },
      ],
    }

    await typesenseClient.collections().create(adminUsersSchema)
    console.log("[TYPESENSE] Admin users collection created successfully")
  } catch (error) {
    if (error.httpStatus !== 409) {
      console.error("[TYPESENSE] Error creating admin users collection:", error)
    } else {
      console.log("[TYPESENSE] Admin users collection already exists")
    }
  }
}

export async function createAdminUser(username: string, password: string, email?: string) {
  try {
    await createAdminUsersCollection()

    const newAdmin = {
      id: Math.random().toString(36).substr(2, 9),
      username,
      email: email || null,
      password, // In production, hash this with bcrypt
      createdAt: new Date().toISOString(),
      lastLoginAt: null,
      updatedAt: null,
    }

    const result = await typesenseClient.collections("admin_users").documents().create(newAdmin)
    console.log(`[TYPESENSE] Created new admin user: ${newAdmin.id}`)
    return result
  } catch (error) {
    console.error("[TYPESENSE] Error creating admin user:", error)
    throw error
  }
}

export async function authenticateAdmin(username: string, password: string) {
  try {
    await createAdminUsersCollection()

    const searchResults = await typesenseClient
      .collections("admin_users")
      .documents()
      .search({
        q: username,
        query_by: "username",
        filter_by: `username:=${username}`,
        per_page: 1,
      })

    if (searchResults.found === 0) {
      return null
    }

    const admin = searchResults.hits[0].document

    // In a real app, you'd use bcrypt to compare hashed passwords
    if (admin.password === password) {
      // Update last login
      await typesenseClient.collections("admin_users").documents(admin.id).update({
        lastLoginAt: new Date().toISOString(),
        updatedAt: new Date().toISOString(),
      })

      return admin
    }

    return null
  } catch (error) {
    console.error("[TYPESENSE] Error authenticating admin:", error)
    return null
  }
}

export async function getAdminById(adminId: string) {
  try {
    const admin = await typesenseClient.collections("admin_users").documents(adminId).retrieve()
    return admin
  } catch (error) {
    console.error("[TYPESENSE] Error getting admin by ID:", error)
    return null
  }
}

export async function updateAdminPassword(
  adminId: string,
  currentPassword: string,
  newPassword: string,
): Promise<boolean> {
  try {
    // First, get the admin to verify current password
    const admin = await typesenseClient.collections("admin_users").documents(adminId).retrieve()

    // Verify current password (in a real app, you'd use bcrypt)
    if (admin.password !== currentPassword) {
      return false
    }

    // Update password (in a real app, you'd hash the new password with bcrypt)
    await typesenseClient.collections("admin_users").documents(adminId).update({
      password: newPassword,
      updatedAt: new Date().toISOString(),
    })

    console.log("[TYPESENSE] Password updated for admin:", adminId)
    return true
  } catch (error) {
    console.error("[TYPESENSE] Error updating admin password:", error)
    return false
  }
}

export async function checkAdminUsernameAvailability(username: string): Promise<boolean> {
  try {
    await createAdminUsersCollection()

    const searchResults = await typesenseClient
      .collections("admin_users")
      .documents()
      .search({
        q: username,
        query_by: "username",
        filter_by: `username:=${username}`,
        per_page: 1,
      })

    return searchResults.found === 0
  } catch (error) {
    console.error("[TYPESENSE] Error checking admin username availability:", error)
    return false
  }
}

export async function createSiteSettingsCollection() {
  try {
    const siteSettingsSchema = {
      name: "site_settings",
      fields: [
        { name: "key", type: "string" },
        { name: "value", type: "string" },
        { name: "type", type: "string" }, // text, json, file, color, boolean
        { name: "category", type: "string" }, // general, appearance, seo, homepage
        { name: "updatedAt", type: "string" },
        { name: "updatedBy", type: "string", optional: true },
      ],
    }

    await typesenseClient.collections().create(siteSettingsSchema)
    console.log("[TYPESENSE] Site settings collection created successfully")
  } catch (error) {
    if (error.httpStatus !== 409) {
      console.error("[TYPESENSE] Error creating site settings collection:", error)
    } else {
      console.log("[TYPESENSE] Site settings collection already exists")
    }
  }
}

export async function getSiteSetting(key: string): Promise<any> {
  try {
    await createSiteSettingsCollection()

    const searchResults = await typesenseClient
      .collections("site_settings")
      .documents()
      .search({
        q: key,
        query_by: "key",
        filter_by: `key:=${key}`,
        per_page: 1,
      })

    if (searchResults.found === 0) {
      return getDefaultSiteSetting(key)
    }

    const setting = searchResults.hits[0].document

    // Parse value based on type
    switch (setting.type) {
      case "json":
        return JSON.parse(setting.value)
      case "boolean":
        return setting.value === "true"
      case "file":
      case "color":
      case "text":
      default:
        return setting.value
    }
  } catch (error) {
    console.error("[TYPESENSE] Error getting site setting:", error)
    return getDefaultSiteSetting(key)
  }
}

export async function setSiteSetting(key: string, value: any, type = "text", category = "general", updatedBy?: string) {
  try {
    await createSiteSettingsCollection()

    // Convert value to string for storage
    let stringValue: string
    switch (type) {
      case "json":
        stringValue = JSON.stringify(value)
        break
      case "boolean":
        stringValue = value ? "true" : "false"
        break
      default:
        stringValue = String(value)
    }

    const settingData = {
      id: key,
      key,
      value: stringValue,
      type,
      category,
      updatedAt: new Date().toISOString(),
      updatedBy,
    }

    // Try to update first, then create if not exists
    try {
      await typesenseClient.collections("site_settings").documents(key).update(settingData)
      console.log(`[TYPESENSE] Updated site setting: ${key}`)
    } catch (updateError) {
      await typesenseClient.collections("site_settings").documents().create(settingData)
      console.log(`[TYPESENSE] Created site setting: ${key}`)
    }

    return true
  } catch (error) {
    console.error("[TYPESENSE] Error setting site setting:", error)
    return false
  }
}

export async function getAllSiteSettings(): Promise<Record<string, any>> {
  try {
    await createSiteSettingsCollection()

    console.log("[v0] Fetching all site settings from Typesense")
    const searchResults = await typesenseClient.collections("site_settings").documents().search({
      q: "*",
      per_page: 250,
    })

    console.log("[v0] Found", searchResults.found, "site settings in database")

    const settings: Record<string, any> = {}

    // Add default settings first
    const defaults = getDefaultSiteSettings()
    Object.assign(settings, defaults)
    console.log("[v0] Loaded default settings:", defaults)

    // Override with stored settings
    for (const hit of searchResults.hits || []) {
      const setting = hit.document
      console.log("[v0] Processing setting:", setting.key, "=", setting.value, "type:", setting.type)

      switch (setting.type) {
        case "json":
          try {
            settings[setting.key] = JSON.parse(setting.value)
            console.log("[v0] Parsed JSON setting", setting.key, ":", settings[setting.key])
          } catch (parseError) {
            console.error("[v0] Error parsing JSON for", setting.key, ":", parseError)
            // Keep default value
          }
          break
        case "boolean":
          settings[setting.key] = setting.value === "true"
          console.log("[v0] Parsed boolean setting", setting.key, ":", settings[setting.key])
          break
        default:
          settings[setting.key] = setting.value
          console.log("[v0] Set text setting", setting.key, ":", settings[setting.key])
      }
    }

    console.log("[v0] Final settings object:", settings)
    return settings
  } catch (error) {
    console.error("[TYPESENSE] Error getting all site settings:", error)
    console.log("[v0] Falling back to default settings due to error")
    return getDefaultSiteSettings()
  }
}

function getDefaultSiteSetting(key: string): any {
  const defaults = getDefaultSiteSettings()
  return defaults[key] || null
}

function getDefaultSiteSettings(): Record<string, any> {
  return {
    // General Settings
    siteName: "IndzsSearch",
    siteTitle: "IndzsSearch - Advanced Search Engine",
    siteDescription:
      "Discover the web with IndzsSearch - your comprehensive search engine for web results, images, and more.",
    siteKeywords: "search engine, web search, image search, IndzsSearch",

    // Appearance Settings
    themeColor: "#df7016", // emerald-500
    favicon: "/favicon.ico",
    logo: "/logo.png",

    // Homepage Settings
    welcomeMessage: "Search the web with IndzsSearch",
    popularSearches: ["technology", "news", "weather", "sports", "entertainment"],
    showSubmitWebsite: true,

    // SEO Settings
    ogImage: "/og-image.png",
    twitterCard: "summary_large_image",
  }
}

export default typesenseClient

export { typesenseClient as client }
