
import typesenseClient, { WebsiteSubmission, createWebsitesCollection } from "@/lib/typesense"

async function checkExistingWebsites(urls: string[]) {
  const filter = `url:=[${urls.map(u => `"${u}"`).join(",")}]`

  const result = await typesenseClient
    .collections("websites")
    .documents()
    .search({
      q: "*",
      query_by: "title",
      filter_by: filter,
      per_page: urls.length
    })

  return result.hits?.map(hit => hit.document.url) || []
}

function generalizeUrl(input: string): string {
  try {
    const url = new URL(input)

    return `${url.protocol}//${url.hostname}`
  } catch {
    return input
  }
}


function normalizeWebsiteUrl(input: string): string | null {
  try {
    const url = new URL(input)

    // Remove www + force https
    const hostname = url.hostname.replace(/^www\./, "")

    return `https://${hostname}`
  } catch {
    // Invalid URL → exclude
    return null
  }
}




async function getExistingUrls(urls: string[]) {
  if (urls.length === 0) return []
  const filter = `url:=[${urls.map(u => `"${u}"`).join(",")}]`

  console.log("filter")
  console.log(filter)

  const result = await typesenseClient
    .collections("websites")
    .documents()
    .search({
      q: "*",
      query_by: "title",
      filter_by: filter,
      per_page: urls.length
    })

    console.log("Fetch results from type sence")
    console.log(result)

  return result.hits?.map(hit => hit.document.url) || []
}


export async function submitWebsitesBulk(
  websites: Omit<WebsiteSubmission, "id" | "submittedAt" | "status">[]
) {
  try {
    await createWebsitesCollection()

    // Map original → normalized
    const normalizedMap = new Map<string, typeof websites[0]>()

    for (const site of websites) {
      const normalized = normalizeWebsiteUrl(site.url)

      if (!normalized) continue // ❌ invalid URL excluded

      normalizedMap.set(normalized, {
        ...site,
        url: normalized
      })
    }

    const normalizedUrls = Array.from(normalizedMap.keys())


    console.log("Normalized urls")
    console.log(normalizedUrls)
    
    const urls = websites.map(site => site.url)
    const existingUrls = await getExistingUrls(urls)
    console.log("Existing Urls")
    console.log(existingUrls)



    const websitesToInsert = websites.filter(
      site => !existingUrls.includes(site.url)
    )

    console.log(websitesToInsert)


    if (websitesToInsert.length == 0) {
      return { successCount: 0, failedCount: 0, skippedCount: websites.length }
    }

    const documents = websitesToInsert.map((website) => ({
      ...website,
      id: Math.random().toString(36).slice(2, 10),
      status: "pending" as const,
      submittedAt: new Date().toISOString(),
    }))




    const result = await typesenseClient
      .collections("websites")
      .documents()
      .import(documents, { action: "create" })

    const successCount = result.filter((r) => r.success).length
    const failedCount = result.length - successCount
    const skippedCount = websites.length - successCount - failedCount

    console.log(`[TYPESENSE] Bulk submitted ${successCount} websites`)

    return { successCount, failedCount, skippedCount }
  } catch (error) {
    console.error("[TYPESENSE] Bulk submit error:", error)
    throw error
  }
}




// function normalizeUrl(url: string) {
//   return url.trim().toLowerCase().replace(/\/$/, "")
// }

// export async function submitWebsitesBulk(
//   websites: Omit<WebsiteSubmission, "id" | "submittedAt" | "status">[]
// ) {
//   await createWebsitesCollection()

//   // Normalize URLs
//   const normalizedMap = new Map(
//     websites.map(w => [normalizeUrl(w.url), w])
//   )

//   const normalizedUrls = [...normalizedMap.keys()]

//   // 🔍 Find existing websites
//   const searchResult = await typesenseClient
//     .collections("websites")
//     .documents()
//     .search({
//       q: "*",
//       query_by: "normalizedUrl",
//       filter_by: `normalizedUrl:=[${normalizedUrls.map(u => `"${u}"`).join(",")}]`,
//       per_page: normalizedUrls.length,
//     })

//   const existingUrlSet = new Set(
//     (searchResult.hits || []).map(
//       (hit: any) => hit.document.normalizedUrl
//     )
//   )

//   // 🚫 Skip existing URLs
//   const newWebsites = websites.filter(
//     w => !existingUrlSet.has(normalizeUrl(w.url))
//   )

//   // Nothing new to submit
//   if (newWebsites.length === 0) {
//     return {
//       submitted: 0,
//       skipped: websites.length,
//       failed: 0,
//     }
//   }

//   const documents = newWebsites.map((website) => ({
//     ...website,
//     normalizedUrl: normalizeUrl(website.url),
//     id: Math.random().toString(36).slice(2, 10),
//     status: "pending" as const,
//     submittedAt: new Date().toISOString(),
//   }))

//   const result = await typesenseClient
//     .collections("websites")
//     .documents()
//     .import(documents, { action: "create" })

//   const successCount = result.filter(r => r.success).length

//   return {
//     submitted: successCount,
//     skipped: websites.length - newWebsites.length,
//     failed: result.length - successCount,
//   }
// }
