import typesenseClient from "@/lib/typesense"
import crypto from "crypto"

const COLLECTION = "image_results"
const CHUNK_SIZE = 1000


function generateCompoundId(sourceUrl: string, imageUrl: string) {
    return crypto
        .createHash("sha1")
        .update(`${sourceUrl}::${imageUrl}`)
        .digest("hex")
}


/* ------------------ Ensure Collection ------------------ */
async function ensureCollection() {
    try {
        await typesenseClient.collections(COLLECTION).retrieve()
    } catch {
        await typesenseClient.collections().create({
            name: COLLECTION,
            fields: [
                { name: "title", type: "string" },
                { name: "url", type: "string" },
                { name: "imgurl", type: "string" },
                { name: "keywords", type: "string[]", optional: true },
                { name: "date", type: "string", optional: true },
                { name: "safesearch", type: "bool" },
                { name: "favicon", type: "string", optional: true },
            ],
        })
    }
}

/* ------------------ Helpers ------------------ */
function normalizeUrl(url: string) {
    if (!url) return ""
    return url.startsWith("http") ? url : `https://${url}`
}

function normalizeDoc(doc: any) {
    const img = doc.imgurl || doc.image
    if (!doc.title || !img || !doc.url) return null

    return {
        id: generateCompoundId(normalizeUrl(doc.url), String(img)),
        title: String(doc.title),
        imgurl: String(img),
        url: normalizeUrl(doc.url),
        keywords: Array.isArray(doc.keywords) ? doc.keywords : [],
        safesearch: true,
        favicon: doc.favicon || undefined,
    }
}

function chunk<T>(arr: T[], size: number) {
    const chunks: T[][] = []
    for (let i = 0; i < arr.length; i += size) {
        chunks.push(arr.slice(i, i + size))
    }
    return chunks
}

/* ------------------ Public API ------------------ */
export async function submitImagesBulk(data: any[]) {
    if (!Array.isArray(data)) {
        throw new Error("Expected an array of image objects")
    }

    await ensureCollection()

    /* ---- Normalize + Deduplicate ---- */
    const seen = new Set<string>()
    const docs: any[] = []

    for (const item of data) {
        const doc = normalizeDoc(item)
        if (!doc) continue

        if (!seen.has(doc.imgurl)) {
            seen.add(doc.imgurl)
            docs.push(doc)
        }
    }

    const batches = chunk(docs, CHUNK_SIZE)

    let imported = 0
    let failed = 0
    for (const batch of batches) {
        try {
            const res = await typesenseClient
                .collections(COLLECTION)
                .documents()
                .import(batch, { action: "create" })

            const results = Array.isArray(res)
                ? res
                : res.split("\n").map((l: string) => JSON.parse(l))

            for (const r of results) {
                r.success ? imported++ : failed++
            }
        } catch (error: any) {
            // 🔑 THIS IS THE IMPORTANT PART
            const results = error?.importResults

            if (Array.isArray(results)) {
                for (const r of results) {
                    if (r.success) {
                        imported++
                    } else if (r.error?.includes("already exists")) {
                        // duplicate → SKIP (expected)
                    } else {
                        failed++
                    }
                }
            } else {
                throw error
            }
        }

        // for (const batch of batches) {
        //     const res = await typesenseClient
        //         .collections(COLLECTION)
        //         .documents()
        //         .import(batch, { action: "create" })

        //     const results = Array.isArray(res)
        //         ? res
        //         : res.split("\n").map((l: string) => JSON.parse(l))

        //     // for (const r of results) {
        //     //   r.success ? imported++ : failed++
        //     // }

        //     for (const r of results) {
        //         if (r.success) {
        //             imported++
        //         } else if (r.error?.includes("already exists")) {
        //             // duplicate → skipped (expected)
        //         } else {
        //             failed++
        //         }
        //     }

    }

    return {
        received: data.length,
        uniqueImages: docs.length,
        imported,
        failed,
    }
}
