import { URL } from "url"

export interface RawWebsite {
  title?: string
  link?: string
  favicon?: string
  thumbnail?: string
  displayUrl?: string
}

export interface NormalizedWebsite {
  title: string
  url: string
  imageUrl: string
  supportImageCrawl: boolean
  supportWebCrawl: boolean
  ownerConfirmed: boolean
}


function resolveUrl(displayUrl?: string, link?: string): string {
  try {
    if (displayUrl) {
      return new URL(
        displayUrl.startsWith("http")
          ? displayUrl
          : `https://${displayUrl}`
      ).toString()
    }

    if (link) {
      return new URL(link).toString()
    }

    return "not-a-valid-url"
  } catch {
    return "not-a-valid-url"
  }
}


/**
 * Safely validates URL
 */
function isValidUrl(url?: string): boolean {
  if (!url) return false
  try {
    new URL(url)
    return true
  } catch {
    return false
  }
}

/**
 * Generates Google favicon fallback
 */
function getFallbackFavicon(url: string) {
  try {
    const hostname = new URL(url).hostname
    return `https://www.google.com/s2/favicons?domain=${hostname}&sz=128`
  } catch {
    return "https://www.google.com/s2/favicons?domain=example.com&sz=128"
  }
}

/**
 * Normalize raw search websites into production-ready format
 */
export function normalizeWebsites(
  rawWebsites: RawWebsite[]
): NormalizedWebsite[] {
  return rawWebsites.map((site) => {
    // const url = isValidUrl(site.link) ? site.link! : "not-a-valid-url"
    
    const url = resolveUrl(site.displayUrl, site.link)

    const imageUrl =
      site.thumbnail ||
      site.favicon ||
      (url !== "not-a-valid-url"
        ? getFallbackFavicon(url)
        : "https://typesense.org/assets/images/typesense-logo.png")

    return {
      title: site.title?.trim() || "Untitled Website",
      url,
      imageUrl,
      supportImageCrawl: true,
      supportWebCrawl: true,
      ownerConfirmed: true
    }
  })
}
