import nodemailer from "nodemailer"

// Email service configuration
export function createTransporter() {
  const smtpConfig = {
    host: process.env.SMTP_HOST,
    port: Number.parseInt(process.env.SMTP_PORT || "587"),
    secure: process.env.SMTP_SECURE === "true", // true for SSL, false for TLS
    auth: {
      user: process.env.SMTP_USER,
      pass: process.env.SMTP_PASS,
    },
  }

  console.log("[EMAIL] SMTP Configuration:", {
    host: smtpConfig.host,
    port: smtpConfig.port,
    secure: smtpConfig.secure,
    hasAuth: !!smtpConfig.auth.user,
  })

  // Fix: Use createTransport instead of createTransporter
  return nodemailer.createTransport(smtpConfig)
}

// Send verification email
export async function sendVerificationEmail(email: string, username: string, verificationUrl: string) {
  try {
    const transporter = createTransporter()

    const mailOptions = {
      from: process.env.SMTP_FROM || process.env.SMTP_USER,
      to: email,
      subject: "Verify Your IndusSearch Account",
      html: `
        <!DOCTYPE html>
        <html>
        <head>
          <meta charset="utf-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>Verify Your IndusSearch Account</title>
        </head>
        <body style="font-family: Arial, sans-serif; line-height: 1.6; color: #333; max-width: 600px; margin: 0 auto; padding: 20px;">
          <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 30px; text-align: center; border-radius: 10px 10px 0 0;">
            <h1 style="color: white; margin: 0; font-size: 28px;">Welcome to IndusSearch!</h1>
          </div>
          
          <div style="background: #f9f9f9; padding: 30px; border-radius: 0 0 10px 10px;">
            <p style="font-size: 16px; margin-bottom: 20px;">Hi ${username},</p>
            
            <p style="font-size: 16px; margin-bottom: 25px;">
              Thank you for registering with IndusSearch. Please verify your email address by clicking the button below:
            </p>
            
            <div style="text-align: center; margin: 30px 0;">
              <a href="${verificationUrl}" style="background: #10b981; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; font-weight: bold; display: inline-block;">Verify Email Address</a>
            </div>
            
            <p style="font-size: 14px; color: #666; margin-top: 25px;">
              Or copy and paste this link into your browser:
            </p>
            <p style="font-size: 14px; color: #10b981; word-break: break-all;">
              ${verificationUrl}
            </p>
            
            <p style="font-size: 14px; color: #666; margin-top: 25px;">
              This verification link will expire in 24 hours.
            </p>
            
            <p style="font-size: 14px; color: #666; margin-top: 25px;">
              If you didn't create an account with IndusSearch, please ignore this email.
            </p>
            
            <hr style="border: none; border-top: 1px solid #eee; margin: 30px 0;">
            
            <p style="font-size: 14px; color: #666; text-align: center;">
              Best regards,<br>
              The IndusSearch Team
            </p>
          </div>
        </body>
        </html>
      `,
      text: `
        Welcome to IndusSearch!
        
        Hi ${username},
        
        Thank you for registering with IndusSearch. Please verify your email address by visiting this link:
        
        ${verificationUrl}
        
        This verification link will expire in 24 hours.
        
        If you didn't create an account with IndusSearch, please ignore this email.
        
        Best regards,
        The IndusSearch Team
      `,
    }

    const result = await transporter.sendMail(mailOptions)
    console.log("[EMAIL] Verification email sent successfully:", result.messageId)
    return result
  } catch (error) {
    console.error("[EMAIL] Error sending verification email:", error)
    throw error
  }
}

// Send password reset email
export async function sendPasswordResetEmail(email: string, username: string, resetUrl: string) {
  try {
    const transporter = createTransporter()

    const mailOptions = {
      from: process.env.SMTP_FROM || process.env.SMTP_USER,
      to: email,
      subject: "Reset Your IndusSearch Password",
      html: `
        <!DOCTYPE html>
        <html>
        <head>
          <meta charset="utf-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>Reset Your IndusSearch Password</title>
        </head>
        <body style="font-family: Arial, sans-serif; line-height: 1.6; color: #333; max-width: 600px; margin: 0 auto; padding: 20px;">
          <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 30px; text-align: center; border-radius: 10px 10px 0 0;">
            <h1 style="color: white; margin: 0; font-size: 28px;">Password Reset Request</h1>
          </div>
          
          <div style="background: #f9f9f9; padding: 30px; border-radius: 0 0 10px 10px;">
            <p style="font-size: 16px; margin-bottom: 20px;">Hi ${username},</p>
            
            <p style="font-size: 16px; margin-bottom: 25px;">
              We received a request to reset your IndusSearch account password. Click the button below to create a new password:
            </p>
            
            <div style="text-align: center; margin: 30px 0;">
              <a href="${resetUrl}" style="background: #ef4444; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; font-weight: bold; display: inline-block;">Reset Password</a>
            </div>
            
            <p style="font-size: 14px; color: #666; margin-top: 25px;">
              Or copy and paste this link into your browser:
            </p>
            <p style="font-size: 14px; color: #ef4444; word-break: break-all;">
              ${resetUrl}
            </p>
            
            <p style="font-size: 14px; color: #666; margin-top: 25px;">
              This password reset link will expire in 1 hour for security reasons.
            </p>
            
            <p style="font-size: 14px; color: #666; margin-top: 25px;">
              If you didn't request a password reset, please ignore this email. Your password will remain unchanged.
            </p>
            
            <hr style="border: none; border-top: 1px solid #eee; margin: 30px 0;">
            
            <p style="font-size: 14px; color: #666; text-align: center;">
              Best regards,<br>
              The IndusSearch Team
            </p>
          </div>
        </body>
        </html>
      `,
      text: `
        Password Reset Request
        
        Hi ${username},
        
        We received a request to reset your IndusSearch account password. Visit this link to create a new password:
        
        ${resetUrl}
        
        This password reset link will expire in 1 hour for security reasons.
        
        If you didn't request a password reset, please ignore this email. Your password will remain unchanged.
        
        Best regards,
        The IndusSearch Team
      `,
    }

    const result = await transporter.sendMail(mailOptions)
    console.log("[EMAIL] Password reset email sent successfully:", result.messageId)
    return result
  } catch (error) {
    console.error("[EMAIL] Error sending password reset email:", error)
    throw error
  }
}

// Test email configuration
export async function testEmailConfiguration() {
  try {
    const transporter = createTransporter()
    const verified = await transporter.verify()
    console.log("[EMAIL] SMTP configuration is valid:", verified)
    return verified
  } catch (error) {
    console.error("[EMAIL] SMTP configuration test failed:", error)
    return false
  }
}

// Helper function for sending any email
export async function sendEmail(to: string, subject: string, html: string, text?: string) {
  try {
    const transporter = createTransporter()
    
    const mailOptions = {
      from: process.env.SMTP_FROM || process.env.SMTP_USER,
      to,
      subject,
      html,
      text: text || html.replace(/<[^>]*>/g, ''), // Strip HTML for text version if not provided
    }

    const result = await transporter.sendMail(mailOptions)
    console.log("[EMAIL] Email sent successfully:", result.messageId)
    return result
  } catch (error) {
    console.error("[EMAIL] Error sending email:", error)
    throw error
  }
}
