export interface SearchResult {
  id: string
  title: string
  description: string
  url: string
  favicon?: string
  date?: string
  keywords?: string[]
  safesearch?: boolean
  maindomain?: boolean
}

export interface CategorizedResults {
  [baseUrl: string]: {
    main: SearchResult | null
    subs: SearchResult[]
  }
}


export default function categorizeByBaseUrl(
  data: SearchResult[]
): CategorizedResults {
  const temp: {
    [baseUrl: string]: {
      main: SearchResult | null
      subs: SearchResult[]
    }
  } = {}

  for (const item of data) {
    let parsedUrl: URL

    try {
      parsedUrl = new URL(item.url)
    } catch {
      continue
    }

    const baseUrl = `${parsedUrl.protocol}//${parsedUrl.hostname}`
    const path = parsedUrl.pathname.replace(/\/+$/, "")

    const isMain =
      item.maindomain === true ||
      path === "" ||
      path === "/"

    if (!temp[baseUrl]) {
      temp[baseUrl] = { main: null, subs: [] }
    }

    if (isMain && !temp[baseUrl].main) {
      temp[baseUrl].main = item
    } else {
      temp[baseUrl].subs.push(item)
    }
  }

  // 🔥 Ensure main is never null
  const result: CategorizedResults = {}

  for (const [baseUrl, group] of Object.entries(temp)) {
    if (!group.main) {
      // pick best candidate: shortest path
      group.subs.sort((a, b) => {
        const aDepth = new URL(a.url).pathname.split("/").length
        const bDepth = new URL(b.url).pathname.split("/").length
        return aDepth - bDepth
      })

      group.main = group.subs.shift()! // promote
    }

    result[baseUrl] = {
      main: group.main,
      subs: group.subs
    }
  }

  return result
}




// export default function categorizeByBaseUrl(
//   data: SearchResult[]
// ): CategorizedResults {
//   const result: CategorizedResults = {}

//   for (const item of data) {
//     let parsedUrl: URL

//     try {
//       parsedUrl = new URL(item.url)
//     } catch {
//       // skip invalid URLs safely
//       continue
//     }

//     const baseUrl = `${parsedUrl.protocol}//${parsedUrl.hostname}`
//     const path = parsedUrl.pathname.replace(/\/+$/, "") // normalize

//     const isMain =
//       item.maindomain === true ||
//       path === "" ||
//       path === "/"

//     if (!result[baseUrl]) {
//       result[baseUrl] = {
//         main: null,
//         subs: []
//       }
//     }

//     if (isMain && !result[baseUrl].main) {
//       result[baseUrl].main = item
//     } else {
//       result[baseUrl].subs.push(item)
//     }
//   }

//   return result
// }


export function dedupeByUrl(
  data: SearchResult[]
): SearchResult[] {
  return Array.from(
    new Map(data.map(item => [item.url, item])).values()
  )
}


export function sortSubsByDate(results: CategorizedResults) {
  Object.values(results).forEach(group => {
    group.subs.sort(
      (a, b) =>
        new Date(b.date ?? 0).getTime() -
        new Date(a.date ?? 0).getTime()
    )
  })
}


// const clean = dedupeByUrl(data)
// const categorized = categorizeByBaseUrl(clean)
// sortSubsByDate(categorized)
