"use client"
import { useState, useEffect, useCallback } from "react";

const STORAGE_KEY = "google_search_history";
const MAX_ITEMS = 10;

export function useSearchHistory() {
  const [history, setHistory] = useState<string[]>([]);

  // Initialize history from localStorage on mount
  useEffect(() => {
    const saved = localStorage.getItem(STORAGE_KEY);
    if (saved) {
      try {
        setHistory(JSON.parse(saved));
      } catch (e) {
        console.error("Failed to parse history", e);
        setHistory([]);
      }
    }
  }, []);

  // Save a new search term
  const addToHistory = useCallback((query: string) => {
    if (!query || !query.trim()) return;

    setHistory((prev) => {
      const cleanQuery = query.trim();
      // Remove duplicate and move to top
      const filtered = prev.filter(
        (item) => item.toLowerCase() !== cleanQuery.toLowerCase()
      );
      const updated = [cleanQuery, ...filtered].slice(0, MAX_ITEMS);
      
      localStorage.setItem(STORAGE_KEY, JSON.stringify(updated));
      return updated;
    });
  }, []);

  // Delete a specific item
  const deleteFromHistory = useCallback((index: number) => {
    setHistory((prev) => {
      const updated = prev.filter((_, i) => i !== index);
      localStorage.setItem(STORAGE_KEY, JSON.stringify(updated));
      return updated;
    });
  }, []);

  // Clear everything
  const clearHistory = useCallback(() => {
    localStorage.removeItem(STORAGE_KEY);
    setHistory([]);
  }, []);

  return {
    history,
    addToHistory,
    deleteFromHistory,
    clearHistory
  };
}