"use client";

/* ------------------ TYPES ------------------ */

export type SearchType = "web" | "images" | "shopping" | "wiki" | "videos" | "news";

export interface SearchHistoryItem {
  id: string;
  query: string;
  type: SearchType;
  filters?: Record<string, any>;
  resultsState?: any;
  createdAt: number;
}

export interface BrowserHistoryItem {
  id: string;
  url: string;
  title?: string;
  source: "search" | "direct";
  searchId?: string;
  visitedAt: number;
}

/* ------------------ KEYS ------------------ */

const SEARCH_KEY = "__search_history__";
const BROWSER_KEY = "__browser_history__";

/* ------------------ UTILS ------------------ */

const generateId = () =>
  typeof crypto !== "undefined" && crypto.randomUUID
    ? crypto.randomUUID()
    : Math.random().toString(36).slice(2) + Date.now();

const read = <T,>(key: string): T[] => {
  if (typeof window === "undefined") return [];
  try {
    return JSON.parse(localStorage.getItem(key) || "[]");
  } catch {
    return [];
  }
};

const write = <T,>(key: string, data: T[]) => {
  localStorage.setItem(key, JSON.stringify(data));
};

/* ======================================================
   🔍 SEARCH HISTORY FUNCTIONS
====================================================== */

export const saveSearchHistory = (
  data: Omit<SearchHistoryItem, "id" | "createdAt">
): SearchHistoryItem => {
  const history = read<SearchHistoryItem>(SEARCH_KEY);

  const item: SearchHistoryItem = {
    ...data,
    id: generateId(),
    createdAt: Date.now(),
  };

  write(SEARCH_KEY, [item, ...history].slice(0, 50));
  return item;
};

export const getSearchHistory = (): SearchHistoryItem[] =>
  read<SearchHistoryItem>(SEARCH_KEY);

export const getLastSearch = (): SearchHistoryItem | null =>
  read<SearchHistoryItem>(SEARCH_KEY)[0] || null;

export const updateSearchState = (
  searchId: string,
  resultsState: any
) => {
  const history = read<SearchHistoryItem>(SEARCH_KEY);

  write(
    SEARCH_KEY,
    history.map(item =>
      item.id === searchId ? { ...item, resultsState } : item
    )
  );
};

export const removeSearchHistory = (searchId: string) => {
  write(
    SEARCH_KEY,
    read<SearchHistoryItem>(SEARCH_KEY).filter(
      item => item.id !== searchId
    )
  );
};

export const clearSearchHistory = () => {
  localStorage.removeItem(SEARCH_KEY);
};

/* ======================================================
   🌐 BROWSER HISTORY FUNCTIONS
====================================================== */

export const saveBrowserHistory = (
  data: Omit<BrowserHistoryItem, "id" | "visitedAt">
) => {
  const history = read<BrowserHistoryItem>(BROWSER_KEY);

  const item: BrowserHistoryItem = {
    ...data,
    id: generateId(),
    visitedAt: Date.now(),
  };

  write(BROWSER_KEY, [item, ...history].slice(0, 100));
};

export const getBrowserHistory = (): BrowserHistoryItem[] =>
  read<BrowserHistoryItem>(BROWSER_KEY);

export const getBrowserHistoryBySearch = (searchId: string) =>
  read<BrowserHistoryItem>(BROWSER_KEY).filter(
    item => item.searchId === searchId
  );

export const removeBrowserHistory = (id: string) => {
  write(
    BROWSER_KEY,
    read<BrowserHistoryItem>(BROWSER_KEY).filter(
      item => item.id !== id
    )
  );
};

export const clearBrowserHistory = () => {
  localStorage.removeItem(BROWSER_KEY);
};

/* ======================================================
   🧹 GLOBAL HELPERS
====================================================== */

export const clearAllHistory = () => {
  clearSearchHistory();
  clearBrowserHistory();
};
