"use client"

import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Button } from "@/components/ui/button"
import { ExternalLink, Calendar, X, Play, ThumbsUp, Download, ThumbsDown, Share2 } from "lucide-react"

interface VideoResult {
  id: string
  title: string
  thumbnail: string
  url: string
  date?: string
  safesearch: boolean
  channelTitle?: string
  description?: string
}

interface VideoModalProps {
  video: VideoResult | null
  isOpen: boolean
  onClose: () => void
}


const decodeHTMLEntities = (text) => {
  if (!text) return "";
  const textArea = document.createElement("textarea");
  textArea.innerHTML = text;
  return textArea.value;
};

export function VideoModal({ video, isOpen, onClose }: VideoModalProps) {
  if (!video) return null

  // Extract YouTube video ID from URL
  const getYouTubeVideoId = (url: string) => {
    const regex = /(?:youtube\.com\/watch\?v=|youtu\.be\/|youtube\.com\/embed\/)([^&\n?#]+)/
    const match = url.match(regex)
    return match ? match[1] : null
  }

  const videoId = getYouTubeVideoId(video.url)
  const embedUrl = videoId ? `https://www.youtube.com/embed/${videoId}` : null

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>



      <DialogContent className="max-w-5xl w-full sm:w-[90vw] h-full sm:h-auto sm:max-h-[92vh] overflow-y-auto border-none sm:border-border/40 bg-background p-0 gap-0 sm:rounded-3xl flex flex-col">

        {/* Mobile: Floating Close | Desktop: Corner Close */}
        <div className="absolute top-2 right-2 sm:top-4 sm:right-4 z-50 hidden sm:flex">
          <Button
            variant="ghost"
            size="icon"
            onClick={(e) => {
              e.stopPropagation(); // Prevents touch from bubbling to the video
              onClose();
            }}
            className="rounded-full bg-black/40 text-white backdrop-blur-md h-9 w-9"
          >
            <X className="w-5 h-5" />
          </Button>
        </div>

        {/* 1. Player Section (Sticks to top on mobile) */}
        <div className="sticky top-0 sm:relative w-full aspect-video bg-black z-40">
          {embedUrl ? (
            <iframe src={`${embedUrl}?autoplay=1`} title={video.title} className="w-full h-full" frameBorder="0" allowFullScreen />
          ) : (
              <div className="w-full h-full bg-secondary/50 flex items-center justify-center">Preview Unavailable</div>
            )}
        </div>

        {/* 2. Content Info */}
        <div className="p-4 sm:p-8 flex-1 flex flex-col">
          <h2 className="text-[18px] sm:text-2xl font-semibold leading-tight mb-3">
            {decodeHTMLEntities(video.title)}
          </h2>

          {/* Action Bar (YouTube Mobile Style) */}
          {/* <div className="flex items-center gap-4 overflow-x-auto pb-4 no-scrollbar">
        <div className="flex items-center gap-2 bg-secondary px-3 py-1.5 rounded-full shrink-0">
          <ThumbsUp className="w-4 h-4" />
          <span className="text-xs font-medium">1.2K</span>
          <div className="w-[1px] h-3 bg-border mx-1" />
          <ThumbsDown className="w-4 h-4" />
        </div>
        <div className="flex items-center gap-2 bg-secondary px-4 py-1.5 rounded-full shrink-0">
          <Share2 className="w-4 h-4" />
          <span className="text-xs font-medium">Share</span>
        </div>
        <div className="flex items-center gap-2 bg-secondary px-4 py-1.5 rounded-full shrink-0">
          <Download className="w-4 h-4" />
          <span className="text-xs font-medium">Download</span>
        </div>
      </div> */}

          <hr className="border-border/40 my-4" />


          <div className="flex items-center justify-between mb-3">
            <div className="flex items-center gap-3">
              <div className="w-10 h-10 rounded-full bg-secondary shrink-0 overflow-hidden">
                <img src={video.thumbnail} className="w-full h-full object-cover" />
              </div>
              <div>
                <p className="text-sm font-bold">{video.channelTitle}</p>
                {/* <p className="text-[11px] text-muted-foreground">4.2M subscribers</p> */}
              </div>
            </div>
            {/* <Button className="rounded-full bg-foreground text-background h-9 px-4 text-xs font-bold">
          Subscribe
        </Button> */}
          </div>

          {/* Description Preview */}
          <div className="bg-secondary/0 p-0 rounded-xl">
            <p className="text-xs font-bold mb-2">
              {video.date ? new Date(video.date).toLocaleDateString() : "Oct 14, 2025"}
            </p>
            <p className="text-sm text-muted-foreground font-sans line-clamp-3">
              {decodeHTMLEntities(video.description)}
            </p>
          </div>

          <div className="flex gap-3 pt-4 border-t mt-auto">
            <Button asChild className="flex items-center gap-2">
              <a href={video.url} target="_blank" rel="noopener noreferrer">
                <ExternalLink className="w-4 h-4" />
                   Open on YouTube
                 </a>
            </Button>
            <Button variant="outline" onClick={onClose}>
              Close
               </Button>
          </div>
        </div>
      </DialogContent>
    </Dialog>

    // <Dialog open={isOpen} onOpenChange={onClose}>
    //   <DialogContent className="max-w-4xl w-full max-h-[90vh] overflow-y-auto">
    //     <DialogHeader>
    //       <DialogTitle className="text-lg font-semibold line-clamp-2 pr-8">{video.title}</DialogTitle>
    //     </DialogHeader>

    //     <div className="space-y-4">
    //       {/* Video Embed */}
    //       {embedUrl ? (
    //         <div className="aspect-video w-full bg-black rounded-lg overflow-hidden">
    //           <iframe
    //             src={embedUrl}
    //             title={video.title}
    //             className="w-full h-full"
    //             frameBorder="0"
    //             allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
    //             allowFullScreen
    //           />
    //         </div>
    //       ) : (
    //         <div className="aspect-video w-full bg-muted rounded-lg flex items-center justify-center">
    //           <img
    //             src={video.thumbnail || "/placeholder.svg"}
    //             alt={video.title}
    //             className="max-w-full max-h-full object-contain"
    //           />
    //         </div>
    //       )}

    //       {/* Video Info */}
    //       <div className="space-y-3">
    //         {video.channelTitle && (
    //           <div className="flex items-center gap-2">
    //             <span className="font-medium text-accent">{video.channelTitle}</span>
    //           </div>
    //         )}

    //         {video.date && (
    //           <div className="flex items-center gap-2 text-sm text-muted-foreground">
    //             <Calendar className="w-4 h-4" />
    //             <span>
    //               {new Date(video.date).toLocaleDateString("en-US", {
    //                 year: "numeric",
    //                 month: "long",
    //                 day: "numeric",
    //               })}
    //             </span>
    //           </div>
    //         )}

    //         {video.description && (
    //           <div className="space-y-2">
    //             <h4 className="font-medium text-sm">Description</h4>
    //             <p className="text-sm text-muted-foreground leading-relaxed">{video.description}</p>
    //           </div>
    //         )}

    //         {/* Action Buttons */}
    //         <div className="flex gap-3 pt-4 border-t">
    //           <Button asChild className="flex items-center gap-2">
    //             <a href={video.url} target="_blank" rel="noopener noreferrer">
    //               <ExternalLink className="w-4 h-4" />
    //               Open on YouTube
    //             </a>
    //           </Button>
    //           <Button variant="outline" onClick={onClose}>
    //             Close
    //           </Button>
    //         </div>
    //       </div>
    //     </div>
    //   </DialogContent>
    // </Dialog>
  )
}
