"use client"

import { useEffect, useState, useRef } from "react"
import AddWebsiteModal from "./AddWebsiteModal"
import { useHistory } from '@/hooks/useHistory'
import "@/styles/quicklinks.css"
export type LinkItem = {
  id: string
  name: string
  url: string
  favicon: string
  visits: number
  color: string
}

const DEFAULT_LINKS: LinkItem[] = [
  {
    id: "youtube",
    name: "YouTube",
    url: "https://youtube.com",
    favicon: "https://www.google.com/s2/favicons?domain=youtube.com&sz=64",
    visits: 0,
    color: "#FFECEC",
  },
  {
    id: "facebook",
    name: "Facebook",
    url: "https://facebook.com",
    favicon: "https://www.google.com/s2/favicons?domain=facebook.com&sz=64",
    visits: 0,
    color: "#e5e8f6",
  },

  {
    id: "instagram",
    name: "Instagram",
    url: "https://instagram.com",
    favicon: "https://www.google.com/s2/favicons?domain=instagram.com&sz=64",
    visits: 0,
    color: "#fae6ef",
  },
  {
    id: "whatsapp",
    name: "Whatsapp",
    url: "https://web.whatsapp.com/",
    favicon: "https://www.google.com/s2/favicons?domain=whatsapp.com&sz=64",
    visits: 0,
    color: "#cffadf",
  },

]

export default function QuickLinks() {
  const [links, setLinks] = useState<LinkItem[]>(DEFAULT_LINKS)
  const [showModal, setShowModal] = useState(false)
  const [deletingId, setDeletingId] = useState<string | null>(null); // Track active delete button
  const timerRef = useRef<NodeJS.Timeout | null>(null);
  const { saveBrowserHistory } = useHistory();
  // /* Load from localStorage */
  // useEffect(() => {
  //   const saved = localStorage.getItem("quick_links")
  //   if (saved) setLinks(JSON.parse(saved))
  // }, [])

  // /* Save */
  // useEffect(() => {
  //   localStorage.setItem("quick_links", JSON.stringify(links))
  // }, [links])

  const [hydrated, setHydrated] = useState(false)

  /* Load */
  useEffect(() => {
    const saved = localStorage.getItem("quick_links")

    if (saved) {
      setLinks(JSON.parse(saved))
    } else {
      setLinks(DEFAULT_LINKS)
    }

    setHydrated(true)
  }, [])

  /* Save (only after hydration) */
  useEffect(() => {
    if (!hydrated) return
    localStorage.setItem("quick_links", JSON.stringify(links))
  }, [links, hydrated])


  const handleVisit = (item: LinkItem) => {
    const updated = links.map((l) =>
      l.id === item.id ? { ...l, visits: l.visits + 1 } : l
    )

    updated.sort((a, b) => b.visits - a.visits) // frequently visited auto top
    setLinks(updated)
    window.open(item.url, "_blank")
  }

  const addWebsite = (site: LinkItem) => {
    setLinks((prev) => [site, ...prev])
  }


  const handleTouchStart = (id: string) => {
    // Start a timer for 600ms
    timerRef.current = setTimeout(() => {
      setDeletingId(id);
    }, 600);
  };

  const handleTouchEnd = () => {
    // If the user lifts their finger before 600ms, cancel the timer
    if (timerRef.current) clearTimeout(timerRef.current);
  };

  const removeLink = (e: React.MouseEvent, id: string) => {
    e.stopPropagation(); // Prevents the link from opening when clicking 'X'
    const updated = links.filter((link) => link.id !== id);
    setLinks(updated);
    setDeletingId(null);
  };


  // Close delete button if user clicks anywhere else
  useEffect(() => {
    const handleClickOutside = () => setDeletingId(null);
    window.addEventListener('click', handleClickOutside);
    return () => window.removeEventListener('click', handleClickOutside);
  }, []);


  return (
    <>
      <div className="quick-links-wrapper">
        <div className="quick-links">
          {links.map((item) => (
            <div key={item.id} className="icon-wrapper">
              <button
                key={item.id}
                onClick={() => {
                  saveBrowserHistory(
                    {
                      url: item.url,
                      title: item.name,
                      source: "direct",

                    }
                  )

                  handleVisit(item)
                }}
                
                className="icon-card"
                style={{ background: item.color }}
                title={item.name}
              >
                <img src={item.favicon} alt={item.name} />
              </button>
              {/* Remove Button */}
              <button
                className="remove-btn"
                onClick={(e) => removeLink(e, item.id)}
                aria-label="Remove link"
              >
                ×
      </button>
            </div>

          ))}

          {/* Add button */}
          <button
            className="icon-card add-card"
            onClick={() => setShowModal(true)}
          >
            +
          </button>
        </div>
      </div>

      {showModal && (
        <AddWebsiteModal
          onClose={() => setShowModal(false)}
          onAdd={addWebsite}
        />
      )}
    </>
  )
}
