import { useEffect, useRef, useState } from "react";
import { Portal } from '@/components/ui/Portal';
import { Loader2 } from 'lucide-react'; // Assuming you have lucide-react, or use a custom SVG

const MENU_URL = process.env.NEXT_PUBLIC_MENU_URL
export default function NineDotMenu() {
    const [isOpen, setIsOpen] = useState(false);
    const [isLoading, setIsLoading] = useState(false);
    const [htmlContent, setHtmlContent] = useState("");
    const menuRef = useRef(null);
    const buttonRef = useRef(null);

    const STORAGE_KEY = "nine_dot_menu_html";

    const toggleMenu = async (e) => {
        e.stopPropagation();

        // If menu is already open, just close it
        if (isOpen) {
            setIsOpen(false);
            return;
        }

        // Check Session Storage first
        const cachedHtml = sessionStorage.getItem(STORAGE_KEY);
        if (cachedHtml) {
            setHtmlContent(cachedHtml);
            setIsOpen(true);
            return;
        }

        // Otherwise, fetch and show loading animation
        setIsLoading(true);
        setIsOpen(true); // Open the portal to show the loading state

        try {
            const res = await fetch(MENU_URL);
            if (!res.ok) throw new Error("Failed to load file");

            const html = await res.text();
            
            // Save to session storage
            sessionStorage.setItem(STORAGE_KEY, html);
            
            setHtmlContent(html);
        } catch (err) {
            console.error(err);
            setHtmlContent("<div className='p-4 text-red-500'>Error loading menu content.</div>");
        } finally {
            setIsLoading(false);
        }
    };

    // Close when clicking outside
    useEffect(() => {
        const handleClickOutside = (e) => {
            if (
                isOpen &&
                menuRef.current &&
                !menuRef.current.contains(e.target) &&
                !buttonRef.current.contains(e.target)
            ) {
                setIsOpen(false);
            }
        };

        document.addEventListener("click", handleClickOutside);
        return () => document.removeEventListener("click", handleClickOutside);
    }, [isOpen]);

    return (
        <>
            {/* Toggle Button */}
            <div
                ref={buttonRef}
                title="Open Menu"
                onClick={toggleMenu}
                className="menu-toggle p-2 rounded-full hover:bg-accent text-foreground/70 hover:text-foreground transition-all cursor-pointer relative"
            >
                <svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 -960 960 960" width="24px" fill="currentColor">
                    <path d="M243.6-168q-31.1 0-53.35-22.15T168-243.4q0-31.1 22.15-53.35T243.4-319q31.1 0 53.35 22.15T319-243.6q0 31.1-22.15 53.35T243.6-168Zm236.5 0q-31.1 0-53.35-22.15T404.5-243.4q0-31.1 22.15-53.35T479.9-319q31.1 0 53.35 22.15t22.25 53.25q0 31.1-22.15 53.35T480.1-168Zm236.5 0q-31.1 0-53.35-22.15T641-243.4q0-31.1 22.15-53.35T716.4-319q31.1 0 53.35 22.15T792-243.6q0 31.1-22.15 53.35T716.6-168Zm-473-236.5q-31.1 0-53.35-22.15T168-479.9q0-31.1 22.15-53.35t53.25-22.25q31.1 0 53.35 22.15T319-480.1q0 31.1-22.15 53.35T243.6-404.5Zm236.5 0q-31.1 0-53.35-22.15T404.5-479.9q0-31.1 22.15-53.35t53.25-22.25q31.1 0 53.35 22.15t22.25 53.25q0 31.1-22.15 53.35T480.1-404.5Zm236.5 0q-31.1 0-53.35-22.15T641-479.9q0-31.1 22.15-53.35t53.25-22.25q31.1 0 53.35 22.15T792-480.1q0 31.1-22.15 53.35T716.6-404.5ZM243.6-641q-31.1 0-53.35-22.15T168-716.4q0-31.1 22.15-53.35T243.4-792q31.1 0 53.35 22.15T319-716.6q0 31.1-22.15 53.35T243.6-641Zm236.5 0q-31.1 0-53.35-22.15T404.5-716.4q0-31.1 22.15-53.35T479.9-792q31.1 0 53.35 22.15t22.25 53.25q0 31.1-22.15 53.35T480.1-641Zm236.5 0q-31.1 0-53.35-22.15T641-716.4q0-31.1 22.15-53.35T716.4-792q31.1 0 53.35 22.15T792-716.6q0 31.1-22.15 53.35T716.6-641Z" />
                </svg>
            </div>

            {/* Menu Portal */}
            {isOpen && (
                <Portal>
                    <div
                        ref={menuRef}
                        className="divclosebtn absolute right-[10px] lg:right-[20px] top-[80px] z-50 transition-all animate-in fade-in slide-in-from-top-2 duration-200"
                        style={{
                            border: '1px solid var(--border)',
                            background: 'var(--background)',
                            borderRadius: '30px',
                            boxShadow: '0 8px 32px rgba(0, 0, 0, 0.15)',
                            minWidth: isLoading ? '200px' : 'auto',
                            minHeight: isLoading ? '150px' : 'auto',
                            display: 'flex',
                            alignItems: 'center',
                            justifyContent: 'center'
                        }}
                    >
                        {isLoading ? (
                            <div className="flex flex-col items-center gap-3 p-6">
                                <Loader2 className="animate-spin text-primary" size={32} />
                                <span className="text-xs font-medium text-muted-foreground uppercase tracking-widest">Loading...</span>
                            </div>
                        ) : (
                            <div 
                                className="w-full h-full"
                                dangerouslySetInnerHTML={{ __html: htmlContent }} 
                            />
                        )}
                    </div>
                </Portal>
            )}
        </>
    );
}