"use client"
import React, { useEffect, useState } from 'react';
import Link from 'next/link';
import { Search, ImageIcon, Video, Settings, Moon, Sun, Plus, User, LogOut, ChevronDown } from "lucide-react"
import { Button } from "@/components/ui/button"
import { useRouter } from "next/navigation"
import { Logo, LogoWithText, LogoTM } from "@/components/logo"
import { SearchWithAutocomplete } from "@/components/search-with-autocomplete"
import { useTheme } from "@/components/theme-provider"
import { useAuth } from "@/components/auth-provider"
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog"
import QuickLinks from "@/components/ui/QuickLinks"
import { useSearchHistory } from "@/hooks/useSearchHistory"
import { MinimalHeader } from "@/components/ui/MinimalHeader"
import NineDotMenu from "./NineDotMenu"
import ActionBar from './ActionBar'


interface SiteSettings {
  siteName: string
  siteTitle: string
  siteDescription: string
  welcomeMessage: string
  popularSearches: string[]
  showSubmitWebsite: boolean
  logo: string
  favicon: string
  themeColor: string
}


const Header = () => {

  const [query, setQuery] = useState("")
  const [isAccountModalOpen, setIsAccountModalOpen] = useState(false)
  const [siteSettings, setSiteSettings] = useState<SiteSettings>({
    siteName: "Indzs Search",
    siteTitle: "Indzs Search - Fast, Comprehensive Search Engine",
    siteDescription: "Search the web with IndzsSearch",
    welcomeMessage: "Fast, comprehensive search across the web",
    popularSearches: ["Technology", "News", "Weather", "Sports", "Entertainment"],
    showSubmitWebsite: true,
    logo: "/logo.png",
    favicon: "/favicon.ico",
    // themeColor: "#10b981",
    themeColor: '#df7016'
  })
  const router = useRouter()
  const { theme, setTheme, setThemeColor } = useTheme()
  const { user, logout } = useAuth()

  const [mounted, setMounted] = useState(false)
  const { history, addToHistory, deleteFromHistory } = useSearchHistory();
  const [isMenuOpen, setIsMenuOpen] = useState(false);
  useEffect(() => {
    setMounted(true)
    fetchSiteSettings()

    // const interval = setInterval(fetchSiteSettings, 30000)
    // return () => clearInterval(interval)
  }, [])

  const fetchSiteSettings = async () => {
    try {
      console.log("[v0] Fetching site settings...")
      const response = await fetch("/api/admin/site-settings", {
        cache: "no-store",
        headers: {
          "Cache-Control": "no-cache",
        },
      })
      if (response.ok) {
        const settings = await response.json()
        console.log("[v0] Received site settings:", settings)
        setSiteSettings(settings)

        if (settings.themeColor && settings.themeColor !== siteSettings.themeColor) {
          setThemeColor(settings.themeColor)
        }
      } else {
        console.error("[v0] Failed to fetch site settings:", response.status, response.statusText)
      }
    } catch (error) {
      console.error("[v0] Failed to fetch site settings:", error)
    }
  }

  const handleSearch = (searchQuery: string) => {
    if (searchQuery.trim()) {
      addToHistory(searchQuery);
      router.push(`/search?q=${encodeURIComponent(searchQuery.trim())}&type=web`)
    }
  }

  const toggleTheme = () => {
    if (theme === "light") {
      setTheme("dark")
    } else if (theme === "dark") {
      setTheme("system")
    } else {
      setTheme("light")
    }
  }

  const getThemeIcon = () => {
    if (!mounted) {
      return <Moon className="w-4 h-4" />
    }
    if (theme === "light") return <Moon className="w-4 h-4" />
    if (theme === "dark") return <Sun className="w-4 h-4" />
    return <Moon className="w-4 h-4" />
  }

  const handleAccountClick = () => {
    setIsAccountModalOpen(false)
    router.push("/account")
  }

  const handleLogout = () => {
    setIsAccountModalOpen(false)
    logout()
  }

  return <div>

    <header className="sticky top-0 flex items-center justify-between px-3 py-4 bg-background/60 backdrop-blur-md z-10 lg:px-6">
      <div className="flex items-center cursor-pointer hover:opacity-70 transition-opacity">
        <Logo size="md" siteSettings={siteSettings} />
      </div>

      {/* Navigation Pill */}
      <nav className="flex items-center gap-1 p-1.5 rounded-full border border-border/40 bg-secondary/20 shadow-sm">

        {/* Theme Toggle */}
        <button
          onClick={toggleTheme}
          className="p-2 rounded-full hover:bg-accent text-foreground/70 hover:text-foreground transition-all"
        >
          {/* {true ? <Sun size={18} /> : <Moon size={18} />} */}
          {getThemeIcon()}
        </button>

        {/* Action: Add (Conditional) */}
        {/* {user && siteSettings.showSubmitWebsite && (
          <button
            // onClick={() => router.push("/submit-website")}
            onClick={() => router.push("/settings?tab=Submit%20Website")}


            className="p-2 rounded-full hover:bg-accent text-foreground/70 hover:text-foreground transition-all"
          >
            <Plus size={18} />
          </button>
        )} */}

        {/* Divider */}
        {/* <div className="w-[1px] h-4 bg-border/60 mx-1" /> */}

        {/* User Profile / Auth */}
        <div className="relative">
          {user ? (
            <>
              <button
                onClick={() => setIsMenuOpen(!isMenuOpen)}
                className="flex items-center gap-2 px-2 py-1.5 rounded-full hover:bg-accent transition-all"
              >
                <div className="w-6 h-6 rounded-full bg-primary flex items-center justify-center">
                  <User size={14} className="text-primary-foreground" />
                </div>
                <span className="text-sm font-medium hidden sm:block">Namaste , {user.username.toUpperCase().charAt(0)} </span>
                {/* <ChevronDown size={14} className={`transition-transform ${isMenuOpen ? 'rotate-180' : ''}`} /> */}
              </button>

              {/* Simple Dropdown Menu */}
              {isMenuOpen && false && (
                <div className="absolute right-0 mt-3 w-48 overflow-hidden rounded-2xl border border-border bg-popover shadow-xl animate-in fade-in zoom-in duration-150">
                  <div className="p-3 border-b border-border/50">
                    <p className="text-[10px] uppercase tracking-widest text-muted-foreground font-semibold">Account</p>
                    <p className="text-sm truncate font-medium">{user.username}</p>
                  </div>
                  <div className="p-1">
                    <button className="flex w-full items-center gap-2 px-3 py-2 text-sm rounded-lg hover:bg-accent transition-colors"


                      onClick={() => router.push("/settings")


                      }
                    >
                      <Settings size={14} /> Settings
                    </button>
                    <button className="flex w-full items-center gap-2 px-3 py-2 text-sm rounded-lg hover:bg-destructive/10 text-destructive transition-colors"
                      style={{ color: 'red' }}
                      onClick={logout}
                    >

                      <LogOut size={14} /> Logout
                    </button>
                  </div>
                </div>
              )}
            </>
          ) : (
              <button
                onClick={() => router.push("/settings")}
                className="flex items-center gap-2 px-4 py-1.5 text-sm font-medium rounded-full bg-primary text-primary-foreground hover:opacity-90 transition-opacity"
              >
                Sign In
              </button>
            )}
        </div>


        <NineDotMenu />

        <button
          onClick={() => router.push("/settings")}
          className="p-2 rounded-full hover:bg-accent text-foreground/70 hover:text-foreground transition-all"
        >
          <Settings size={18} />
        </button>
      </nav>
    </header>
  </div>
}

export default Header