'use client';

import React, { useEffect, useState, useRef } from 'react';
import { getEngine } from '@/lib/engines';
import { ImageModal } from "@/components/image-modal"
import { useQuery } from "@tanstack/react-query";
import { usePreservedResults } from '@/hooks/usePreservedResults'
interface ImageSearchResultsProps {
  query: string;
}

function extractKeywords(title: string): string[] {
  return title
    .toLowerCase()
    .split(/\s+/)
    .filter(w => w.length > 3)
    .slice(0, 5);
}


function getGoogleImageResults() {
  const wrapper = document.querySelector('.google-search-wrapper');
  if (!wrapper) return [];

  const results = Array.from(
    wrapper.querySelectorAll('.gsc-imageResult')
  );

  return results.map(result => {
    // Image element
    const imgEl = result.querySelector('img.gs-image');

    // Thumbnail (what you see)
    const thumbnail =
      imgEl?.getAttribute('data-src') ||
      imgEl?.getAttribute('src') ||
      '';

    // Full image URL (sometimes stored here)
    const fullImage =
      imgEl?.getAttribute('data-iurl') ||
      imgEl?.getAttribute('data-src') ||
      thumbnail;

    // Title / alt text
    const title =
      imgEl?.getAttribute('alt') ||
      result.querySelector('.gs-title')?.textContent?.trim() ||
      '';

    // Page link where image belongs
    const pageLink =
      result.querySelector('a')?.href || '';

    console.log(pageLink)
    // Domain / source
    const source =
      result.querySelector('.gs-visibleUrl')?.textContent?.trim() ||
      pageLink ||
      '';

    // Image size (if Google shows it)
    const size =
      result.querySelector('.gs-image-size')?.textContent?.trim() ||
      '';

    let favicon;
    try {
      favicon = `https://www.google.com/s2/favicons?domain=${new URL(source).hostname}&sz=32` || ''
    } catch (e) {
      favicon = ''
    }

    const keywords = extractKeywords(title)
    return {
      title,
      imgurl: thumbnail,
      image: fullImage,
      // url: pageLink,
      url: source,
      size,
      favicon,
      keywords


    };
  });
}



function injectCSEInlineStyles(styles = {
  backgroundColor: 'transparent',
  padding: '0 !important',
  margin: '0',
  border: 'none',
  boxShadow: 'none',
  width: '100%',

}) {
  const selector = '.gsc-control-cse, .gsc-control-cse-en';


  const applyStyles = () => {
    const el = document.querySelector(selector);
    if (!el) return false;

    Object.assign(el.style, styles);
    return true;
  };
  applyStyles()

  document.querySelectorAll('.gsc-control-cse.gsc-control-cse-en').forEach(el=>{
    el.style.background = 'transparent',
    el.style.border = 'none'
  })
  document.querySelectorAll('.gsc-webResult .gsc-result').forEach(el => el.style.border = 'none')


  document.querySelectorAll('.gsc-cursor-box').forEach(el => el.style.width = '100%')

  const nextBtn = document.querySelectorAll('.gsc-cursor-container-next')

  nextBtn.forEach(el => {

    el.innerHTML = "Load More"
    Object.assign(
      el.style, {
      justifyContent: 'center',
      background: 'var(--secondary)',
      borderRadius: '50px'
    })
  })

  document.querySelectorAll('.gsc-cursor-container-previous, .gsc-cursor-next-page').forEach(el => el.style.display = 'none')
  document.querySelectorAll('.gsc-cursor-numbered-page').forEach(el => el.style.display = 'none')


  const currentPages = document.querySelectorAll(
    '.gsc-cursor-page.gsc-cursor-current-page'
  );

  if (!currentPages) {
    console.log('Current page not found');
  } else {

    currentPages.forEach(currentPage => {
      const nextPage = currentPage.nextElementSibling;

      if (nextPage?.classList.contains('gsc-cursor-page')) {
        nextPage.innerText = "Load More"
        Object.assign(
          nextPage.style, {
          justifyContent: 'center',
          background: 'var(--secondary)',
          borderRadius: '50px',
          display: 'block',
          color: 'white',
          padding: '10px 20px'
        })
        nextPage.style.setProperty('display', 'block', 'important');
        nextPage.classList.add('force-block');

        const cursorBox = document.querySelector('.gsc-cursor-box');
        // if (!cursorBox) return;

      } else {
        console.log('No next page available');
      }


    })


  }

  const alltPages = document.querySelectorAll(
    '.gsc-cursor-page'
  ).forEach(el => {
    if (el.classList.contains('force-block')) {

    } else {

      el.style.display = "none"
    }
  });


}


// export interface GoogleImageResult {
//   /** Image title or alt text */
//   title: string;

//   /** Thumbnail URL (what is displayed in grid) */
//   thumbnail: string;

//   /** Full resolution image URL (if available) */
//   image: string;

//   /** Webpage URL where the image is hosted */
//   pageLink: string;

//   /** Source domain or visible URL */
//   source: string;

//   /** Image dimensions like "1920 × 1080" (optional) */
//   size?: string;
// }

function getGooglePaginationElement(): HTMLElement | null {
  // return document.querySelector('.gsc-cursor-box') as HTMLElement | null;

  const nodes = document.querySelectorAll('.gsc-cursor-box');
  console.log(nodes)
  return nodes.length
    ? (nodes[nodes.length - 1] as HTMLElement)
    : null;

}


export interface ImageResult {
  id: string;
  title: string;

  /** Image thumbnail / display image */
  imgurl: string;

  /** Page URL where the image comes from */
  url: string;

  /** Source favicon */
  favicon?: string;

  /** Optional keywords */
  keywords?: string[];
}

export interface ImageSeedItem {
  title: string
  imgurl?: string
  image?: string
  url: string
  keywords?: string[]
  favicon?: string
}

export interface ImageSeedResponse {
  success: boolean
  received?: number
  uniqueImages?: number
  imported?: number
  failed?: number
  error?: string
}


export async function seedImagesBulk(
  images: ImageSeedItem[],
  signal?: AbortSignal,
): Promise<ImageSeedResponse> {
  try {
    const res = await fetch("/api/images-bulk-submit", {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify(images),
      signal,
    })

    if (!res.ok) {
      const text = await res.text()
      throw new Error(text || "Bulk image seed failed")
    }

    return await res.json()
  } catch (error) {
    if (error instanceof DOMException && error.name === "AbortError") {
      throw new Error("Request aborted")
    }

    throw error
  }
}



const ImageSearchResults: React.FC<ImageSearchResultsProps> = ({ query }) => {
  const [loading, setLoading] = useState(true);
  const engine = getEngine('image');
  const [showGoogle, setShowGoogle] = useState(true)
  const [imageResults, setImageResults] = useState<ImageResult[]>([]);
  const [pageUpdate, setPageUpdate] = useState(1)
  const paginationContainerRef = useRef<HTMLDivElement | null>(null);
  const [selectedImage, setSelectedImage] = useState<ImageResult | null>(null)
  const [isModalOpen, setIsModalOpen] = useState(false)
  const { setPreservedData } = usePreservedResults(`image:${query}`)



  const handleImageClick = (image: ImageResult) => {
    console.log("image clicked")
    setSelectedImage(image)
    setIsModalOpen(true)
  }
  const handleCloseModal = () => {
    setIsModalOpen(false)
    setSelectedImage(null)
  }


  /* 🔹 Load once from sessionStorage */
  // useEffect(() => {
  //   if (typeof window === "undefined") return;

  //   // const savedText = sessionStorage.getItem("textResults");
  //   const savedImages = sessionStorage.getItem("imageResults");
  //   console.log(JSON.parse(savedImages))
  //   // if (savedText) setTextResults(JSON.parse(savedText));
  //   if (savedImages) setImageResults([...JSON.parse(savedImages)]);
  // }, []);


  useEffect(() => {
    console.log("inside result change")
    const wrapper = document.querySelector('.google-search-wrapper');
    if (!wrapper) {
      setLoading(false)
      // setShowGoogle(true)
      return;
    }

    let hasRun = false;

    const observer = new MutationObserver(async () => {
      if (hasRun) return;
      // if(data) return
      console.log(imageResults)
      // queryKey: ["image-search", query],
      console.log("Fetching google results")
      setLoading(true)
      // await delay(3000);
      const results = getGoogleImageResults();
      injectCSEInlineStyles()
      if (results.length > 0) {
        console.log(results)
        // const pagination = getGooglePaginationElement();
        // console.log(pagination)
        setShowGoogle(false)

        setLoading(false)
        setImageResults([...imageResults, ...results])
        await seedImagesBulk(results)
        // setPreservedData([...imageResults, ...results])
        // setGoogleResults(results);

        // setResultNumber(document.getElementsByClassName('gsc-result-info')[0].innerHTML || '')
        hasRun = true;
        observer.disconnect();
      }
      setLoading(false)
    });

    // setLoading(false)
    observer.observe(wrapper, { childList: true, subtree: true });

    return () => observer.disconnect();
  }, [pageUpdate]);

  useEffect(() => {
    const wrapper = document.querySelector('.google-search-wrapper');
    if (!wrapper) return;

    const observer = new MutationObserver(() => {
      const pagination = getGooglePaginationElement();
      const container = paginationContainerRef.current;
      console.log(pagination)
      setPageUpdate(pageUpdate + 1)
      if (
        pagination &&
        container &&
        !container.contains(pagination)
      ) {

        container.innerHTML = '';

        container.appendChild(pagination);


        observer.disconnect();
      }
    });

    observer.observe(wrapper, {
      childList: true,
      subtree: true,
    });

    return () => observer.disconnect();
  }, [pageUpdate]);

  useEffect(() => {
    if (!query) return;
    // if(data){
    //   setImageResults(data);
    //   setLoading(false)
    //   const old = document.getElementById('google-cse-image');
    //   document.body.appendChild(old);
    //   return
    // }
    console.log("image fetching")
    setLoading(true);

    const old = document.getElementById('google-cse-image');
    if (old) old.remove();

    const script = document.createElement('script');
    script.id = 'google-cse-image';
    script.async = true;
    script.src = `https://cse.google.com/cse.js?cx=${engine.cx}`;
    script.onload = () => setLoading(false);

    document.body.appendChild(script);
  }, [engine.cx, query]);

  useEffect(() => {
    if (!loading) {
      // window.scrollTo({ top: 0, behavior: 'smooth' });
    }
  }, [imageResults]);



  // useEffect(() => {
  //   sessionStorage.setItem("imageResults", JSON.stringify(imageResults));
  // }, [imageResults]);


  return (
    <div>
      {loading &&

        <div className="flex items-center justify-center py-12">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-accent"></div>
        </div>


      }
      <>

        <div className="text-sm text-muted-foreground mb-6">
          {/* About {totalResults.toLocaleString()} results */}
        </div>
        <ImageModal image={selectedImage} isOpen={isModalOpen} onClose={handleCloseModal} />
        <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 xl:grid-cols-6 gap-4">
          {imageResults.map((image, index) => (
            // if (!image.title) return null;

            (image.title && <div
              key={image.id || index}
              className="group cursor-pointer"
              onClick={() => handleImageClick(image)}
            >
              <div className="aspect-square bg-muted rounded-lg overflow-hidden mb-2 hover:shadow-md transition-shadow">
                <img
                  src={image.imgurl || "/placeholder.svg"}
                  alt={image.title}
                  className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-200"
                  onError={(e) => {
                    e.currentTarget.src =
                      "/placeholder.svg?height=200&width=200&text=" +
                      encodeURIComponent(image.title?.substring(0, 20) || "Image");
                  }}
                />
              </div>

              <h3 className="text-sm font-medium line-clamp-2 mb-1 group-hover:text-blue-600 transition-colors">
                {image.title}
              </h3>

              <div className="flex items-center gap-1 mb-2">
                <img
                  src={image.favicon}
                  alt=""
                  className="w-3 h-3 flex-shrink-0"
                  onError={(e) => {
                    e.currentTarget.src = "/placeholder.svg?height=12&width=12";
                  }}
                />
                <span className="text-xs text-muted-foreground truncate">
                  {image.url ? (image.url) : ""}
                </span>
              </div>

              {image.keywords?.length > 0 && (
                <div className="flex flex-wrap gap-1">
                  {image.keywords.slice(0, 3).map((keyword, i) => (
                    <span
                      key={i}
                      className="text-xs text-blue-600 hover:underline cursor-pointer"
                      style={{
                        wordBreak: 'break-all'
                      }}
                    >
                      {keyword}
                    </span>
                  ))}
                </div>
              )}
            </div>)
          ))}

        </div>
        <div
          ref={paginationContainerRef}
          className="mt-8 flex justify-center lg:justify-start"
        ></div>
      </>


      <div className="google-search-wrapper"
        style={{
          backgroundColor: "transparent",
          padding: 0,
          margin: 0,
          display: showGoogle ? 'block' : 'none',
          opacity: showGoogle ? 1 : 0
        }}
      >
        <div
          className="gcse-searchresults-only"
          data-enableimagesearch="true"
          data-defaulttoimagesearch="true"


        />
      </div>
    </div>
  );
};

export default React.memo(ImageSearchResults, (prev, next) => {
  return prev.value === next.value;
}

);
