"use client"

import type React from "react"
import { createContext, useContext, useEffect, useState } from "react"

type Theme = "dark" | "light" | "system"

type ThemeProviderProps = {
  children: React.ReactNode
  defaultTheme?: Theme
  storageKey?: string
}

type ThemeProviderState = {
  theme: Theme
  setTheme: (theme: Theme) => void
  themeColor: string
  setThemeColor: (color: string) => void
}

const initialState: ThemeProviderState = {
  theme: "system",
  setTheme: () => null,
  themeColor: "#df7016",
  setThemeColor: () => null,
}

const ThemeProviderContext = createContext<ThemeProviderState>(initialState)

export function ThemeProvider({
  children,
  defaultTheme = "system",
  storageKey = "indus-search-theme",
  ...props
}: ThemeProviderProps) {
  const [theme, setThemeState] = useState<Theme>(defaultTheme)
  const [themeColor, setThemeColorState] = useState<string>("#df7016")

  useEffect(() => {
    // Get theme from cookie on mount
    if (typeof document !== "undefined") {
      const cookies = document.cookie.split(";")
      const themeCookie = cookies.find((cookie) => cookie.trim().startsWith(`${storageKey}=`))

      if (themeCookie) {
        const savedTheme = themeCookie.split("=")[1] as Theme
        setThemeState(savedTheme)
      }
    }
  }, [storageKey])

  useEffect(() => {
    const fetchThemeColor = async () => {
      try {
        const response = await fetch("/api/site-settings")
        if (response.ok) {
          const settings = await response.json()
          if (settings.themeColor) {
            setThemeColorState(settings.themeColor)
          }
        }
      } catch (error) {
        console.error("Failed to fetch theme color:", error)
      }
    }

    fetchThemeColor()
  }, [])

  useEffect(() => {
    if (typeof window === "undefined") return

    const root = window.document.documentElement

    root.classList.remove("light", "dark")

    if (theme === "system") {
      const systemTheme = window.matchMedia("(prefers-color-scheme: dark)").matches ? "dark" : "light"
      root.classList.add(systemTheme)
      return
    }

    root.classList.add(theme)
  }, [theme])

    useEffect(() => {
    // 1. Get the existing object first to avoid overwriting other keys (like language/region)
    const savedSettings = localStorage.getItem("indus-search-settings");
    const currentSettings = savedSettings ? JSON.parse(savedSettings) : {};

    // 2. Merge the new state values into the object
    const updatedSettings = {
      ...currentSettings,
      theme: theme.toLowerCase(), // Ensuring it matches your "system" lowercase requirement
      // autoComplete: autoComplete
    };

    // 3. Save back to localStorage
    localStorage.setItem("indus-search-settings", JSON.stringify(updatedSettings));
  }, [theme]); // This effect runs whenever these two change



  useEffect(() => {
    if (typeof window === "undefined") return

    const root = window.document.documentElement

    // Convert hex to RGB for alpha variants
    const hexToRgb = (hex: string) => {
      const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex)
      return result
        ? {
            r: Number.parseInt(result[1], 16),
            g: Number.parseInt(result[2], 16),
            b: Number.parseInt(result[3], 16),
          }
        : null
    }

    const rgb = hexToRgb(themeColor)
    if (rgb) {
      // Update CSS custom properties with the new theme color
      root.style.setProperty("--primary", themeColor)
      root.style.setProperty("--secondary", themeColor)
      root.style.setProperty("--accent", themeColor)
      root.style.setProperty("--ring", `rgba(${rgb.r}, ${rgb.g}, ${rgb.b}, 0.5)`)
      root.style.setProperty("--chart-1", themeColor)
      root.style.setProperty("--sidebar-accent", themeColor)
      root.style.setProperty("--sidebar-ring", `rgba(${rgb.r}, ${rgb.g}, ${rgb.b}, 0.5)`)
    }
  }, [themeColor])

  const setTheme = (newTheme: Theme) => {
    console.log("[v0] ThemeProvider: Setting theme to", newTheme)

    // Save to cookie with 1 year expiration
    if (typeof document !== "undefined") {
      const expires = new Date()
      expires.setFullYear(expires.getFullYear() + 1)
      document.cookie = `${storageKey}=${newTheme}; expires=${expires.toUTCString()}; path=/; SameSite=Lax`
    }

    // Update state
    setThemeState(newTheme)
  }

  const setThemeColor = (newColor: string) => {
    console.log("[v0] ThemeProvider: Setting theme color to", newColor)
    setThemeColorState(newColor)
  }

  const value = {
    theme,
    setTheme,
    themeColor,
    setThemeColor,
  }

  return (
    <ThemeProviderContext.Provider {...props} value={value}>
      {children}
    </ThemeProviderContext.Provider>
  )
}

export const useTheme = () => {
  const context = useContext(ThemeProviderContext)

  if (context === undefined) throw new Error("useTheme must be used within a ThemeProvider")

  return context
}
