"use client"

import { createContext, useContext, useState, useEffect, type ReactNode } from "react"

interface SettingsContextType {
  theme: "light" | "dark" | "system"
  resultsPerPage: number
  safeSearch: boolean
  autoComplete: boolean
  language: string
  region: string
  setTheme: (theme: "light" | "dark" | "system") => void
  setResultsPerPage: (count: number) => void
  setSafeSearch: (enabled: boolean) => void
  setAutoComplete: (enabled: boolean) => void
  setLanguage: (language: string) => void
  setRegion: (region: string) => void
  resetSettings: () => void
}

const SettingsContext = createContext<SettingsContextType | undefined>(undefined)

export function useSettings() {
  const context = useContext(SettingsContext)
  if (context === undefined) {
    throw new Error("useSettings must be used within a SettingsProvider")
  }
  return context
}

interface SettingsProviderProps {
  children: ReactNode
}

const defaultSettings = {
  theme: "system" as const,
  resultsPerPage: 10,
  safeSearch: true,
  autoComplete: true,
  language: "en",
  region: "global",
}

export function SettingsProvider({ children }: SettingsProviderProps) {
  const [theme, setThemeState] = useState<"light" | "dark" | "system">(defaultSettings.theme)
  const [resultsPerPage, setResultsPerPageState] = useState(defaultSettings.resultsPerPage)
  const [safeSearch, setSafeSearchState] = useState(defaultSettings.safeSearch)
  const [autoComplete, setAutoCompleteState] = useState(defaultSettings.autoComplete)
  const [language, setLanguageState] = useState(defaultSettings.language)
  const [region, setRegionState] = useState(defaultSettings.region)
  const [mounted, setMounted] = useState(false)

  useEffect(() => {
    setMounted(true)

    if (typeof window !== "undefined") {
      const savedSettings = localStorage.getItem("indus-search-settings")

      if (savedSettings) {
        try {
          const parsed = JSON.parse(savedSettings)
          setThemeState(parsed.theme || defaultSettings.theme)
          setResultsPerPageState(parsed.maxResults || defaultSettings.resultsPerPage)
          setSafeSearchState(parsed.safeSearch ?? defaultSettings.safeSearch)
          setAutoCompleteState(parsed.autoComplete ?? defaultSettings.autoComplete)
          setLanguageState(parsed.language || defaultSettings.language)
          setRegionState(parsed.region || defaultSettings.region)
        } catch (error) {
          console.error("Failed to parse saved settings:", error)
        }
      } else {
        const legacyTheme = localStorage.getItem("indus-theme") as "light" | "dark" | "system" | null
        const legacyResultsPerPage = localStorage.getItem("indus-results-per-page")

        if (legacyTheme) {
          setThemeState(legacyTheme)
        }
        if (legacyResultsPerPage) {
          setResultsPerPageState(Number.parseInt(legacyResultsPerPage, 10))
        }
      }
    }
  }, [])

  useEffect(() => {
    if (!mounted) return

    if (typeof document !== "undefined") {
      const root = document.documentElement

      if (theme === "system") {
        const systemTheme = window.matchMedia("(prefers-color-scheme: dark)").matches ? "dark" : "light"
        root.classList.toggle("dark", systemTheme === "dark")
      } else {
        root.classList.toggle("dark", theme === "dark")
      }
    }
  }, [theme, mounted])

  useEffect(() => {
    if (!mounted) return

    const settings = {
      theme,
      maxResults: resultsPerPage,
      safeSearch,
      autoComplete,
      language,
      region,
    }

    if (typeof window !== "undefined") {
      localStorage.setItem("indus-search-settings", JSON.stringify(settings))
    }
  }, [theme, resultsPerPage, safeSearch, autoComplete, language, region, mounted])

  const setTheme = (newTheme: "light" | "dark" | "system") => {
    setThemeState(newTheme)
  }

  const setResultsPerPage = (count: number) => {
    setResultsPerPageState(count)
  }

  const setSafeSearch = (enabled: boolean) => {
    setSafeSearchState(enabled)
  }

  const setAutoComplete = (enabled: boolean) => {
    setAutoCompleteState(enabled)
  }

  const setLanguage = (newLanguage: string) => {
    setLanguageState(newLanguage)
  }

  const setRegion = (newRegion: string) => {
    setRegionState(newRegion)
  }

  const resetSettings = () => {
    setThemeState(defaultSettings.theme)
    setResultsPerPageState(defaultSettings.resultsPerPage)
    setSafeSearchState(defaultSettings.safeSearch)
    setAutoCompleteState(defaultSettings.autoComplete)
    setLanguageState(defaultSettings.language)
    setRegionState(defaultSettings.region)

    if (typeof window !== "undefined") {
      localStorage.removeItem("indus-search-settings")
      localStorage.removeItem("indus-theme")
      localStorage.removeItem("indus-results-per-page")
    }
  }

  return (
    <SettingsContext.Provider
      value={{
        theme,
        resultsPerPage,
        safeSearch,
        autoComplete,
        language,
        region,
        setTheme,
        setResultsPerPage,
        setSafeSearch,
        setAutoComplete,
        setLanguage,
        setRegion,
        resetSettings,
      }}
    >
      {children}
    </SettingsContext.Provider>
  )
}
