"use client"

import { useState } from "react"
import { Settings, Palette, Hash, Check } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Label } from "@/components/ui/label"
import { useSettings } from "@/components/settings-provider"
import { useTheme } from "@/components/theme-provider"

export function SettingsModal() {
  const {  resultsPerPage,  setResultsPerPage } = useSettings()
  const { theme, setTheme } = useTheme()
  // console.log(theme)
  const [isOpen, setIsOpen] = useState(false)

  const handleThemeChange = (value: string) => {
    setTheme(value as "light" | "dark" | "system")
  }

  const handleResultsPerPageChange = (value: string) => {
    setResultsPerPage(Number.parseInt(value, 10))
  }

  return (
    <Dialog open={isOpen} onOpenChange={setIsOpen}>
      <DialogTrigger asChild>
        <Button 
          variant="ghost" 
          size="sm" 
          className="rounded-full flex items-center gap-2 hover:bg-secondary/60 text-muted-foreground hover:text-foreground transition-all px-2 md:px-3"
        >
          <Settings className="w-4 h-4 shrink-0" />
          <span className="hidden md:inline text-xs font-medium">Settings</span>
        </Button>
      </DialogTrigger>
      
      <DialogContent className="sm:max-w-[400px] bg-card border-border rounded-3xl p-0 overflow-hidden shadow-2xl">
        {/* Aesthetic Header */}
        <div className="px-6 pt-6 pb-4  border-b border-border/50">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2 text-lg font-semibold tracking-tight">
              <div className="p-1.5 rounded-lg bg-primary text-primary-foreground">
                <Settings className="w-4 h-4" />
              </div>
              Preferences
            </DialogTitle>
          </DialogHeader>
        </div>

        <div className="p-6 space-y-8">
          {/* Theme Setting */}
          <div className="space-y-3">
            <div className="flex items-center justify-between">
              <Label className="flex items-center gap-2 text-xs font-bold uppercase tracking-widest text-muted-foreground">
                <Palette className="w-3.5 h-3.5" />
                Appearance
              </Label>
              <span className="text-[10px] bg-muted px-2 py-0.5 rounded-full text-muted-foreground uppercase font-bold">Visual</span>
            </div>
            
            <Select value={theme} onValueChange={handleThemeChange}>
              <SelectTrigger className="w-full h-11 rounded-xl border-border bg-secondary/30 hover:bg-secondary/50 focus:ring-1 focus:ring-ring transition-all border-none shadow-none">
                <SelectValue placeholder="Select theme" />
              </SelectTrigger>
              <SelectContent className="bg-popover border-border rounded-xl">
                <SelectItem value="light" className="rounded-lg">Light</SelectItem>
                <SelectItem value="dark" className="rounded-lg">Dark</SelectItem>
                <SelectItem value="system" className="rounded-lg">System</SelectItem>
              </SelectContent>
            </Select>
            <p className="text-[11px] text-muted-foreground leading-relaxed px-1">
              Switch between light and dark modes to suit your environment.
            </p>
          </div>

          {/* Results Per Page Setting */}
          <div className="space-y-3">
            <div className="flex items-center justify-between">
              <Label className="flex items-center gap-2 text-xs font-bold uppercase tracking-widest text-muted-foreground">
                <Hash className="w-3.5 h-3.5" />
                Density
              </Label>
              <span className="text-[10px] bg-muted px-2 py-0.5 rounded-full text-muted-foreground uppercase font-bold">Data</span>
            </div>

            <Select value={resultsPerPage.toString()} onValueChange={handleResultsPerPageChange}>
              <SelectTrigger className="w-full h-11 rounded-xl border-border bg-secondary/30 hover:bg-secondary/50 focus:ring-1 focus:ring-ring transition-all border-none shadow-none">
                <SelectValue placeholder="Select results per page" />
              </SelectTrigger>
              <SelectContent className="bg-popover border-border rounded-xl">
                <SelectItem value="10" className="rounded-lg">10 results / page</SelectItem>
                <SelectItem value="25" className="rounded-lg">25 results / page</SelectItem>
                <SelectItem value="50" className="rounded-lg">50 results / page</SelectItem>
                <SelectItem value="100" className="rounded-lg">100 results / page</SelectItem>
              </SelectContent>
            </Select>
            <p className="text-[11px] text-muted-foreground leading-relaxed px-1">
              Adjust how much information is displayed in search results.
            </p>
          </div>
        </div>

        {/* Footer Action */}
        <div className="px-6 py-4  border-t border-border/50 flex justify-end">
          <Button 
            onClick={() => setIsOpen(false)} 
            className="rounded-full px-6 bg-primary text-primary-foreground hover:opacity-90 transition-all font-semibold text-xs h-9"
          >
            Save Changes
          </Button>
        </div>
      </DialogContent>
    </Dialog>
  )
}