"use client";
//componets/search-with-automcplete.tsx
import type React from "react";
import { useState, useEffect, useRef, useCallback, useMemo } from "react";
import { Search, Clock, TrendingUp, RotateCcw, Trash2 } from "lucide-react";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import {
  Command,
  CommandEmpty,
  CommandGroup,
  CommandItem,
  CommandList,
} from "@/components/ui/command";
import { cn } from "@/lib/utils";
import VoiceSearchBox from "@/components/VoiceSearchBox";
import { useHistory } from "@/hooks/useHistory";
import { useDebounce } from "@/hooks/useDebounce.ts";
interface SearchWithAutocompleteProps {
  value: string;
  onChange: (value: string) => void;
  onSubmit: (query: string) => void;
  placeholder?: string;
  size?: "sm" | "lg";
  className?: string;
  showButton?: boolean;
  buttonText?: string;
}

interface AutocompleteSuggestion {
  text: string;
  type: "suggestion" | "recent" | "trending";
}

export function SearchWithAutocomplete({
  value,
  onChange,
  onSubmit,
  placeholder = "Search...",
  size = "lg",
  className,
  showButton = true,
  buttonText = "Search",
}: SearchWithAutocompleteProps) {
  const [suggestions, setSuggestions] = useState<AutocompleteSuggestion[]>([]);
  const [showSuggestions, setShowSuggestions] = useState(false);
  const [selectedIndex, setSelectedIndex] = useState(-1);
  const [loading, setLoading] = useState(false);
  const inputRef = useRef<HTMLInputElement>(null);
  const containerRef = useRef<HTMLDivElement>(null);
  const abortControllerRef = useRef<AbortController>();
  const {
    getSearchHistory,
    updateSearchState,
    removeSearchHistory,
    clearSearchHistory,
    getLastSearch,
  } = useHistory();
  const [lastSearches, setLastSearches] = useState(getSearchHistory());
  // console.log(lastSearches.slice(0,5));
  const lastFiveSearches = lastSearches.slice(0,5);
  const scrollContainerRef = useRef<HTMLDivElement>(null);
  // const [filteredHistory, setFilteredHistory] = useState(lastSearches)

  const debouncedQuery = useDebounce(value, 300);
  const filteredHistory = useMemo(() => {
    // setSuggestions( [...lastFiveSearches,...suggestions] )
    if (!debouncedQuery.trim()) return lastFiveSearches;

    return lastSearches
      .filter((item) =>
        item.query.toLowerCase().includes(debouncedQuery.toLowerCase())
      )
      .slice(-5)
      .reverse();
  }, [debouncedQuery, lastSearches]);

  const deleteItem = (id: string) => {
    removeSearchHistory(id);
    setLastSearches((prev) => prev.filter((item) => item.id !== id));
  };

  const clearSearches = () => {
    clearSearchHistory();
    setLastSearches([]);
  }

  const fetchSuggestions = useCallback(async (query: string) => {
    if (query.trim().length < 1) {
      setSuggestions([]);
      return;
    }

    if (abortControllerRef.current) {
      try {
        abortControllerRef.current.abort();
      } catch (e) { }
    }
    abortControllerRef.current = new AbortController();

    setLoading(true);
    try {
      const response = await fetch(
        `/api/autocomplete?q=${encodeURIComponent(query.trim())}`,
        {
          signal: abortControllerRef.current.signal,
        }
      );

      const data = await response.json();

      const formattedSuggestions: AutocompleteSuggestion[] = data.suggestions.map(
        (text: string) => ({
          text,
          type: "suggestion" as const,
        })
      );

      setSuggestions(formattedSuggestions);
    } catch (error) {
      if (error instanceof Error && error.name !== "AbortError") {
        console.error("Failed to fetch suggestions:", error);
        setSuggestions([]);
      }
    } finally {
      setLoading(false);
    }
  }, []);

  useEffect(() => {
    if (value && showSuggestions) {
      fetchSuggestions(value);
    } else if (!value) {
      setSuggestions([]);
    }
  }, [value, showSuggestions, fetchSuggestions]);

  const handleKeyDown = (e: React.KeyboardEvent) => {
    if (!showSuggestions || mergedSuggestions.length === 0) {
      if (e.key === "Enter") {
        handleSubmit();
      }
      return;
    }

    switch (e.key) {
      case "ArrowDown":
        e.preventDefault();
        // setSelectedIndex((prev) => (prev < suggestions.length - 1 ? prev + 1 : prev))
        setSelectedIndex((prev) =>
          prev < mergedSuggestions.length - 1 ? prev + 1 : prev
        );
        break;

      case "ArrowUp":
        e.preventDefault();
        setSelectedIndex((prev) => (prev > 0 ? prev - 1 : -1));

        break;
      case "Enter":
        e.preventDefault();
        if (selectedIndex >= 0 && selectedIndex < mergedSuggestions.length) {
          const selectedSuggestion = mergedSuggestions[selectedIndex].text;
          onChange(selectedSuggestion);
          // if (inputRef.current) {
          //   inputRef.current.value = selectedSuggestion;
          // }
          onSubmit(selectedSuggestion);
        } else {
          handleSubmit();
        }
        setShowSuggestions(false);
        break;
      case "Escape":
        setShowSuggestions(false);
        setSelectedIndex(-1);
        inputRef.current?.blur();
        break;
    }
  };

  const handleSubmit = () => {
    if (value.trim()) {
      onSubmit(value.trim());
      setShowSuggestions(false);
    }
  };

  const handleFocus = () => {
    setShowSuggestions(true);
    if (value.trim().length >= 1) {
      fetchSuggestions(value);
    }
  };

  const handleBlur = (e: React.FocusEvent) => {
    setTimeout(() => {
      if (!containerRef.current?.contains(document.activeElement)) {
        setShowSuggestions(false);
        setSelectedIndex(-1);
      }
    }, 150);
  };

  const handleSuggestionSelect = (suggestion: string) => {
    console.log(suggestion);
    onChange(suggestion);
    onSubmit(suggestion);
    setShowSuggestions(false);
    setSelectedIndex(-1);
  };

  const getSuggestionIcon = (type: string) => {
    switch (type) {
      case "recent":
        return <Clock className="w-4 h-4 text-foreground" />;
      case "trending":
        return <TrendingUp className="w-4 h-4 text-foreground" />;
      default:
        return <Search className="w-4 h-4 text-foreground" />;
    }
  };

  const inputHeight = size === "lg" ? "h-14" : "h-10";
  const inputPadding = size === "lg" ? "pl-12 pr-4" : "pl-10 pr-4";
  const iconSize = size === "lg" ? "w-5 h-5" : "w-4 h-4";
  const iconPosition = size === "lg" ? "left-4" : "left-3";
  const textSize = size === "lg" ? "text-lg" : "text-sm";

  const recentSuggestions: AutocompleteSuggestion[] = filteredHistory.map(
    (item) => ({
      text: item.query,
      type: "recent",
    })
  );

  const mergedSuggestions = useMemo<AutocompleteSuggestion[]>(() => {
    const map = new Map<string, AutocompleteSuggestion>();

    // recent first (higher priority)
    recentSuggestions.forEach((item) => {
      map.set(item.text.toLowerCase(), item);
    });

    // api suggestions after
    suggestions.forEach((item) => {
      if (!map.has(item.text.toLowerCase())) {
        map.set(item.text.toLowerCase(), item);
      }
    });

    return Array.from(map.values()).slice(0, 10);
  }, [recentSuggestions, suggestions]);

  // const scrollContainerRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    if (selectedIndex === -1 || !scrollContainerRef.current) return;

    const container = scrollContainerRef.current;
    // We use querySelectorAll to find the actual list items
    const items = container.querySelectorAll(".suggestion-item");
    const selectedElement = items[selectedIndex] as HTMLElement;

    if (selectedElement) {
      const containerTop = container.scrollTop;
      const containerBottom = containerTop + container.clientHeight;
      const elementTop = selectedElement.offsetTop;
      const elementBottom = elementTop + selectedElement.offsetHeight;

      // Logic: If the bottom of the element is below the visible area
      if (elementBottom > containerBottom) {
        container.scrollTo({
          top: elementBottom - container.clientHeight,
          behavior: "smooth",
        });
      }
      // Logic: If the top of the element is above the visible area
      else if (elementTop < containerTop) {
        container.scrollTo({
          top: elementTop,
          behavior: "smooth",
        });
      }
    }
  }, [selectedIndex]);

  return (
    <div ref={containerRef} className={cn("relative w-full", className)}>
      <div className="relative">
        <Input
          ref={inputRef}
          type="text"
          placeholder={placeholder}
          value={value}
          onChange={(e) => onChange(e.target.value)}
          onKeyDown={handleKeyDown}
          onFocus={handleFocus}
          onBlur={handleBlur}
          className={cn(
            inputHeight,
            inputPadding,
            textSize,
            "rounded-full border-2 border-border focus:border-accent focus:ring-2 focus:ring-ring bg-input text-black dark:text-white !placeholder:text-black dark:!placeholder:text-white",
            showButton && size === "lg" && "pr-24",
            showButton && size === "sm" && "pr-20"
          )}
        />
        <Search
          className={cn(
            "absolute top-1/2 transform -translate-y-1/2 text-foreground",
            iconPosition,
            iconSize
          )}
        />

        <div
          className={cn(
            "flex justify-center align-middle absolute right-2 top-1/2 transform -translate-y-1/2 gap-1"
            // size === "lg" ? "h-10 px-6" : "h-6 px-4 text-xs",
          )}
        >
          <VoiceSearchBox onChange={onChange} />

          {showButton && (
            <Button
              type="button"
              onClick={handleSubmit}
              className={cn(
                "  rounded-full bg-accent hover:bg-accent/90 home-search-button",
                size === "lg" ? "h-10 px-6" : "h-6 px-4 text-xs"
              )}
            >
              {buttonText}
            </Button>
          )}
        </div>
      </div>

      {/* {filteredHistory && showSuggestions && (
        <div className="rounded-lg border border-border shadow-lg bg-popover relative z-[99999]">
          {filteredHistory.map((item, index) => (
          <div
            key={`${item.id}`}
            className="history-item-anim group flex items-center justify-between p-4 rounded-2xl border border-transparent hover:border-white/5 active:scale-[0.98] cursor-pointer"
          // style={{ backgroundColor: 'rgba(255,255,255,0.02)' }}
          style={{
            padding:'12px 16px'
          }}
          >
            <div className="flex items-center gap-4 min-w-0">
              <RotateCcw size={16} className="opacity-100 group-hover:rotate-[-45deg] transition-transform" />
              <span className="text-sm md:text-base font-medium truncate opacity-80 group-hover:opacity-100 transition-opacity">
                {item.query}
              </span>
            </div>
            <button
              onClick={(e) => { e.stopPropagation(); deleteItem(item.id); }}
              className="p-2 rounded-xl opacity-100 group-hover:opacity-100 hover:bg-white/5 transition-all"
              style={{ color: 'var(--destructive)',
                    padding: '0 8px'

              }}
            >
              <Trash2 size={16} />
            </button>
          </div>
        ))}
        </div>
      )} */}

      {/* {showSuggestions && (suggestions.length > 0 || loading) && (
        <div className="absolute top-full left-0 right-0 z-[99999] mt-1">
          <Command className="rounded-lg border border-border shadow-lg bg-popover relative z-[99999]">
            <CommandList className="max-h-[300px]">
              {loading ? (
                <div className="flex items-center justify-center py-4">
                  <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-accent"></div>
                </div>
              ) : suggestions.length === 0 ? (
                <CommandEmpty>No suggestions found.</CommandEmpty>
              ) : (
                    <CommandGroup>
                      {suggestions.map((suggestion, index) => (
                        <CommandItem
                          key={`${suggestion.text}-${index}`}
                          value={suggestion.text}
                          onSelect={() => handleSuggestionSelect(suggestion.text)}
                          className={cn(
                            "flex items-center gap-3 px-4 py-3 cursor-pointer",
                            index === selectedIndex && "bg-accent text-accent-foreground",
                          )}
                        >
                          {getSuggestionIcon(suggestion.type)}
                          <span className="flex-1 text-left">{suggestion.text}</span>
                        </CommandItem>
                      ))}
                    </CommandGroup>
                  )}
            </CommandList>
          </Command>
        </div>
      )}   */}

      {showSuggestions && mergedSuggestions.length > 0 && (
        <div className="absolute top-full left-0 right-0 mt-2 w-full overflow-hidden rounded-xl border border-border bg-popover/80 backdrop-blur-xl shadow-lg animate-in fade-in zoom-in-95 duration-200 z-50">
          <Command className="rounded-lg border border-border shadow-lg relative">
            <div
              ref={scrollContainerRef}
              className="max-h-[320px] overflow-y-auto p-1.5 custom-scrollbar"
            >
              {loading ? (
                // <div className="flex items-center justify-center py-4">
                //   <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-accent"></div>
                // </div>
                <div className="flex items-center justify-center py-6">
                  <div className="h-5 w-5 animate-spin rounded-full border-2 border-white/20 border-t-white"></div>
                </div>
              ) : mergedSuggestions.length === 0 ? (
                <CommandEmpty>No suggestions found.</CommandEmpty>
              ) : (
                    <div>
                      {!loading && mergedSuggestions.length > 0 && (
                        <div className="px-3 py-2 text-[11px] font-bold uppercase tracking-widest text-white/40">
                          Suggestions
                        </div>
                      )}
                      {mergedSuggestions.map((suggestion, index) => (
                        <CommandItem
                          key={`${suggestion.text}-${index}`}
                          value={suggestion.text}
                          onSelect={() => handleSuggestionSelect(suggestion.text)}
                          // className={cn(
                          //   "flex items-center gap-3 px-4 py-3 cursor-pointer",
                          //   index === selectedIndex && "bg-accent text-accent-foreground",
                          // )}

                          className={cn(
                            "suggestion-item group flex items-center justify-between rounded-lg px-3 py-2.5 transition-all duration-200 cursor-pointer",
                            index === selectedIndex
                              ? "bg-white/10 text-white translate-x-1"
                              : "text-white/70 hover:bg-white/5 hover:text-white"
                          )}
                        >
                          {getSuggestionIcon(suggestion.type)}
                          <span className="flex-1 text-left text-foreground">
                            {suggestion.text}
                          </span>
                          {suggestion.type === "recent" && (
                            <button
                              onClick={(e) => {
                                e.stopPropagation();
                                const historyItem = lastSearches.find(
                                  (h) => h.query === suggestion.text
                                );
                                if (historyItem) deleteItem(historyItem.id);
                              }}
                              className="opacity-100 group-hover:opacity-100 p-1.5 hover:bg-red-500/20 hover:text-red-400 rounded-md transition-all"
                            >
                              <Trash2 className="h-3.5 w-3.5" />
                            </button>
                          )}
                        </CommandItem>
                      ))}
                    </div>
                  )}
              {lastSearches.length > 0 && (
                <div className="mt-2 border-t border-foreground/5 pt-1.5 text-foreground">
                  <button
                    onClick={clearSearches}
                    className="flex w-full items-center justify-center gap-2 rounded-lg py-2 text-xs font-medium  transition-colors"
                  >
                    <RotateCcw className="h-3 w-3" />
                    Clear search history
                  </button>
                </div>
              )}
            </div>
          </Command>
        </div>
      )}

      {/* {showSuggestions && (
        <div className="absolute top-full left-0 right-0 z-50 mt-2 w-full overflow-hidden rounded-xl border border-white/10 bg-black/80 backdrop-blur-xl shadow-[0_8px_32px_rgba(0,0,0,0.5)] animate-in fade-in zoom-in-95 duration-200">
          <div 
          ref={scrollContainerRef}
          className="max-h-[320px] overflow-y-auto p-1.5 custom-scrollbar">

  
            {loading && (
              <div className="flex items-center justify-center py-6">
                <div className="h-5 w-5 animate-spin rounded-full border-2 border-white/20 border-t-white"></div>
              </div>
            )}

 
            {!loading && mergedSuggestions.length > 0 && (
              <div className="px-3 py-2 text-[11px] font-bold uppercase tracking-widest text-white/40">
                Suggestions
              </div>
            )}

   
            <div className="space-y-1">
              {mergedSuggestions.map((item, index) => (
                <div
                  key={`${item.type}-${item.text}`}
                  onClick={() => handleSuggestionSelect(item.text)}
                  className={cn(
                    "suggestion-item group flex items-center justify-between rounded-lg px-3 py-2.5 transition-all duration-200 cursor-pointer",
                    index === selectedIndex
                      ? "bg-white/10 text-white translate-x-1"
                      : "text-white/70 hover:bg-white/5 hover:text-white"
                  )}
                >
                  <div className="flex items-center gap-3">
                    <div className={cn(
                      "flex h-7 w-7 items-center justify-center rounded-md border border-white/10 bg-white/5 shadow-inner",
                      index === selectedIndex && "border-white/20 bg-white/10"
                    )}>
                      {item.type === "recent" ? (
                        <Clock className="h-3.5 w-3.5" />
                      ) : (
                          <TrendingUp className="h-3.5 w-3.5" />
                        )}
                    </div>
                    <span className="text-sm font-medium">{item.text}</span>
                  </div>

    
                  {item.type === "recent" && (
                    <button
                      onClick={(e) => {
                        e.stopPropagation();
                        const historyItem = lastSearches.find(h => h.query === item.text);
                        if (historyItem) deleteItem(historyItem.id);
                      }}
                      className="opacity-100 group-hover:opacity-100 p-1.5 hover:bg-red-500/20 hover:text-red-400 rounded-md transition-all"
                    >
                      <Trash2 className="h-3.5 w-3.5" />
                    </button>
                  )}
                </div>
              ))}
            </div>

      
            {lastSearches.length > 0 && (
              <div className="mt-2 border-t border-white/5 pt-1.5">
                <button
                  onClick={clearSearchHistory}
                  className="flex w-full items-center justify-center gap-2 rounded-lg py-2 text-xs font-medium text-white/40 hover:bg-white/5 hover:text-white transition-colors"
                >
                  <RotateCcw className="h-3 w-3" />
            Clear search history
          </button>
              </div>
            )}
          </div>
        </div>
      )} */}

      {/* {showSuggestions && mergedSuggestions.length > 0 && (
        <div 
        className="absolute top-full left-0 right-0 z-50 mt-2 w-full overflow-hidden rounded-xl border border-border bg-popover/80 backdrop-blur-xl shadow-lg animate-in fade-in zoom-in-95 duration-200">
          <div
            // ref={scrollContainerRef}
            className="max-h-[320px] overflow-y-auto p-1.5 custom-scrollbar"
          >

   
            {loading && (
              <div className="flex items-center justify-center py-6">
                <div className="h-5 w-5 animate-spin rounded-full border-2 border-muted border-t-foreground"></div>
              </div>
            )}


            {!loading && mergedSuggestions.length > 0 && (
              <div className="px-3 py-2 text-[11px] font-bold uppercase tracking-widest text-muted-foreground/70">
                Suggestions
              </div>
            )}

            <div className="space-y-1">
              {mergedSuggestions.map((item, index) => (
                <div
                  key={`${item.type}-${item.text}`}
                  onClick={() => handleSuggestionSelect(item.text)}
                  className={cn(
                    "suggestion-item group flex items-center justify-between rounded-lg px-3 py-2.5 transition-all duration-200 cursor-pointer",
                    index === selectedIndex
                      ? "bg-accent text-accent-foreground translate-x-1"
                      : "text-muted-foreground hover:bg-accent/50 hover:text-foreground"
                  )}
                >
                  <div className="flex items-center gap-3">
                    <div className={cn(
                      "flex h-7 w-7 items-center justify-center rounded-md border border-border bg-muted/30 shadow-inner",
                      index === selectedIndex && "border-primary/20 bg-accent"
                    )}>
                      {item.type === "recent" ? (
                        <Clock className="h-3.5 w-3.5" />
                      ) : (
                          <TrendingUp className="h-3.5 w-3.5" />
                        )}
                    </div>
                    <span className="text-sm font-medium">{item.text}</span>
                  </div>

   
                  {item.type === "recent" && (
                    <button
                      onClick={(e) => {
                        e.stopPropagation();
                        const historyItem = lastSearches.find(h => h.query === item.text);
                        if (historyItem) deleteItem(historyItem.id);
                      }}
                      className="opacity-0 group-hover:opacity-100 p-1.5 hover:bg-destructive/20 hover:text-destructive rounded-md transition-all"
                    >
                      <Trash2 className="h-3.5 w-3.5" />
                    </button>
                  )}
                </div>
              ))}
            </div>


            {lastSearches.length > 0 && (
              <div className="mt-2 border-t border-border pt-1.5">
                <button
                  onClick={clearSearchHistory}
                  className="flex w-full items-center justify-center gap-2 rounded-lg py-2 text-xs font-medium text-muted-foreground hover:bg-muted hover:text-foreground transition-colors"
                >
                  <RotateCcw className="h-3 w-3" />
            Clear search history
          </button>
              </div>
            )}
          </div>
        </div>
      )} */}
    </div>
  );
}
