"use client"

import { useState, useEffect } from "react"
import Image from "next/image"
import { useRouter } from "next/navigation"

interface LogoProps {
  size?: "sm" | "md" | "lg" | "xl" | "xl2" | "xl3"
  className?: string
  siteSettings?: {
    siteName?: string
    logo?: string
  }
}

export function Logo({ size = "md", className = "", siteSettings }: LogoProps) {
  const [logoSrc, setLogoSrc] = useState<string | null>('/logo.png')
  const [siteName, setSiteName] = useState("IndusSearch")
  const router = useRouter()
  useEffect(() => {
    if (siteSettings?.logo && siteSettings.logo !== "/logo.png") {
      setLogoSrc(siteSettings.logo)
      setLogoSrc('/logo.png')
    }
    if (siteSettings?.siteName) {
      setSiteName(siteSettings.siteName)
    }
  }, [siteSettings])

  const sizeClasses = {
    sm: "w-6 h-6",
    md: "w-8 h-8",
    lg: "w-12 h-12",
    xl :"w-16 h-16"
  }

  if (logoSrc) {
    return (
      <div className={`${sizeClasses[size]} relative ${className}`}
        onClick={() => { router.push('/') }}
      >
        <Image
          src={logoSrc || "/placeholder.svg"}
          alt={`${siteName} Logo`}
          fill
          className="object-contain rounded-lg"
          sizes={size === "sm" ? "24px" : size === "md" ? "32px" : size ==="lg" ? "48px" : "60px"}
        />
      </div>
    )
  }

  return (
    <div
      className={`${sizeClasses[size]} bg-gradient-to-br from-accent to-secondary rounded-lg flex items-center justify-center shadow-sm ${className}`}
    >
      <svg
        className={`${size === "sm" ? "w-3 h-3" : size === "md" ? "w-5 h-5" : "w-7 h-7"} text-white`}
        fill="none"
        stroke="currentColor"
        viewBox="0 0 24 24"
      >
        <circle cx="11" cy="11" r="8" />
        <path d="21 21l-4.35-4.35" />
      </svg>
    </div>
  )
}

export function LogoWithText({ size = "md", className = "", siteSettings }: LogoProps) {
  const [siteName, setSiteName] = useState("IndusSearch")

  useEffect(() => {
    if (siteSettings?.siteName) {
      setSiteName(siteSettings.siteName)
    }
  }, [siteSettings])

  const textSizes = {
    sm: "text-lg",
    md: "text-xl",
    lg: "text-2xl",
  }

  const getDisplayName = () => {
    if (siteName.length <= 8) {
      return { main: siteName, sub: "" }
    }

    // Try to split at common patterns
    const patterns = ["Search", "Engine", "Web", "Net", "Tech"]
    for (const pattern of patterns) {
      if (siteName.includes(pattern)) {
        const parts = siteName.split(pattern)
        if (parts.length === 2 && parts[0].trim() && parts[1].trim() === "") {
          return { main: parts[0].trim(), sub: pattern }
        }
      }
    }

    // Default split for IndusSearch-like names
    if (siteName.toLowerCase().includes("search")) {
      const main = siteName.replace(/search/i, "").trim()
      return { main: main || siteName, sub: "Search" }
    }

    // Fallback: use full name
    return { main: siteName, sub: "" }
  }

  const { main, sub } = getDisplayName()

  return (
    <div className={`flex items-center gap-2 ${className}`}>
      <Logo size={size} siteSettings={siteSettings} />
      <div className="flex flex-col">
        <span className={`${textSizes[size]} font-bold text-foreground leading-none`}>{main}</span>
        {sub && (
          <span className={`${size === "sm" ? "text-xs" : "text-sm"} font-medium text-muted-foreground leading-none`}>
            {sub}
          </span>
        )}
      </div>
    </div>
  )
}


export function LogoTM({ size = "md", className = "", siteSettings }: LogoProps) {
  const [logoSrc, setLogoSrc] = useState<string | null>('/logo-tm.png')
  const [siteName, setSiteName] = useState("IndusSearch")
  const router = useRouter()
  useEffect(() => {
    if (siteSettings?.logo && siteSettings.logo !== "/logo-tm.png") {
      setLogoSrc(siteSettings.logo)
      setLogoSrc('/logo-tm.png')
    }
    if (siteSettings?.siteName) {
      setSiteName(siteSettings.siteName)
    }
  }, [siteSettings])

  const sizeClasses = {
    sm: "w-6 h-3",
    md: "w-12 h-6",
    lg: "w-24 h-12",
    xl :"w-32 h-16",
    xl2 : "w-48 h-24",
  }

  if (logoSrc) {
    return (
      <div className={`${sizeClasses[size]} relative ${className}`}
        onClick={() => { router.push('/') }}
      >
        <Image
          src={logoSrc || "/placeholder.svg"}
          alt={`${siteName} Logo`}
          fill
          className="object-contain rounded-lg"
          sizes={size === "sm" ? "24px" : size === "md" ? "32px" : "48px"}
        />
      </div>
    )
  }

  return (
    <div
      className={`${sizeClasses[size]} bg-gradient-to-br from-accent to-secondary rounded-lg flex items-center justify-center shadow-sm ${className}`}
    >
      <svg
        className={`${size === "sm" ? "w-3 h-3" : size === "md" ? "w-5 h-5" : "w-7 h-7"} text-white`}
        fill="none"
        stroke="currentColor"
        viewBox="0 0 24 24"
      >
        <circle cx="11" cy="11" r="8" />
        <path d="21 21l-4.35-4.35" />
      </svg>
    </div>
  )
}