"use client"
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { ExternalLink, Calendar } from "lucide-react"

interface ImageResult {
  id: string
  title: string
  url: string
  imgurl: string
  keywords?: string[]
  favicon?: string
  date?: string
  safesearch: boolean
}

interface ImageModalProps {
  image: ImageResult | null
  isOpen: boolean
  onClose: () => void
}

export function ImageModal({ image, isOpen, onClose }: ImageModalProps) {
  if (!image) return null

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-7xl max-h-[90vh] overflow-hidden p-0">
        <div className="flex flex-col h-full">
          {/* Header */}
          <DialogHeader className="p-6 pb-4 border-b">
            <div className="flex items-start justify-between">
              <div className="flex-1 pr-4">
                <DialogTitle className="text-lg font-semibold line-clamp-2 mb-2">
                  {image.title}
                </DialogTitle>
                <div className="flex items-center gap-2 text-sm text-muted-foreground">
                  {image.favicon && (
                    <img
                      src={image.favicon || "/placeholder.svg"}
                      alt=""
                      className="w-4 h-4"
                      onError={(e) => {
                        e.currentTarget.src = `https://www.google.com/s2/favicons?domain=${new URL(image.url).hostname}&sz=16`
                      }}
                    />
                  )}
                  {/* <span>{new URL(image.url).hostname}</span> */}
                  {image.date && (
                    <>
                      <span>•</span>
                      <div className="flex items-center gap-1">
                        <Calendar className="w-3 h-3" />
                        <span>{new Date(image.date).toLocaleDateString()}</span>
                      </div>
                    </>
                  )}
                </div>
              </div>
            </div>
          </DialogHeader>

          {/* Image */}
          <div className="flex-1 flex items-center justify-center p-6 bg-muted/20">
            <img
              src={image.imgurl || "/placeholder.svg"}
              alt={image.title}
              className="max-w-full max-h-[60vh] object-contain rounded-lg shadow-lg"
              onError={(e) => {
                e.currentTarget.src = "/broken-image.png"
              }}
            />
          </div>

          {/* Footer */}
          <div className="p-6 pt-4 border-t bg-card">
          <span className="text-xs text-blue-600 hover:underline cursor-pointer __web-inspector-hide-shortcut__" style={{wordBreak: 'break-all'}}>{image.url}</span>
            <div className="flex justify-between mt-2">
              <div className="flex-1">
                {image.keywords && image.keywords.length > 0 && (
                  <div className="flex flex-wrap gap-2 mb-3">
                    {image.keywords.map((keyword) => (
                      <Badge key={keyword} variant="secondary" className="text-xs">
                        {keyword}
                      </Badge>
                    ))}
                  </div>
                )} 
                
              </div>
              <Button asChild>
                <a
                  href={`https://${image.url.replace(/^https:\/\/https:\/\//, 'https://')}`}
                  target="_blank"
                  rel="noopener noreferrer"
                  className="flex items-center gap-2"
                >
                  <ExternalLink className="w-4 h-4" />
                  Visit Website
                </a>
              </Button>
            </div>

            {/* === Copyright Disclaimer === */}
            <div className="mt-4 text-xs text-muted-foreground text-center border-t pt-3">
              © All images are copyright of their respective owners.  
              Please seek permission from the original author or rights holder before reuse.
            </div>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  )
}
