"use client"

import { createContext, useContext, useState, useEffect, type ReactNode } from "react"
import type { User } from "@/lib/typesense"

interface AuthContextType {
  user: User | null
  login: (
    username: string,
    password: string,
  ) => Promise<{ success: boolean; error?: string; requiresVerification?: boolean }>
  logout: () => void
  register: (
    username: string,
    email: string,
    password: string,
  ) => Promise<{ success: boolean; error?: string; requiresVerification?: boolean }>
  loading: boolean // renamed from isLoading to match usage in ads center
}

const AuthContext = createContext<AuthContextType | undefined>(undefined)

export function AuthProvider({ children }: { children: ReactNode }) {
  const [user, setUser] = useState<User | null>()
  const [loading, setLoading] = useState(true) // renamed from isLoading

  useEffect(() => {
    const checkSession = async () => {
      try {
        const cookies = document.cookie.split(";")
        const sessionCookie = cookies.find((cookie) => cookie.trim().startsWith("indus-session="))
        console.log(sessionCookie)
        if (sessionCookie) {
          try {
            const sessionData = sessionCookie.split("=")[1]
            const userData = JSON.parse(decodeURIComponent(sessionData))

            if (userData && userData.id) {
              const response = await fetch(`/api/auth/validate?userId=${userData.id}`)
              if (response.ok) {
                const freshUserData = await response.json()
                setUser(freshUserData.user)
                localStorage.setItem("indus-user", JSON.stringify(freshUserData.user))
              } else {
                // Invalid session, clear it
                document.cookie = "indus-session=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;"
                localStorage.removeItem("indus-user")
              }
            }
            setLoading(false)
            return
          } catch (error) {
            console.error("Error parsing session cookie:", error)
            document.cookie = "indus-session=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;"
          }
        }

        const storedUser = localStorage.getItem("indus-user")
        if (storedUser) {
          try {
            const userData = JSON.parse(storedUser)
            if (userData && userData.id) {
              const response = await fetch(`/api/auth/validate?userId=${userData.id}`)
              if (response.ok) {
                const freshUserData = await response.json()
                setUser(freshUserData.user)
                localStorage.setItem("indus-user", JSON.stringify(freshUserData.user))
              } else {
                localStorage.removeItem("indus-user")
              }
            }
          } catch (error) {
            console.error("Error parsing stored user:", error)
            localStorage.removeItem("indus-user")
          }
        }
      } catch (error) {
        console.error("Session check error:", error)
      } finally {
        setLoading(false)
      }
    }

    checkSession()
  }, [])

  const login = async (
    username: string,
    password: string,
  ): Promise<{ success: boolean; error?: string; requiresVerification?: boolean }> => {
    try {
      const response = await fetch("/api/auth/login", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ username, password }),
      })

      const data = await response.json()

      if (response.ok) {
        setUser(data.user)
        localStorage.setItem("indus-user", JSON.stringify(data.user))
        return { success: true }
      } else {
        if (data.requiresVerification) {
          return { success: false, error: data.error, requiresVerification: true }
        }
        return { success: false, error: data.error || "Login failed" }
      }
    } catch (error) {
      console.error("Login error:", error)
      return { success: false, error: "Network error occurred" }
    }
  }

  const register = async (
    username: string,
    email: string,
    password: string,
  ): Promise<{ success: boolean; error?: string; requiresVerification?: boolean }> => {
    try {
      const response = await fetch("/api/auth/register", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ username, email, password }),
      })

      const data = await response.json()

      if (response.ok) {
        if (data.requiresVerification) {
          return { success: true, requiresVerification: true }
        } else {
          setUser(data.user)
          localStorage.setItem("indus-user", JSON.stringify(data.user))
          return { success: true }
        }
      } else {
        return { success: false, error: data.error || "Registration failed" }
      }
    } catch (error) {
      console.error("Register error:", error)
      return { success: false, error: "Network error occurred" }
    }
  }

  const logout = () => {
    setUser(null)
    localStorage.removeItem("indus-user")
    document.cookie = "indus-session=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=/;"
  }

  return <AuthContext.Provider value={{ user, login, logout, register, loading }}>{children}</AuthContext.Provider>
}

export function useAuth() {
  const context = useContext(AuthContext)
  if (context === undefined) {
    throw new Error("useAuth must be used within an AuthProvider")
  }
  return context
}
