"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Switch } from "@/components/ui/switch"
import { Textarea } from "@/components/ui/textarea"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from "@/components/ui/alert-dialog"
import { Edit, Trash2, ExternalLink } from "lucide-react"
import type { WebSearchResult } from "@/lib/typesense"

interface WebResultsTableProps {
  results: WebSearchResult[]
  loading: boolean
  onResultUpdated: () => void
  onResultDeleted: () => void
}

export function WebResultsTable({ results, loading, onResultUpdated, onResultDeleted }: WebResultsTableProps) {
  const [editingResult, setEditingResult] = useState<WebSearchResult | null>(null)
  const [editForm, setEditForm] = useState<Partial<WebSearchResult>>({})
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false)
  const [updating, setUpdating] = useState(false)
  const [deleting, setDeleting] = useState<string | null>(null)

  const handleEdit = (result: WebSearchResult) => {
    setEditingResult(result)
    setEditForm({
      title: result.title,
      description: result.description,
      url: result.url,
      keywords: result.keywords,
      safesearch: result.safesearch,
      maindomain: result.maindomain,
      date: result.date,
    })
    setIsEditDialogOpen(true)
  }

  const handleUpdate = async () => {
    if (!editingResult) return

    setUpdating(true)
    try {
      const response = await fetch(`/api/admin/web-results/${editingResult.id}`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(editForm),
      })

      if (response.ok) {
        setIsEditDialogOpen(false)
        onResultUpdated()
      }
    } catch (error) {
      console.error("Failed to update result:", error)
    } finally {
      setUpdating(false)
    }
  }

  const handleDelete = async (id: string) => {
    setDeleting(id)
    try {
      const response = await fetch(`/api/admin/web-results/${id}`, {
        method: "DELETE",
      })

      if (response.ok) {
        onResultDeleted()
      }
    } catch (error) {
      console.error("Failed to delete result:", error)
    } finally {
      setDeleting(null)
    }
  }

  if (loading) {
    return (
      <div className="border rounded-lg">
        <div className="p-6 space-y-4">
          {[...Array(5)].map((_, i) => (
            <div key={i} className="flex items-center space-x-4 animate-pulse">
              <div className="h-4 bg-muted rounded w-1/4" />
              <div className="h-4 bg-muted rounded w-1/2" />
              <div className="h-4 bg-muted rounded w-1/4" />
            </div>
          ))}
        </div>
      </div>
    )
  }

  return (
    <>
      <div className="border rounded-lg">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>Title</TableHead>
              <TableHead>URL</TableHead>
              <TableHead>Date</TableHead>
              <TableHead>Safe Search</TableHead>
              <TableHead>Main Domain</TableHead>
              <TableHead className="text-right">Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {results.length === 0 ? (
              <TableRow>
                <TableCell colSpan={6} className="text-center py-8 text-muted-foreground">
                  No web results found
                </TableCell>
              </TableRow>
            ) : (
              results.map((result) => (
                <TableRow key={result.id}>
                  <TableCell className="max-w-xs">
                    <div className="flex items-center space-x-2">
                      {result.favicon && (
                        <img src={result.favicon || "/placeholder.svg"} alt="" className="h-4 w-4 flex-shrink-0" />
                      )}
                      <span className="truncate font-medium">{result.title}</span>
                    </div>
                    <p className="text-sm text-muted-foreground truncate mt-1">{result.description}</p>
                  </TableCell>
                  <TableCell>
                    <a
                      href={result.url}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="flex items-center space-x-1 text-primary hover:underline"
                    >
                      <span className="truncate max-w-xs">{result.url}</span>
                      <ExternalLink className="h-3 w-3 flex-shrink-0" />
                    </a>
                  </TableCell>
                  <TableCell>{result.date}</TableCell>
                  <TableCell>
                    <Badge variant={result.safesearch ? "default" : "secondary"}>
                      {result.safesearch ? "On" : "Off"}
                    </Badge>
                  </TableCell>
                  <TableCell>
                    <Badge variant={result.maindomain ? "default" : "outline"}>
                      {result.maindomain ? "Yes" : "No"}
                    </Badge>
                  </TableCell>
                  <TableCell className="text-right">
                    <div className="flex items-center justify-end space-x-2">
                      <Button variant="outline" size="sm" onClick={() => handleEdit(result)}>
                        <Edit className="h-4 w-4" />
                      </Button>
                      <AlertDialog>
                        <AlertDialogTrigger asChild>
                          <Button variant="outline" size="sm" disabled={deleting === result.id}>
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </AlertDialogTrigger>
                        <AlertDialogContent>
                          <AlertDialogHeader>
                            <AlertDialogTitle>Delete Web Result</AlertDialogTitle>
                            <AlertDialogDescription>
                              Are you sure you want to delete "{result.title}"? This action cannot be undone.
                            </AlertDialogDescription>
                          </AlertDialogHeader>
                          <AlertDialogFooter>
                            <AlertDialogCancel>Cancel</AlertDialogCancel>
                            <AlertDialogAction onClick={() => handleDelete(result.id)}>Delete</AlertDialogAction>
                          </AlertDialogFooter>
                        </AlertDialogContent>
                      </AlertDialog>
                    </div>
                  </TableCell>
                </TableRow>
              ))
            )}
          </TableBody>
        </Table>
      </div>

      <Dialog open={isEditDialogOpen} onOpenChange={setIsEditDialogOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Edit Web Result</DialogTitle>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="title">Title</Label>
              <Input
                id="title"
                value={editForm.title || ""}
                onChange={(e) => setEditForm({ ...editForm, title: e.target.value })}
              />
            </div>
            <div>
              <Label htmlFor="description">Description</Label>
              <Textarea
                id="description"
                value={editForm.description || ""}
                onChange={(e) => setEditForm({ ...editForm, description: e.target.value })}
                rows={3}
              />
            </div>
            <div>
              <Label htmlFor="url">URL</Label>
              <Input
                id="url"
                type="url"
                value={editForm.url || ""}
                onChange={(e) => setEditForm({ ...editForm, url: e.target.value })}
              />
            </div>
            <div>
              <Label htmlFor="keywords">Keywords (comma-separated)</Label>
              <Input
                id="keywords"
                value={editForm.keywords?.join(", ") || ""}
                onChange={(e) => setEditForm({ ...editForm, keywords: e.target.value.split(",").map((k) => k.trim()) })}
              />
            </div>
            <div className="flex items-center space-x-4">
              <div className="flex items-center space-x-2">
                <Switch
                  id="safesearch"
                  checked={editForm.safesearch || false}
                  onCheckedChange={(checked) => setEditForm({ ...editForm, safesearch: checked })}
                />
                <Label htmlFor="safesearch">Safe Search</Label>
              </div>
              <div className="flex items-center space-x-2">
                <Switch
                  id="maindomain"
                  checked={editForm.maindomain || false}
                  onCheckedChange={(checked) => setEditForm({ ...editForm, maindomain: checked })}
                />
                <Label htmlFor="maindomain">Main Domain</Label>
              </div>
            </div>
            <div className="flex justify-end space-x-2">
              <Button variant="outline" onClick={() => setIsEditDialogOpen(false)}>
                Cancel
              </Button>
              <Button onClick={handleUpdate} disabled={updating}>
                {updating ? "Updating..." : "Update"}
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>
    </>
  )
}
