"use client"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Switch } from "@/components/ui/switch"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from "@/components/ui/alert-dialog"
import { Edit, Trash2, ExternalLink } from "lucide-react"
import type { ImageSearchResult } from "@/lib/typesense"

interface ImageResultsTableProps {
  results: ImageSearchResult[]
  loading: boolean
  onResultUpdated: () => void
  onResultDeleted: () => void
}

export function ImageResultsTable({ results, loading, onResultUpdated, onResultDeleted }: ImageResultsTableProps) {
  const [editingResult, setEditingResult] = useState<ImageSearchResult | null>(null)
  const [editForm, setEditForm] = useState<Partial<ImageSearchResult>>({})
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false)
  const [updating, setUpdating] = useState(false)
  const [deleting, setDeleting] = useState<string | null>(null)

  const handleEdit = (result: ImageSearchResult) => {
    setEditingResult(result)
    setEditForm({
      title: result.title,
      url: result.url,
      imgurl: result.imgurl,
      keywords: result.keywords,
      safesearch: result.safesearch,
      date: result.date,
    })
    setIsEditDialogOpen(true)
  }

  const handleUpdate = async () => {
    if (!editingResult) return

    setUpdating(true)
    try {
      const response = await fetch(`/api/admin/image-results/${editingResult.id}`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(editForm),
      })

      if (response.ok) {
        setIsEditDialogOpen(false)
        onResultUpdated()
      }
    } catch (error) {
      console.error("Failed to update result:", error)
    } finally {
      setUpdating(false)
    }
  }

  const handleDelete = async (id: string) => {
    setDeleting(id)
    try {
      const response = await fetch(`/api/admin/image-results/${id}`, {
        method: "DELETE",
      })

      if (response.ok) {
        onResultDeleted()
      }
    } catch (error) {
      console.error("Failed to delete result:", error)
    } finally {
      setDeleting(null)
    }
  }

  if (loading) {
    return (
      <div className="border rounded-lg">
        <div className="p-6 space-y-4">
          {[...Array(5)].map((_, i) => (
            <div key={i} className="flex items-center space-x-4 animate-pulse">
              <div className="h-16 w-16 bg-muted rounded" />
              <div className="flex-1 space-y-2">
                <div className="h-4 bg-muted rounded w-3/4" />
                <div className="h-3 bg-muted rounded w-1/2" />
              </div>
            </div>
          ))}
        </div>
      </div>
    )
  }

  return (
    <>
      <div className="border rounded-lg">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>Image</TableHead>
              <TableHead>Title</TableHead>
              <TableHead>Source URL</TableHead>
              <TableHead>Date</TableHead>
              <TableHead>Safe Search</TableHead>
              <TableHead className="text-right">Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {results.length === 0 ? (
              <TableRow>
                <TableCell colSpan={6} className="text-center py-8 text-muted-foreground">
                  No image results found
                </TableCell>
              </TableRow>
            ) : (
              results.map((result) => (
                <TableRow key={result.id}>
                  <TableCell>
                    <div className="flex items-center space-x-2">
                      <img
                        src={result.imgurl || "/placeholder.svg"}
                        alt={result.title}
                        className="h-16 w-16 object-cover rounded border"
                        onError={(e) => {
                          const target = e.target as HTMLImageElement
                          target.src = "/abstract-colorful-swirls.png"
                        }}
                      />
                    </div>
                  </TableCell>
                  <TableCell className="max-w-xs">
                    <div className="flex items-center space-x-2">
                      {result.favicon && (
                        <img src={result.favicon || "/placeholder.svg"} alt="" className="h-4 w-4 flex-shrink-0" />
                      )}
                      <span className="truncate font-medium">{result.title}</span>
                    </div>
                    {result.keywords && result.keywords.length > 0 && (
                      <div className="flex flex-wrap gap-1 mt-1">
                        {result.keywords.slice(0, 3).map((keyword, index) => (
                          <Badge key={index} variant="outline" className="text-xs">
                            {keyword}
                          </Badge>
                        ))}
                        {result.keywords.length > 3 && (
                          <Badge variant="outline" className="text-xs">
                            +{result.keywords.length - 3}
                          </Badge>
                        )}
                      </div>
                    )}
                  </TableCell>
                  <TableCell>
                    <a
                      href={result.url}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="flex items-center space-x-1 text-primary hover:underline"
                    >
                      <span className="truncate max-w-xs">{result.url}</span>
                      <ExternalLink className="h-3 w-3 flex-shrink-0" />
                    </a>
                  </TableCell>
                  <TableCell>{result.date}</TableCell>
                  <TableCell>
                    <Badge variant={result.safesearch ? "default" : "secondary"}>
                      {result.safesearch ? "On" : "Off"}
                    </Badge>
                  </TableCell>
                  <TableCell className="text-right">
                    <div className="flex items-center justify-end space-x-2">
                      <Button variant="outline" size="sm" onClick={() => handleEdit(result)}>
                        <Edit className="h-4 w-4" />
                      </Button>
                      <AlertDialog>
                        <AlertDialogTrigger asChild>
                          <Button variant="outline" size="sm" disabled={deleting === result.id}>
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </AlertDialogTrigger>
                        <AlertDialogContent>
                          <AlertDialogHeader>
                            <AlertDialogTitle>Delete Image Result</AlertDialogTitle>
                            <AlertDialogDescription>
                              Are you sure you want to delete "{result.title}"? This action cannot be undone.
                            </AlertDialogDescription>
                          </AlertDialogHeader>
                          <AlertDialogFooter>
                            <AlertDialogCancel>Cancel</AlertDialogCancel>
                            <AlertDialogAction onClick={() => handleDelete(result.id)}>Delete</AlertDialogAction>
                          </AlertDialogFooter>
                        </AlertDialogContent>
                      </AlertDialog>
                    </div>
                  </TableCell>
                </TableRow>
              ))
            )}
          </TableBody>
        </Table>
      </div>

      <Dialog open={isEditDialogOpen} onOpenChange={setIsEditDialogOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Edit Image Result</DialogTitle>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label htmlFor="title">Title</Label>
              <Input
                id="title"
                value={editForm.title || ""}
                onChange={(e) => setEditForm({ ...editForm, title: e.target.value })}
              />
            </div>
            <div>
              <Label htmlFor="url">Source URL</Label>
              <Input
                id="url"
                type="url"
                value={editForm.url || ""}
                onChange={(e) => setEditForm({ ...editForm, url: e.target.value })}
              />
            </div>
            <div>
              <Label htmlFor="imgurl">Image URL</Label>
              <Input
                id="imgurl"
                type="url"
                value={editForm.imgurl || ""}
                onChange={(e) => setEditForm({ ...editForm, imgurl: e.target.value })}
              />
            </div>
            <div>
              <Label htmlFor="keywords">Keywords (comma-separated)</Label>
              <Input
                id="keywords"
                value={editForm.keywords?.join(", ") || ""}
                onChange={(e) => setEditForm({ ...editForm, keywords: e.target.value.split(",").map((k) => k.trim()) })}
              />
            </div>
            <div className="flex items-center space-x-2">
              <Switch
                id="safesearch"
                checked={editForm.safesearch || false}
                onCheckedChange={(checked) => setEditForm({ ...editForm, safesearch: checked })}
              />
              <Label htmlFor="safesearch">Safe Search</Label>
            </div>
            <div className="flex justify-end space-x-2">
              <Button variant="outline" onClick={() => setIsEditDialogOpen(false)}>
                Cancel
              </Button>
              <Button onClick={handleUpdate} disabled={updating}>
                {updating ? "Updating..." : "Update"}
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>
    </>
  )
}
