"use client"

import type React from "react"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Upload, X, ImageIcon, FileImage } from "lucide-react"
import { useToast } from "@/hooks/use-toast"

interface FileUploadProps {
  type: "favicon" | "logo"
  currentPath?: string
  onUploadSuccess: (path: string) => void
  adminId: string
}

export function FileUpload({ type, currentPath, onUploadSuccess, adminId }: FileUploadProps) {
  const [isUploading, setIsUploading] = useState(false)
  const [dragActive, setDragActive] = useState(false)
  const { toast } = useToast()

  const handleFileUpload = async (file: File) => {
    if (!file) return

    setIsUploading(true)
    try {
      const formData = new FormData()
      formData.append("file", file)
      formData.append("type", type)
      formData.append("adminId", adminId)

      const response = await fetch("/api/admin/upload", {
        method: "POST",
        body: formData,
      })

      const result = await response.json()

      if (result.success) {
        onUploadSuccess(result.path)
        toast({
          title: "Upload successful",
          description: `${type === "favicon" ? "Favicon" : "Logo"} uploaded successfully`,
        })
      } else {
        throw new Error(result.error)
      }
    } catch (error) {
      console.error("Upload error:", error)
      toast({
        title: "Upload failed",
        description: error.message || "Failed to upload file",
        variant: "destructive",
      })
    } finally {
      setIsUploading(false)
    }
  }

  const handleReset = async () => {
    setIsUploading(true)
    try {
      const response = await fetch(`/api/admin/upload?type=${type}&adminId=${adminId}`, {
        method: "DELETE",
      })

      const result = await response.json()

      if (result.success) {
        onUploadSuccess(result.path)
        toast({
          title: "Reset successful",
          description: `${type === "favicon" ? "Favicon" : "Logo"} reset to default`,
        })
      } else {
        throw new Error(result.error)
      }
    } catch (error) {
      console.error("Reset error:", error)
      toast({
        title: "Reset failed",
        description: error.message || "Failed to reset file",
        variant: "destructive",
      })
    } finally {
      setIsUploading(false)
    }
  }

  const handleDrag = (e: React.DragEvent) => {
    e.preventDefault()
    e.stopPropagation()
    if (e.type === "dragenter" || e.type === "dragover") {
      setDragActive(true)
    } else if (e.type === "dragleave") {
      setDragActive(false)
    }
  }

  const handleDrop = (e: React.DragEvent) => {
    e.preventDefault()
    e.stopPropagation()
    setDragActive(false)

    if (e.dataTransfer.files && e.dataTransfer.files[0]) {
      handleFileUpload(e.dataTransfer.files[0])
    }
  }

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      handleFileUpload(e.target.files[0])
    }
  }

  const acceptedTypes = type === "favicon" ? ".ico,.png,.jpg,.jpeg" : ".png,.jpg,.svg,.webp"

  return (
    <div className="space-y-4">
      <div className="flex items-center gap-4">
        <div className="flex-1">
          <Label htmlFor={`${type}-upload`} className="text-sm font-medium">
            {type === "favicon" ? "Favicon" : "Logo"}
          </Label>
          <p className="text-xs text-muted-foreground mt-1">
            {type === "favicon"
              ? "Upload a favicon (.ico, .png, .jpg) - Max 5MB"
              : "Upload a logo (.png, .jpg, .svg, .webp) - Max 5MB"}
          </p>
        </div>

        {currentPath && (
          <div className="flex items-center gap-2">
            <div className="w-8 h-8 border rounded flex items-center justify-center bg-muted">
              {type === "favicon" ? <FileImage className="w-4 h-4" /> : <ImageIcon className="w-4 h-4" />}
            </div>
            <Button variant="outline" size="sm" onClick={handleReset} disabled={isUploading}>
              <X className="w-3 h-3 mr-1" />
              Reset
            </Button>
          </div>
        )}
      </div>

      <div
        className={`border-2 border-dashed rounded-lg p-6 text-center transition-colors ${
          dragActive ? "border-primary bg-primary/5" : "border-muted-foreground/25 hover:border-muted-foreground/50"
        }`}
        onDragEnter={handleDrag}
        onDragLeave={handleDrag}
        onDragOver={handleDrag}
        onDrop={handleDrop}
      >
        <div className="flex flex-col items-center gap-2">
          <Upload className="w-8 h-8 text-muted-foreground" />
          <div>
            <p className="text-sm font-medium">
              Drop your {type} here, or{" "}
              <Label htmlFor={`${type}-upload`} className="text-primary cursor-pointer hover:underline">
                browse
              </Label>
            </p>
            <p className="text-xs text-muted-foreground">{acceptedTypes} up to 5MB</p>
          </div>
        </div>

        <Input
          id={`${type}-upload`}
          type="file"
          accept={acceptedTypes}
          onChange={handleFileChange}
          disabled={isUploading}
          className="hidden"
        />
      </div>

      {isUploading && (
        <div className="flex items-center justify-center py-2">
          <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-primary"></div>
          <span className="ml-2 text-sm text-muted-foreground">Uploading...</span>
        </div>
      )}
    </div>
  )
}
