"use client"

import { useState, useEffect } from "react"
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter } from "@/components/ui/dialog"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { Switch } from "@/components/ui/switch"
import { Badge } from "@/components/ui/badge"
import { X, Plus } from "lucide-react"
import { toast } from "sonner"
import type { AdResult } from "@/lib/typesense"

export function EditAdDialog({ 
  open, 
  onOpenChange, 
  ad, 
  onAdUpdated 
}: { 
  open: boolean, 
  onOpenChange: (open: boolean) => void, 
  ad: AdResult | null, 
  onAdUpdated: () => void 
}) {
  const [loading, setLoading] = useState(false)
  const [formData, setFormData] = useState({
    title: "",
    description: "",
    imageUrl: "",
    targetUrl: "",
    keywords: [] as string[],
    dailyLimit: 100,
    monthlyLimit: 1000,
    active: true,
  })
  const [newKeyword, setNewKeyword] = useState("")

  // Update form data when ad changes
  useEffect(() => {
    if (ad && open) {
      console.log("Setting form data for ad:", ad)
      setFormData({
        title: ad.title || "",
        description: ad.description || "",
        imageUrl: ad.imageUrl || "",
        targetUrl: ad.targetUrl || "",
        keywords: ad.keywords || [],
        dailyLimit: ad.dailyLimit || 100,
        monthlyLimit: ad.monthlyLimit || 1000,
        active: ad.active !== false,
      })
    }
  }, [ad, open])

  // Reset form when dialog closes
  useEffect(() => {
    if (!open) {
      setNewKeyword("")
    }
  }, [open])

  const addKeyword = () => {
    const keyword = newKeyword.trim()
    if (keyword && !formData.keywords.includes(keyword)) {
      setFormData(prev => ({
        ...prev,
        keywords: [...prev.keywords, keyword]
      }))
      setNewKeyword("")
    }
  }

  const removeKeyword = (keywordToRemove: string) => {
    setFormData(prev => ({
      ...prev,
      keywords: prev.keywords.filter(keyword => keyword !== keywordToRemove)
    }))
  }

  const handleKeywordKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter') {
      e.preventDefault()
      addKeyword()
    }
  }

  const handleSave = async () => {
    if (!ad?.id) return

    if (!formData.title.trim() || !formData.description.trim() || !formData.targetUrl.trim()) {
      toast.error("Please fill in all required fields")
      return
    }

    setLoading(true)
    try {
      const response = await fetch(`/api/admin/ads-results/${ad.id}`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          title: formData.title,
          description: formData.description,
          imageUrl: formData.imageUrl,
          targetUrl: formData.targetUrl,
          keywords: formData.keywords,
          dailyLimit: formData.dailyLimit,
          monthlyLimit: formData.monthlyLimit,
          active: formData.active,
        }),
      })

      const result = await response.json()

      if (result.success) {
        toast.success("Ad updated successfully")
        onAdUpdated()
        onOpenChange(false)
      } else {
        toast.error(result.error || "Failed to update ad")
      }
    } catch (error) {
      console.error("Error updating ad:", error)
      toast.error("Failed to update ad")
    } finally {
      setLoading(false)
    }
  }

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle>Edit Advertisement</DialogTitle>
        </DialogHeader>
        
        <div className="space-y-6">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            {/* Title */}
            <div className="md:col-span-2">
              <Label htmlFor="edit-title">
                Title <span className="text-red-500">*</span>
              </Label>
              <Input
                id="edit-title"
                value={formData.title}
                onChange={(e) => setFormData(prev => ({ ...prev, title: e.target.value }))}
                placeholder="Enter ad title"
                required
              />
            </div>

            {/* Description */}
            <div className="md:col-span-2">
              <Label htmlFor="edit-description">
                Description <span className="text-red-500">*</span>
              </Label>
              <Textarea
                id="edit-description"
                value={formData.description}
                onChange={(e) => setFormData(prev => ({ ...prev, description: e.target.value }))}
                placeholder="Enter ad description"
                rows={3}
                required
              />
            </div>

            {/* Target URL */}
            <div className="md:col-span-2">
              <Label htmlFor="edit-targetUrl">
                Target URL <span className="text-red-500">*</span>
              </Label>
              <Input
                id="edit-targetUrl"
                type="url"
                value={formData.targetUrl}
                onChange={(e) => setFormData(prev => ({ ...prev, targetUrl: e.target.value }))}
                placeholder="https://example.com"
                required
              />
            </div>

            {/* Image URL */}
            <div className="md:col-span-2">
              <Label htmlFor="edit-imageUrl">Image URL</Label>
              <Input
                id="edit-imageUrl"
                type="url"
                value={formData.imageUrl}
                onChange={(e) => setFormData(prev => ({ ...prev, imageUrl: e.target.value }))}
                placeholder="https://example.com/image.jpg"
              />
              {formData.imageUrl && (
                <div className="mt-2">
                  <img 
                    src={formData.imageUrl} 
                    alt="Preview" 
                    className="w-20 h-20 object-cover rounded border"
                    onError={(e) => {
                      e.currentTarget.style.display = 'none'
                    }}
                  />
                </div>
              )}
            </div>

            {/* Daily Limit */}
            <div>
              <Label htmlFor="edit-dailyLimit">Daily Limit</Label>
              <Input
                id="edit-dailyLimit"
                type="number"
                min="1"
                value={formData.dailyLimit}
                onChange={(e) => setFormData(prev => ({ ...prev, dailyLimit: parseInt(e.target.value) || 100 }))}
              />
            </div>

            {/* Monthly Limit */}
            <div>
              <Label htmlFor="edit-monthlyLimit">Monthly Limit</Label>
              <Input
                id="edit-monthlyLimit"
                type="number"
                min="1"
                value={formData.monthlyLimit}
                onChange={(e) => setFormData(prev => ({ ...prev, monthlyLimit: parseInt(e.target.value) || 1000 }))}
              />
            </div>

            {/* Active Status */}
            <div className="md:col-span-2 flex items-center space-x-2">
              <Switch
                id="edit-active"
                checked={formData.active}
                onCheckedChange={(checked) => setFormData(prev => ({ ...prev, active: checked }))}
              />
              <Label htmlFor="edit-active">Active</Label>
            </div>
          </div>

          {/* Keywords */}
          <div>
            <Label>Keywords</Label>
            <div className="flex gap-2 mt-2">
              <Input
                value={newKeyword}
                onChange={(e) => setNewKeyword(e.target.value)}
                onKeyPress={handleKeywordKeyPress}
                placeholder="Add keyword"
                className="flex-1"
              />
              <Button type="button" onClick={addKeyword} size="sm" variant="outline">
                <Plus className="w-4 h-4" />
              </Button>
            </div>
            <div className="flex flex-wrap gap-2 mt-2">
              {formData.keywords.map((keyword) => (
                <Badge key={keyword} variant="secondary" className="flex items-center gap-1">
                  {keyword}
                  <Button
                    type="button"
                    variant="ghost"
                    size="sm"
                    className="h-auto p-0 hover:bg-transparent"
                    onClick={() => removeKeyword(keyword)}
                  >
                    <X className="w-3 h-3" />
                  </Button>
                </Badge>
              ))}
            </div>
          </div>

          {/* Current Stats */}
          {ad && (
            <div className="bg-muted p-4 rounded-lg">
              <h4 className="font-medium mb-2">Current Statistics</h4>
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div>
                  <span className="text-muted-foreground">Daily Shown:</span>
                  <span className="ml-2 font-medium">{ad.dailyShown || 0} / {ad.dailyLimit}</span>
                </div>
                <div>
                  <span className="text-muted-foreground">Monthly Shown:</span>
                  <span className="ml-2 font-medium">{ad.monthlyShown || 0} / {ad.monthlyLimit}</span>
                </div>
                <div className="col-span-2">
                  <span className="text-muted-foreground">Created:</span>
                  <span className="ml-2 font-medium">
                    {ad.createdAt ? new Date(ad.createdAt).toLocaleDateString() : "Unknown"}
                  </span>
                </div>
              </div>
            </div>
          )}
        </div>

        <DialogFooter>
          <Button variant="outline" onClick={() => onOpenChange(false)} disabled={loading}>
            Cancel
          </Button>
          <Button onClick={handleSave} disabled={loading}>
            {loading ? "Updating..." : "Update Ad"}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  )
}
