"use client"

import { useEffect, useState } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Globe, ImageIcon, Video, Database, AlertCircle } from "lucide-react"

interface Stats {
  totalCollections: number
  webResults: number
  imageResults: number
  videoResults: number
  totalUsers: number
  totalRevenue: number
  totalAds: number
  lastUpdated: string
  error?: string
}


export function DashboardStats() {
  const [stats, setStats] = useState<Stats | null>(null)
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  useEffect(() => {
    async function fetchStats() {
      try {
        console.log("[v0] Fetching admin stats...")
        const response = await fetch("/api/admin/stats")

        if (!response.ok) {
          throw new Error(`HTTP ${response.status}`)
        }

        const data = await response.json()
        console.log("[v0] Stats received:", data)
        setStats(data)

        if (data.error) {
          setError(data.error)
        }
      } catch (error) {
        console.error("[v0] Failed to fetch stats:", error)
        setError("Failed to connect to database")
        setStats({
          totalCollections: 0,
          webResults: 0,
          imageResults: 0,
          videoResults: 0,
          totalUsers: 0,
          totalRevenue: 0,
          totalAds: 0,
          lastUpdated: new Date().toISOString(),
        })
      } finally {
        setLoading(false)
      }
    }

    fetchStats()
  }, [])

  if (loading) {
    return (
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        {[...Array(4)].map((_, i) => (
          <Card key={i} className="animate-pulse">
            <CardHeader className="pb-2">
              <div className="h-4 bg-muted rounded w-24" />
            </CardHeader>
            <CardContent>
              <div className="h-8 bg-muted rounded w-16" />
            </CardContent>
          </Card>
        ))}
      </div>
    )
  }

  if (!stats) {
    return (
      <Card>
        <CardContent className="p-6">
          <div className="flex items-center gap-2 text-destructive">
            <AlertCircle className="h-4 w-4" />
            <p>Failed to load statistics</p>
          </div>
        </CardContent>
      </Card>
    )
  }

  const statCards = [
    {
      title: "Total Collections",
      value: stats.totalCollections || 0,
      icon: Database,
      color: "text-chart-1",
    },
    {
      title: "Web Results",
      value: stats.webResults || 0,
      icon: Globe,
      color: "text-chart-2",
    },
    {
      title: "Image Results",
      value: stats.imageResults || 0,
      icon: ImageIcon,
      color: "text-chart-3",
    },
    {
      title: "Total Revenue",
      value: `₹${(stats.totalRevenue || 0).toFixed(2)}`,
      icon: Video,
      color: "text-chart-4",
    },
  ]

  return (
    <div className="space-y-6">
      {error && (
        <Card className="border-destructive/50 bg-destructive/5">
          <CardContent className="p-4">
            <div className="flex items-center gap-2 text-destructive">
              <AlertCircle className="h-4 w-4" />
              <p className="text-sm">{error}</p>
            </div>
          </CardContent>
        </Card>
      )}

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        {statCards.map((stat) => (
          <Card key={stat.title}>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium text-muted-foreground">{stat.title}</CardTitle>
              <stat.icon className={`h-4 w-4 ${stat.color}`} />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold font-playfair">
                {typeof stat.value === "string" ? stat.value : (stat.value || 0).toLocaleString()}
              </div>
              <p className="text-xs text-muted-foreground mt-1">
                {stat.title === "Total Revenue"
                  ? "from user ads"
                  : `Last updated: ${new Date(stats.lastUpdated).toLocaleDateString()}`}
              </p>
            </CardContent>
          </Card>
        ))}
      </div>
    </div>
  )
}
