"use client"

import { useEffect, useState } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Progress } from "@/components/ui/progress"
import { Database, TrendingUp } from "lucide-react"

interface CollectionData {
  name: string
  count: number
  percentage: number
  color: string
}

export function CollectionOverview() {
  const [collections, setCollections] = useState<CollectionData[]>([])
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    async function fetchStats() {
      try {
        const response = await fetch("/api/admin/stats")
        const stats = await response.json()

        const total = stats.webResults + stats.imageResults + stats.videoResults

        const collectionsData: CollectionData[] = [
          {
            name: "Web Results",
            count: stats.webResults,
            percentage: total > 0 ? (stats.webResults / total) * 100 : 0,
            color: "bg-chart-2",
          },
          {
            name: "Image Results",
            count: stats.imageResults,
            percentage: total > 0 ? (stats.imageResults / total) * 100 : 0,
            color: "bg-chart-3",
          },
          {
            name: "Video Results",
            count: stats.videoResults,
            percentage: total > 0 ? (stats.videoResults / total) * 100 : 0,
            color: "bg-chart-4",
          },
        ]

        setCollections(collectionsData)
      } catch (error) {
        console.error("Failed to fetch collection overview:", error)
      } finally {
        setLoading(false)
      }
    }

    fetchStats()
  }, [])

  if (loading) {
    return (
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center space-x-2">
            <Database className="h-5 w-5" />
            <span>Collection Overview</span>
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          {[...Array(3)].map((_, i) => (
            <div key={i} className="space-y-2 animate-pulse">
              <div className="flex justify-between">
                <div className="h-4 bg-muted rounded w-24" />
                <div className="h-4 bg-muted rounded w-16" />
              </div>
              <div className="h-2 bg-muted rounded" />
            </div>
          ))}
        </CardContent>
      </Card>
    )
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center space-x-2">
          <Database className="h-5 w-5" />
          <span>Collection Overview</span>
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-6">
        {collections.map((collection) => (
          <div key={collection.name} className="space-y-2">
            <div className="flex items-center justify-between">
              <span className="text-sm font-medium text-foreground">{collection.name}</span>
              <span className="text-sm text-muted-foreground">{collection.count.toLocaleString()} items</span>
            </div>
            <Progress value={collection.percentage} className="h-2" />
            <div className="flex items-center justify-between text-xs text-muted-foreground">
              <span>{collection.percentage.toFixed(1)}% of total</span>
              <TrendingUp className="h-3 w-3" />
            </div>
          </div>
        ))}
      </CardContent>
    </Card>
  )
}
