"use client"

import type React from "react"

import { useState } from "react"
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { Upload, AlertCircle, CheckCircle } from "lucide-react"
import { Alert, AlertDescription } from "@/components/ui/alert"

interface BulkUploadDialogProps {
  type: "web" | "image"
  open: boolean
  onOpenChange: (open: boolean) => void
  onUploadComplete: () => void
}

export function BulkUploadDialog({ type, open, onOpenChange, onUploadComplete }: BulkUploadDialogProps) {
  const [file, setFile] = useState<File | null>(null)
  const [jsonText, setJsonText] = useState("")
  const [loading, setLoading] = useState(false)
  const [result, setResult] = useState<{ success: boolean; message: string; count?: number } | null>(null)

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const selectedFile = e.target.files?.[0]
    if (selectedFile && selectedFile.type === "application/json") {
      setFile(selectedFile)
      const reader = new FileReader()
      reader.onload = (event) => {
        setJsonText(event.target?.result as string)
      }
      reader.readAsText(selectedFile)
    }
  }

  const handleUpload = async () => {
    if (!jsonText) return

    setLoading(true)
    setResult(null)

    try {
      const data = JSON.parse(jsonText)
      const response = await fetch(`/api/admin/bulk-upload`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          type,
          data: Array.isArray(data) ? data : [data],
        }),
      })

      const result = await response.json()

      if (response.ok) {
        setResult({ success: true, message: result.message, count: result.count })
        onUploadComplete()
        // Reset form after successful upload
        setTimeout(() => {
          setFile(null)
          setJsonText("")
          setResult(null)
          onOpenChange(false)
        }, 2000)
      } else {
        setResult({ success: false, message: result.error || "Upload failed" })
      }
    } catch (error) {
      setResult({ success: false, message: "Invalid JSON format or upload failed" })
    } finally {
      setLoading(false)
    }
  }

  const exampleData =
    type === "web"
      ? `[
  {
    "title": "Example Website",
    "description": "This is an example website description",
    "url": "https://example.com",
    "keywords": ["example", "website", "demo"],
    "safesearch": true,
    "maindomain": true
  }
]`
      : `[
  {
    "title": "Example Image",
    "url": "https://example.com/page",
    "imgurl": "https://example.com/image.jpg",
    "keywords": ["example", "image", "demo"],
    "safesearch": true
  }
]`

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-2xl max-h-[80vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="flex items-center space-x-2">
            <Upload className="h-5 w-5" />
            <span>Bulk Upload {type === "web" ? "Web" : "Image"} Results</span>
          </DialogTitle>
        </DialogHeader>

        <div className="space-y-6">
          <div className="space-y-2">
            <Label htmlFor="json-file">Upload JSON File</Label>
            <Input id="json-file" type="file" accept=".json" onChange={handleFileChange} className="cursor-pointer" />
          </div>

          <div className="space-y-2">
            <Label htmlFor="json-text">JSON Data</Label>
            <Textarea
              id="json-text"
              placeholder={`Paste your JSON data here or upload a file...\n\nExample format:\n${exampleData}`}
              value={jsonText}
              onChange={(e) => setJsonText(e.target.value)}
              rows={12}
              className="font-mono text-sm"
            />
          </div>

          {result && (
            <Alert className={result.success ? "border-green-200 bg-green-50" : "border-red-200 bg-red-50"}>
              {result.success ? (
                <CheckCircle className="h-4 w-4 text-green-600" />
              ) : (
                <AlertCircle className="h-4 w-4 text-red-600" />
              )}
              <AlertDescription className={result.success ? "text-green-800" : "text-red-800"}>
                {result.message}
                {result.count && ` (${result.count} items processed)`}
              </AlertDescription>
            </Alert>
          )}

          <div className="flex justify-end space-x-2">
            <Button variant="outline" onClick={() => onOpenChange(false)}>
              Cancel
            </Button>
            <Button onClick={handleUpload} disabled={!jsonText || loading}>
              {loading ? "Uploading..." : "Upload"}
            </Button>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  )
}
