"use client"

import type React from "react"
import { createContext, useContext, useEffect, useState, useRef } from "react"

interface AdminUser {
  id: string
  username: string
  email?: string
  createdAt: string
  lastLoginAt?: string
}

interface AdminAuthContextType {
  adminUser: AdminUser | null
  isLoading: boolean
  login: (username: string, password: string) => Promise<{ success: boolean; message: string }>
  logout: () => void
  updatePassword: (currentPassword: string, newPassword: string) => Promise<{ success: boolean; message: string }>
}

const AdminAuthContext = createContext<AdminAuthContextType | undefined>(undefined)

export function AdminAuthProvider({ children }: { children: React.ReactNode }) {
  const [adminUser, setAdminUser] = useState<AdminUser | null>(null)
  const [isLoading, setIsLoading] = useState(true)
  const isMountedRef = useRef(true)

  useEffect(() => {
    isMountedRef.current = true
    checkAdminAuth()

    return () => {
      isMountedRef.current = false
    }
  }, [])

  const checkAdminAuth = async () => {
    try {
      const response = await fetch("/api/admin/auth/validate", {
        credentials: "include",
      })

      if (response.ok) {
        const data = await response.json()
        if (isMountedRef.current) {
          setAdminUser(data.admin)
        }
      }
    } catch (error) {
      console.error("Admin auth check failed:", error)
    } finally {
      if (isMountedRef.current) {
        setIsLoading(false)
      }
    }
  }

  const login = async (username: string, password: string) => {
    try {
      const response = await fetch("/api/admin/auth/login", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify({ username, password }),
      })

      const data = await response.json()

      if (response.ok) {
        if (isMountedRef.current) {
          setAdminUser(data.admin)
        }
        return { success: true, message: "Login successful" }
      } else {
        return { success: false, message: data.message || "Login failed" }
      }
    } catch (error) {
      return { success: false, message: "Network error occurred" }
    }
  }

  const logout = async () => {
    try {
      await fetch("/api/admin/auth/logout", {
        method: "POST",
        credentials: "include",
      })
    } catch (error) {
      console.error("Logout error:", error)
    } finally {
      if (isMountedRef.current) {
        setAdminUser(null)
      }
    }
  }

  const updatePassword = async (currentPassword: string, newPassword: string) => {
    try {
      const response = await fetch("/api/admin/auth/update-password", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify({ currentPassword, newPassword }),
      })

      const data = await response.json()
      return { success: response.ok, message: data.message }
    } catch (error) {
      return { success: false, message: "Network error occurred" }
    }
  }

  return (
    <AdminAuthContext.Provider value={{ adminUser, isLoading, login, logout, updatePassword }}>
      {children}
    </AdminAuthContext.Provider>
  )
}

export function useAdminAuth() {
  const context = useContext(AdminAuthContext)
  if (context === undefined) {
    throw new Error("useAdminAuth must be used within an AdminAuthProvider")
  }
  return context
}
