"use client"

import type React from "react"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Switch } from "@/components/ui/switch"
import { Textarea } from "@/components/ui/textarea"
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"

interface AddWebResultDialogProps {
  open: boolean
  onOpenChange: (open: boolean) => void
  onResultAdded: () => void
}

export function AddWebResultDialog({ open, onOpenChange, onResultAdded }: AddWebResultDialogProps) {
  const [form, setForm] = useState({
    title: "",
    description: "",
    url: "",
    keywords: "",
    safesearch: true,
    maindomain: false,
  })
  const [adding, setAdding] = useState(false)

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setAdding(true)

    try {
      const response = await fetch("/api/admin/add-web", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          ...form,
          keywords: form.keywords
            .split(",")
            .map((k) => k.trim())
            .filter(Boolean),
        }),
      })

      if (response.ok) {
        setForm({
          title: "",
          description: "",
          url: "",
          keywords: "",
          safesearch: true,
          maindomain: false,
        })
        onResultAdded()
      }
    } catch (error) {
      console.error("Failed to add result:", error)
    } finally {
      setAdding(false)
    }
  }

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-2xl">
        <DialogHeader>
          <DialogTitle>Add Web Result</DialogTitle>
        </DialogHeader>
        <form onSubmit={handleSubmit} className="space-y-4">
          <div>
            <Label htmlFor="title">Title *</Label>
            <Input
              id="title"
              value={form.title}
              onChange={(e) => setForm({ ...form, title: e.target.value })}
              required
            />
          </div>
          <div>
            <Label htmlFor="description">Description *</Label>
            <Textarea
              id="description"
              value={form.description}
              onChange={(e) => setForm({ ...form, description: e.target.value })}
              rows={3}
              required
            />
          </div>
          <div>
            <Label htmlFor="url">URL *</Label>
            <Input
              id="url"
              type="url"
              value={form.url}
              onChange={(e) => setForm({ ...form, url: e.target.value })}
              required
            />
          </div>
          <div>
            <Label htmlFor="keywords">Keywords (comma-separated)</Label>
            <Input
              id="keywords"
              value={form.keywords}
              onChange={(e) => setForm({ ...form, keywords: e.target.value })}
              placeholder="technology, web development, programming"
            />
          </div>
          <div className="flex items-center space-x-4">
            <div className="flex items-center space-x-2">
              <Switch
                id="safesearch"
                checked={form.safesearch}
                onCheckedChange={(checked) => setForm({ ...form, safesearch: checked })}
              />
              <Label htmlFor="safesearch">Safe Search</Label>
            </div>
            <div className="flex items-center space-x-2">
              <Switch
                id="maindomain"
                checked={form.maindomain}
                onCheckedChange={(checked) => setForm({ ...form, maindomain: checked })}
              />
              <Label htmlFor="maindomain">Main Domain</Label>
            </div>
          </div>
          <div className="flex justify-end space-x-2">
            <Button type="button" variant="outline" onClick={() => onOpenChange(false)}>
              Cancel
            </Button>
            <Button type="submit" disabled={adding}>
              {adding ? "Adding..." : "Add Result"}
            </Button>
          </div>
        </form>
      </DialogContent>
    </Dialog>
  )
}
