"use client"

import type React from "react"

import { useState } from "react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Switch } from "@/components/ui/switch"
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog"

interface AddImageResultDialogProps {
  open: boolean
  onOpenChange: (open: boolean) => void
  onResultAdded: () => void
}

export function AddImageResultDialog({ open, onOpenChange, onResultAdded }: AddImageResultDialogProps) {
  const [form, setForm] = useState({
    title: "",
    url: "",
    imgurl: "",
    keywords: "",
    safesearch: true,
  })
  const [adding, setAdding] = useState(false)

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setAdding(true)

    try {
      const response = await fetch("/api/admin/add-image", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          ...form,
          keywords: form.keywords
            .split(",")
            .map((k) => k.trim())
            .filter(Boolean),
        }),
      })

      if (response.ok) {
        setForm({
          title: "",
          url: "",
          imgurl: "",
          keywords: "",
          safesearch: true,
        })
        onResultAdded()
      }
    } catch (error) {
      console.error("Failed to add image result:", error)
    } finally {
      setAdding(false)
    }
  }

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-2xl">
        <DialogHeader>
          <DialogTitle>Add Image Result</DialogTitle>
        </DialogHeader>
        <form onSubmit={handleSubmit} className="space-y-4">
          <div>
            <Label htmlFor="title">Title *</Label>
            <Input
              id="title"
              value={form.title}
              onChange={(e) => setForm({ ...form, title: e.target.value })}
              required
            />
          </div>
          <div>
            <Label htmlFor="url">Source URL *</Label>
            <Input
              id="url"
              type="url"
              value={form.url}
              onChange={(e) => setForm({ ...form, url: e.target.value })}
              required
            />
          </div>
          <div>
            <Label htmlFor="imgurl">Image URL *</Label>
            <Input
              id="imgurl"
              type="url"
              value={form.imgurl}
              onChange={(e) => setForm({ ...form, imgurl: e.target.value })}
              required
            />
          </div>
          <div>
            <Label htmlFor="keywords">Keywords (comma-separated)</Label>
            <Input
              id="keywords"
              value={form.keywords}
              onChange={(e) => setForm({ ...form, keywords: e.target.value })}
              placeholder="nature, landscape, photography"
            />
          </div>
          <div className="flex items-center space-x-2">
            <Switch
              id="safesearch"
              checked={form.safesearch}
              onCheckedChange={(checked) => setForm({ ...form, safesearch: checked })}
            />
            <Label htmlFor="safesearch">Safe Search</Label>
          </div>
          <div className="flex justify-end space-x-2">
            <Button type="button" variant="outline" onClick={() => onOpenChange(false)}>
              Cancel
            </Button>
            <Button type="submit" disabled={adding}>
              {adding ? "Adding..." : "Add Image"}
            </Button>
          </div>
        </form>
      </DialogContent>
    </Dialog>
  )
}
