"use client"

import React from "react"
import { ExternalLink } from "lucide-react"

interface AdCardProps {
  ad: {
    id: string
    title: string
    description: string
    url?: string
    targetUrl?: string
    company?: string
    favicon?: string
    imageUrl?: string
    isAd: boolean
    adType?: string
    keywords?: string[]
    date?: string
    isUserAd?: boolean
    userId?: string
    costPerClick?: number
    costPerImpression?: number
    // Add these properties that might be missing
    active?: boolean
    dailyLimit?: number
    monthlyLimit?: number
    dailyShown?: number
    monthlyShown?: number
  }
}

export function AdCard({ ad }: AdCardProps) {
  const adUrl = ad.targetUrl || ad.url
  const displayUrl = adUrl ? new URL(adUrl).hostname : ""

  const getAdImage = () => {
    if (ad.imageUrl) {
      return ad.imageUrl
    }
    if (ad.favicon) {
      return ad.favicon
    }
    if (adUrl) {
      try {
        const domain = new URL(adUrl).hostname
        return `https://www.google.com/s2/favicons?domain=${domain}&sz=32`
      } catch {
        return null
      }
    }
    return null
  }

  const adImage = getAdImage()

  const handleClick = async () => {
    if (adUrl) {
      try {
        await fetch("/api/ads/track", {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify({
            adId: ad.id,
            action: "click",
          }),
        })
      } catch (error) {
        console.error("Failed to track ad click:", error)
      }

      window.open(adUrl, "_blank", "noopener,noreferrer")
    }
  }

  React.useEffect(() => {
    const trackImpression = async () => {
      try {
        await fetch("/api/ads/track", {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify({
            adId: ad.id,
            action: "impression",
          }),
        })
      } catch (error) {
        console.error("Failed to track ad impression:", error)
      }
    }

    trackImpression()
  }, [ad.id])

  return (
    <div className="border border-orange-200 bg-gradient-to-r from-orange-50 to-amber-50 dark:bg-gradient-to-r dark:from-orange-900/20 dark:to-amber-900/20 dark:border-orange-800 rounded-lg p-4 mb-6 max-w-2xl hover:shadow-md transition-shadow">
      {/* Ad Label */}
      <div className="flex items-center justify-between mb-3">
        <div className="flex items-center gap-2">
          <span className="inline-flex items-center px-2 py-1 text-xs font-medium bg-orange-600 text-white rounded">
            Ad
          </span>
          {ad.isUserAd && (
            <span className="inline-flex items-center px-2 py-1 text-xs font-medium bg-blue-600 text-white rounded">
              Sponsored
            </span>
          )}
          {!ad.isUserAd && (
            <span className="text-xs text-muted-foreground">• Featured</span>
          )}
          {ad.adType && <span className="text-xs text-muted-foreground">• {ad.adType}</span>}
        </div>
        {adUrl && <ExternalLink className="w-4 h-4 text-muted-foreground" />}
      </div>

      {/* Rest of your existing component code */}
      <div
        className={`cursor-pointer ${adUrl ? "hover:bg-orange-100 dark:hover:bg-orange-900/30 rounded-md p-2 -m-2 transition-colors" : ""}`}
        onClick={adUrl ? handleClick : undefined}
      >
        <div className="flex items-start gap-3">
          {adImage && (
            <div className="flex-shrink-0 mt-1">
              <img
                src={adImage || "/placeholder.svg"}
                alt={ad.company || ad.title}
                className="w-8 h-8 rounded-sm object-contain"
                onError={(e) => {
                  if (adUrl && !e.currentTarget.src.includes("google.com/s2/favicons")) {
                    try {
                      const domain = new URL(adUrl).hostname
                      e.currentTarget.src = `https://www.google.com/s2/favicons?domain=${domain}&sz=32`
                    } catch {
                      e.currentTarget.style.display = "none"
                    }
                  } else {
                    e.currentTarget.style.display = "none"
                  }
                }}
              />
            </div>
          )}

          <div className="flex-1 min-w-0">
            {displayUrl && (
              <div className="text-sm text-green-600 dark:text-green-400 truncate mb-1">
                {displayUrl}
                {ad.company && ad.company !== displayUrl && (
                  <span className="text-muted-foreground ml-2">by {ad.company}</span>
                )}
              </div>
            )}

            <h3 className="text-lg font-medium mb-1">
              <span className="text-blue-600 dark:text-blue-400 hover:underline">{ad.title}</span>
            </h3>

            <p className="text-gray-700 dark:text-gray-300 text-sm leading-relaxed mb-2 max-w-xl">{ad.description}</p>

            {ad.imageUrl && (
              <div className="mt-2">
                <img
                  src={ad.imageUrl || "/placeholder.svg"}
                  alt={`${ad.title} ad image`}
                  className="w-full h-auto max-h-48 object-cover rounded-md"
                  onError={(e) => {
                    e.currentTarget.style.display = "none"
                  }}
                />
              </div>
            )}

            {/* Only show keywords for admin ads, not user ads */}
            {!ad.isUserAd && ad.keywords && ad.keywords.length > 0 && (
              <div className="flex flex-wrap gap-2 text-sm mt-2">
                {ad.keywords.slice(0, 4).map((keyword) => (
                  <span
                    key={keyword}
                    className="text-blue-600 dark:text-blue-400 hover:underline capitalize cursor-pointer text-xs bg-blue-50 dark:bg-blue-900/30 px-2 py-1 rounded"
                    onClick={(e) => {
                      e.stopPropagation()
                      if (adUrl) {
                        window.open(`${adUrl}?q=${encodeURIComponent(keyword)}`, "_blank", "noopener,noreferrer")
                      }
                    }}
                  >
                    {keyword}
                  </span>
                ))}
              </div>
            )}

            <div className="flex items-center gap-4 text-xs text-muted-foreground mt-2">
              {ad.company && !displayUrl && <span>{ad.company}</span>}
              {ad.date && (
                <span>
                  {new Date(ad.date).toLocaleDateString("en-US", {
                    year: "numeric",
                    month: "short",
                    day: "numeric",
                  })}
                </span>
              )}
            </div>
          </div>
        </div>
      </div>

      {adUrl && (
        <div className="mt-3 pt-3 border-t border-orange-300 dark:border-orange-700">
          <button
            onClick={handleClick}
            className="inline-flex items-center gap-2 text-sm font-medium text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300 transition-colors"
          >
            Visit Site
            <ExternalLink className="w-3 h-3" />
          </button>
        </div>
      )}
    </div>
  )
}
