"use client"

import type React from "react"

import { useState, useEffect, useRef } from "react"
import { Search, Plus, Edit, Trash2, Eye, CheckCircle, XCircle, Clock, Filter, Upload, Download } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { Switch } from "@/components/ui/switch"
import { Alert, AlertDescription } from "@/components/ui/alert"
import type { WebsiteSubmission } from "@/lib/typesense"

export default function WebsitesPage() {
  const [websites, setWebsites] = useState<WebsiteSubmission[]>([])
  const [loading, setLoading] = useState(true)
  const [searchQuery, setSearchQuery] = useState("")
  const [statusFilter, setStatusFilter] = useState("all")
  const [selectedWebsite, setSelectedWebsite] = useState<WebsiteSubmission | null>(null)
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false)
  const [isViewDialogOpen, setIsViewDialogOpen] = useState(false)
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false)
  const [isBulkUploadDialogOpen, setIsBulkUploadDialogOpen] = useState(false)

  const [stats, setStats] = useState({
    total: 0,
    pending: 0,
    approved: 0,
    rejected: 0,
  })

  useEffect(() => {
    fetchWebsites()
  }, [searchQuery, statusFilter])

  const fetchWebsites = async () => {
    try {
      setLoading(true)
      const params = new URLSearchParams()
      if (searchQuery) params.append("q", searchQuery)
      if (statusFilter !== "all") params.append("status", statusFilter)

      const response = await fetch(`/api/admin/websites?${params}`)
      const data = await response.json()

      setWebsites(data.websites || [])
      setStats(data.stats || { total: 0, pending: 0, approved: 0, rejected: 0 })
    } catch (error) {
      console.error("Error fetching websites:", error)
    } finally {
      setLoading(false)
    }
  }

  const handleStatusUpdate = async (
    websiteId: string,
    status: string,
    notes?: string,
    webcrawl_status?: string,
    imgcrawl_status?: string,
  ) => {
    try {
      const response = await fetch(`/api/admin/websites/${websiteId}`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          status,
          notes,
          webcrawl_status,
          imgcrawl_status,
          reviewedAt: new Date().toISOString(),
          reviewedBy: "Admin",
        }),
      })

      if (response.ok) {
        fetchWebsites()
        setIsEditDialogOpen(false)
        setSelectedWebsite(null)
      }
    } catch (error) {
      console.error("Error updating website:", error)
    }
  }

  const handleDelete = async (websiteId: string) => {
    if (!confirm("Are you sure you want to delete this website submission?")) return

    try {
      const response = await fetch(`/api/admin/websites/${websiteId}`, {
        method: "DELETE",
      })

      if (response.ok) {
        fetchWebsites()
      }
    } catch (error) {
      console.error("Error deleting website:", error)
    }
  }

  const getStatusBadge = (status: string) => {
    switch (status) {
      case "approved":
        return <Badge className="bg-green-100 text-green-800 dark:bg-green-900/20 dark:text-green-400">Approved</Badge>
      case "rejected":
        return <Badge className="bg-red-100 text-red-800 dark:bg-red-900/20 dark:text-red-400">Rejected</Badge>
      default:
        return (
          <Badge className="bg-yellow-100 text-yellow-800 dark:bg-yellow-900/20 dark:text-yellow-400">Pending</Badge>
        )
    }
  }

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "approved":
        return <CheckCircle className="w-4 h-4 text-green-600" />
      case "rejected":
        return <XCircle className="w-4 h-4 text-red-600" />
      default:
        return <Clock className="w-4 h-4 text-yellow-600" />
    }
  }

  const downloadCSVTemplate = () => {
    const csvContent = `title,url,imageUrl,supportWebCrawl,supportImageCrawl,ownerConfirmed,notes
Example Website,https://example.com,https://example.com/image.jpg,true,false,true,Sample website for demonstration
Another Site,https://another-site.com,,false,true,false,Website with image crawling enabled`

    const blob = new Blob([csvContent], { type: "text/csv;charset=utf-8;" })
    const link = document.createElement("a")
    const url = URL.createObjectURL(blob)
    link.setAttribute("href", url)
    link.setAttribute("download", "websites_template.csv")
    link.style.visibility = "hidden"
    document.body.appendChild(link)
    link.click()
    document.body.removeChild(link)
  }

  return (
    <div className="space-y-6">
      <div>
        <h1 className="text-3xl font-bold">Websites Management</h1>
        <p className="text-muted-foreground">Manage website submissions and review requests</p>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Total Submissions</p>
                <p className="text-2xl font-bold">{stats.total}</p>
              </div>
              <Plus className="w-8 h-8 text-muted-foreground" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Pending Review</p>
                <p className="text-2xl font-bold text-yellow-600">{stats.pending}</p>
              </div>
              <Clock className="w-8 h-8 text-yellow-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Approved</p>
                <p className="text-2xl font-bold text-green-600">{stats.approved}</p>
              </div>
              <CheckCircle className="w-8 h-8 text-green-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-muted-foreground">Rejected</p>
                <p className="text-2xl font-bold text-red-600">{stats.rejected}</p>
              </div>
              <XCircle className="w-8 h-8 text-red-600" />
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Search and Filters */}
      <Card>
        <CardContent className="p-6">
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="flex-1">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
                <Input
                  placeholder="Search websites by title or URL..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="pl-10"
                />
              </div>
            </div>
            <Select value={statusFilter} onValueChange={setStatusFilter}>
              <SelectTrigger className="w-full sm:w-48">
                <SelectValue placeholder="Filter by status" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Status</SelectItem>
                <SelectItem value="pending">Pending</SelectItem>
                <SelectItem value="approved">Approved</SelectItem>
                <SelectItem value="rejected">Rejected</SelectItem>
              </SelectContent>
            </Select>
            <Button onClick={() => setIsAddDialogOpen(true)} variant="default">
              <Plus className="w-4 h-4 mr-2" />
              Add Website
            </Button>
            <Button onClick={() => setIsBulkUploadDialogOpen(true)} variant="outline">
              <Upload className="w-4 h-4 mr-2" />
              Bulk Upload
            </Button>
            <Button onClick={fetchWebsites} variant="outline">
              <Filter className="w-4 h-4 mr-2" />
              Refresh
            </Button>
          </div>
        </CardContent>
      </Card>

      {/* Websites List */}
      <Card>
        <CardHeader>
          <CardTitle>Website Submissions</CardTitle>
          <CardDescription>
            Review and manage website submissions. Showing {websites.length} of {stats.total} submissions
          </CardDescription>
        </CardHeader>
        <CardContent>
          {loading ? (
            <div className="text-center py-8">Loading websites...</div>
          ) : websites.length === 0 ? (
            <div className="text-center py-8 text-muted-foreground">No website submissions found</div>
          ) : (
            <div className="space-y-4">
              {websites.map((website) => (
                <div
                  key={website.id}
                  className="flex items-center justify-between p-4 border rounded-lg hover:bg-muted/50 transition-colors"
                >
                  <div className="flex items-center space-x-4 flex-1">
                    {getStatusIcon(website.status)}
                    <div className="flex-1 min-w-0">
                      <div className="flex items-center gap-2 mb-1">
                        <h3 className="font-medium truncate">{website.title}</h3>
                        {getStatusBadge(website.status)}
                      </div>
                      <p className="text-sm text-muted-foreground truncate">{website.url}</p>
                      <div className="flex items-center gap-4 mt-2 text-xs text-muted-foreground">
                        <span>Submitted: {new Date(website.submittedAt).toLocaleDateString()}</span>
                        <span>Web Crawl: {website.supportWebCrawl ? "Yes" : "No"}</span>
                        <span>Image Crawl: {website.supportImageCrawl ? "Yes" : "No"}</span>
                        <span
                          className={`px-2 py-1 rounded text-xs font-medium ${
                            website.webcrawl_status === "complete"
                              ? "bg-green-100 text-green-800"
                              : website.webcrawl_status === "failed"
                                ? "bg-red-100 text-red-800"
                                : "bg-yellow-100 text-yellow-800"
                          }`}
                        >
                          Web: {website.webcrawl_status}
                        </span>
                        <span
                          className={`px-2 py-1 rounded text-xs font-medium ${
                            website.imgcrawl_status === "complete"
                              ? "bg-green-100 text-green-800"
                              : website.imgcrawl_status === "failed"
                                ? "bg-red-100 text-red-800"
                                : "bg-yellow-100 text-yellow-800"
                          }`}
                        >
                          Img: {website.imgcrawl_status || "pending"}
                        </span>
                        {website.ownerConfirmed && <span className="text-green-600">Owner Confirmed</span>}
                      </div>
                    </div>
                  </div>

                  <div className="flex items-center space-x-2">
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => {
                        setSelectedWebsite(website)
                        setIsViewDialogOpen(true)
                      }}
                    >
                      <Eye className="w-4 h-4" />
                    </Button>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => {
                        setSelectedWebsite(website)
                        setIsEditDialogOpen(true)
                      }}
                    >
                      <Edit className="w-4 h-4" />
                    </Button>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => handleDelete(website.id)}
                      className="text-red-600 hover:text-red-700"
                    >
                      <Trash2 className="w-4 h-4" />
                    </Button>
                  </div>
                </div>
              ))}
            </div>
          )}
        </CardContent>
      </Card>

      {/* View Dialog */}
      <Dialog open={isViewDialogOpen} onOpenChange={setIsViewDialogOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Website Details</DialogTitle>
            <DialogDescription>View complete website submission information</DialogDescription>
          </DialogHeader>
          {selectedWebsite && (
            <div className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label className="font-medium">Title</Label>
                  <p className="text-sm text-muted-foreground">{selectedWebsite.title}</p>
                </div>
                <div>
                  <Label className="font-medium">Status</Label>
                  <div className="mt-1">{getStatusBadge(selectedWebsite.status)}</div>
                </div>
              </div>

              <div>
                <Label className="font-medium">URL</Label>
                <p className="text-sm text-muted-foreground break-all">{selectedWebsite.url}</p>
              </div>

              {selectedWebsite.imageUrl && (
                <div>
                  <Label className="font-medium">Image</Label>
                  <div className="mt-2">
                    <img
                      src={selectedWebsite.imageUrl || "/placeholder.svg"}
                      alt="Website preview"
                      className="max-w-full h-32 object-cover rounded border"
                      onError={(e) => {
                        e.currentTarget.style.display = "none"
                      }}
                    />
                  </div>
                </div>
              )}

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label className="font-medium">Web Crawling</Label>
                  <p className="text-sm text-muted-foreground">
                    {selectedWebsite.supportWebCrawl ? "Enabled" : "Disabled"}
                  </p>
                </div>
                <div>
                  <Label className="font-medium">Image Crawling</Label>
                  <p className="text-sm text-muted-foreground">
                    {selectedWebsite.supportImageCrawl ? "Enabled" : "Disabled"}
                  </p>
                </div>
              </div>

              <div>
                <Label className="font-medium">Owner Confirmation</Label>
                <p className="text-sm text-muted-foreground">
                  {selectedWebsite.ownerConfirmed ? "Confirmed" : "Not confirmed"}
                </p>
              </div>

              <div>
                <Label className="font-medium">Submitted</Label>
                <p className="text-sm text-muted-foreground">
                  {new Date(selectedWebsite.submittedAt).toLocaleString()}
                </p>
              </div>

              {selectedWebsite.reviewedAt && (
                <div>
                  <Label className="font-medium">Reviewed</Label>
                  <p className="text-sm text-muted-foreground">
                    {new Date(selectedWebsite.reviewedAt).toLocaleString()} by {selectedWebsite.reviewedBy}
                  </p>
                </div>
              )}

              {selectedWebsite.notes && (
                <div>
                  <Label className="font-medium">Notes</Label>
                  <p className="text-sm text-muted-foreground">{selectedWebsite.notes}</p>
                </div>
              )}

              <div>
                <Label className="font-medium">Webcrawl Status</Label>
                <div className="mt-1">
                  <Badge
                    className={
                      selectedWebsite.webcrawl_status === "complete"
                        ? "bg-green-100 text-green-800"
                        : selectedWebsite.webcrawl_status === "failed"
                          ? "bg-red-100 text-red-800"
                          : "bg-yellow-100 text-yellow-800"
                    }
                  >
                    {selectedWebsite.webcrawl_status}
                  </Badge>
                </div>
              </div>

              <div>
                <Label className="font-medium">Image Crawl Status</Label>
                <div className="mt-1">
                  <Badge
                    className={
                      selectedWebsite.imgcrawl_status === "complete"
                        ? "bg-green-100 text-green-800"
                        : selectedWebsite.imgcrawl_status === "failed"
                          ? "bg-red-100 text-red-800"
                          : "bg-yellow-100 text-yellow-800"
                    }
                  >
                    {selectedWebsite.imgcrawl_status || "pending"}
                  </Badge>
                </div>
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>

      {/* Edit Dialog */}
      <Dialog open={isEditDialogOpen} onOpenChange={setIsEditDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Review Website</DialogTitle>
            <DialogDescription>Update the status and add review notes</DialogDescription>
          </DialogHeader>
          {selectedWebsite && (
            <WebsiteReviewForm
              website={selectedWebsite}
              onSubmit={(status, notes, webcrawl_status, imgcrawl_status) =>
                handleStatusUpdate(selectedWebsite.id, status, notes, webcrawl_status, imgcrawl_status)
              }
              onCancel={() => setIsEditDialogOpen(false)}
            />
          )}
        </DialogContent>
      </Dialog>

      {/* Add Website Dialog */}
      <Dialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Add New Website</DialogTitle>
            <DialogDescription>Add a website directly to the search index</DialogDescription>
          </DialogHeader>
          <AddWebsiteForm
            onSubmit={() => {
              setIsAddDialogOpen(false)
              fetchWebsites()
            }}
            onCancel={() => setIsAddDialogOpen(false)}
          />
        </DialogContent>
      </Dialog>

      {/* Bulk Upload Dialog */}
      <Dialog open={isBulkUploadDialogOpen} onOpenChange={setIsBulkUploadDialogOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Bulk Upload Websites</DialogTitle>
            <DialogDescription>Upload multiple websites using a CSV file</DialogDescription>
          </DialogHeader>
          <BulkUploadForm
            onSubmit={() => {
              setIsBulkUploadDialogOpen(false)
              fetchWebsites()
            }}
            onCancel={() => setIsBulkUploadDialogOpen(false)}
            onDownloadTemplate={downloadCSVTemplate}
          />
        </DialogContent>
      </Dialog>
    </div>
  )
}

function AddWebsiteForm({
  onSubmit,
  onCancel,
}: {
  onSubmit: () => void
  onCancel: () => void
}) {
  const [formData, setFormData] = useState({
    title: "",
    url: "",
    imageUrl: "",
    supportWebCrawl: true,
    supportImageCrawl: false,
  })
  const [isSubmitting, setIsSubmitting] = useState(false)
  const [error, setError] = useState("")

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsSubmitting(true)
    setError("")

    try {
      const response = await fetch("/api/admin/websites", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          ...formData,
          status: "approved", // Admin-added websites are auto-approved
          webcrawl_status: "pending", // Set default webcrawl_status for new websites
          imgcrawl_status: "pending", // Set default imgcrawl_status for new websites
          ownerConfirmed: true, // Admin confirmation
          submittedAt: new Date().toISOString(),
          reviewedAt: new Date().toISOString(),
          reviewedBy: "Admin",
        }),
      })

      if (response.ok) {
        onSubmit()
      } else {
        const data = await response.json()
        setError(data.error || "Failed to add website")
      }
    } catch (error) {
      setError("Failed to add website")
    } finally {
      setIsSubmitting(false)
    }
  }

  return (
    <form onSubmit={handleSubmit} className="space-y-4">
      {error && <div className="p-3 text-sm text-red-600 bg-red-50 border border-red-200 rounded">{error}</div>}

      <div>
        <Label htmlFor="title">Website Title *</Label>
        <Input
          id="title"
          value={formData.title}
          onChange={(e) => setFormData({ ...formData, title: e.target.value })}
          placeholder="Enter website title"
          required
        />
      </div>

      <div>
        <Label htmlFor="url">Website URL *</Label>
        <Input
          id="url"
          type="url"
          value={formData.url}
          onChange={(e) => setFormData({ ...formData, url: e.target.value })}
          placeholder="https://example.com"
          required
        />
      </div>

      <div>
        <Label htmlFor="imageUrl">Image URL (Optional)</Label>
        <Input
          id="imageUrl"
          type="url"
          value={formData.imageUrl}
          onChange={(e) => setFormData({ ...formData, imageUrl: e.target.value })}
          placeholder="https://example.com/image.jpg"
        />
      </div>

      <div className="space-y-4">
        <div className="flex items-center justify-between">
          <div>
            <Label htmlFor="webCrawl">Allow Web Crawling</Label>
            <p className="text-sm text-muted-foreground">Enable indexing of website content</p>
          </div>
          <Switch
            id="webCrawl"
            checked={formData.supportWebCrawl}
            onCheckedChange={(checked) => setFormData({ ...formData, supportWebCrawl: checked })}
          />
        </div>

        <div className="flex items-center justify-between">
          <div>
            <Label htmlFor="imageCrawl">Allow Image Crawling</Label>
            <p className="text-sm text-muted-foreground">Enable indexing of website images</p>
          </div>
          <Switch
            id="imageCrawl"
            checked={formData.supportImageCrawl}
            onCheckedChange={(checked) => setFormData({ ...formData, supportImageCrawl: checked })}
          />
        </div>
      </div>

      <div className="flex justify-end space-x-2 pt-4">
        <Button type="button" variant="outline" onClick={onCancel} disabled={isSubmitting}>
          Cancel
        </Button>
        <Button type="submit" disabled={isSubmitting}>
          {isSubmitting ? "Adding..." : "Add Website"}
        </Button>
      </div>
    </form>
  )
}

function WebsiteReviewForm({
  website,
  onSubmit,
  onCancel,
}: {
  website: WebsiteSubmission
  onSubmit: (status: string, notes: string, webcrawl_status?: string, imgcrawl_status?: string) => void
  onCancel: () => void
}) {
  const [formData, setFormData] = useState({
    title: website.title,
    url: website.url,
    imageUrl: website.imageUrl || "",
    supportWebCrawl: website.supportWebCrawl,
    supportImageCrawl: website.supportImageCrawl,
    ownerConfirmed: website.ownerConfirmed,
    status: website.status,
    notes: website.notes || "",
    webcrawl_status: website.webcrawl_status || "pending",
    imgcrawl_status: website.imgcrawl_status || "pending",
  })

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault()
    onSubmit(formData.status, formData.notes, formData.webcrawl_status, formData.imgcrawl_status)
  }

  return (
    <form onSubmit={handleSubmit} className="space-y-4">
      <div>
        <Label htmlFor="title">Website Title *</Label>
        <Input
          id="title"
          value={formData.title}
          onChange={(e) => setFormData({ ...formData, title: e.target.value })}
          placeholder="Enter website title"
          required
        />
      </div>

      <div>
        <Label htmlFor="url">Website URL *</Label>
        <Input
          id="url"
          type="url"
          value={formData.url}
          onChange={(e) => setFormData({ ...formData, url: e.target.value })}
          placeholder="https://example.com"
          required
        />
      </div>

      <div>
        <Label htmlFor="imageUrl">Image URL (Optional)</Label>
        <Input
          id="imageUrl"
          type="url"
          value={formData.imageUrl}
          onChange={(e) => setFormData({ ...formData, imageUrl: e.target.value })}
          placeholder="https://example.com/image.jpg"
        />
      </div>

      <div className="space-y-4">
        <div className="flex items-center justify-between">
          <div>
            <Label htmlFor="webCrawl">Allow Web Crawling</Label>
            <p className="text-sm text-muted-foreground">Enable indexing of website content</p>
          </div>
          <Switch
            id="webCrawl"
            checked={formData.supportWebCrawl}
            onCheckedChange={(checked) => setFormData({ ...formData, supportWebCrawl: checked })}
          />
        </div>

        <div className="flex items-center justify-between">
          <div>
            <Label htmlFor="imageCrawl">Allow Image Crawling</Label>
            <p className="text-sm text-muted-foreground">Enable indexing of website images</p>
          </div>
          <Switch
            id="imageCrawl"
            checked={formData.supportImageCrawl}
            onCheckedChange={(checked) => setFormData({ ...formData, supportImageCrawl: checked })}
          />
        </div>

        <div className="flex items-center justify-between">
          <div>
            <Label htmlFor="ownerConfirmed">Owner Confirmed</Label>
            <p className="text-sm text-muted-foreground">Website owner has confirmed submission</p>
          </div>
          <Switch
            id="ownerConfirmed"
            checked={formData.ownerConfirmed}
            onCheckedChange={(checked) => setFormData({ ...formData, ownerConfirmed: checked })}
          />
        </div>
      </div>

      <div>
        <Label htmlFor="status">Status</Label>
        <Select value={formData.status} onValueChange={(value) => setFormData({ ...formData, status: value })}>
          <SelectTrigger>
            <SelectValue />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="pending">Pending</SelectItem>
            <SelectItem value="approved">Approved</SelectItem>
            <SelectItem value="rejected">Rejected</SelectItem>
          </SelectContent>
        </Select>
      </div>

      <div>
        <Label htmlFor="notes">Review Notes</Label>
        <Textarea
          id="notes"
          value={formData.notes}
          onChange={(e) => setFormData({ ...formData, notes: e.target.value })}
          placeholder="Add notes about your review decision..."
          rows={3}
        />
      </div>

      <div>
        <Label htmlFor="webcrawl_status">Webcrawl Status</Label>
        <Select
          value={formData.webcrawl_status}
          onValueChange={(value) => setFormData({ ...formData, webcrawl_status: value })}
        >
          <SelectTrigger>
            <SelectValue />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="pending">Pending</SelectItem>
            <SelectItem value="complete">Complete</SelectItem>
            <SelectItem value="failed">Failed</SelectItem>
          </SelectContent>
        </Select>
      </div>

      <div>
        <Label htmlFor="imgcrawl_status">Image Crawl Status</Label>
        <Select
          value={formData.imgcrawl_status}
          onValueChange={(value) => setFormData({ ...formData, imgcrawl_status: value })}
        >
          <SelectTrigger>
            <SelectValue />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="pending">Pending</SelectItem>
            <SelectItem value="complete">Complete</SelectItem>
            <SelectItem value="failed">Failed</SelectItem>
          </SelectContent>
        </Select>
      </div>

      <div className="flex justify-end space-x-2">
        <Button type="button" variant="outline" onClick={onCancel}>
          Cancel
        </Button>
        <Button type="submit">Update Website</Button>
      </div>
    </form>
  )
}

function BulkUploadForm({
  onSubmit,
  onCancel,
  onDownloadTemplate,
}: {
  onSubmit: () => void
  onCancel: () => void
  onDownloadTemplate: () => void
}) {
  const [file, setFile] = useState<File | null>(null)
  const [isUploading, setIsUploading] = useState(false)
  const [uploadResult, setUploadResult] = useState<{
    success: number
    failed: number
    errors: string[]
  } | null>(null)
  const [error, setError] = useState("")
  const fileInputRef = useRef<HTMLInputElement>(null)

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const selectedFile = e.target.files?.[0]
    if (selectedFile) {
      if (selectedFile.type !== "text/csv" && !selectedFile.name.endsWith(".csv")) {
        setError("Please select a valid CSV file")
        return
      }
      setFile(selectedFile)
      setError("")
      setUploadResult(null)
    }
  }

  const handleUpload = async () => {
    if (!file) {
      setError("Please select a CSV file")
      return
    }

    setIsUploading(true)
    setError("")

    try {
      const formData = new FormData()
      formData.append("file", file)

      const response = await fetch("/api/admin/websites/bulk-upload", {
        method: "POST",
        body: formData,
      })

      const data = await response.json()

      if (response.ok) {
        setUploadResult(data)
        if (data.success > 0) {
          setTimeout(() => {
            onSubmit()
          }, 2000)
        }
      } else {
        setError(data.error || "Upload failed")
      }
    } catch (error) {
      setError("Upload failed. Please try again.")
    } finally {
      setIsUploading(false)
    }
  }

  const resetForm = () => {
    setFile(null)
    setError("")
    setUploadResult(null)
    if (fileInputRef.current) {
      fileInputRef.current.value = ""
    }
  }

  return (
    <div className="space-y-6">
      {/* Template Download Section */}
      <div className="p-4 bg-blue-50 border border-blue-200 rounded-lg">
        <div className="flex items-start justify-between">
          <div>
            <h4 className="font-medium text-blue-900">Download CSV Template</h4>
            <p className="text-sm text-blue-700 mt-1">
              Download the template file to see the required format and column headers.
            </p>
          </div>
          <Button onClick={onDownloadTemplate} variant="outline" size="sm">
            <Download className="w-4 h-4 mr-2" />
            Download Template
          </Button>
        </div>
      </div>

      {/* File Upload Section */}
      <div className="space-y-4">
        <div>
          <Label htmlFor="csvFile">Select CSV File</Label>
          <Input
            ref={fileInputRef}
            id="csvFile"
            type="file"
            accept=".csv"
            onChange={handleFileChange}
            className="mt-1"
          />
          {file && (
            <p className="text-sm text-muted-foreground mt-2">
              Selected: {file.name} ({(file.size / 1024).toFixed(1)} KB)
            </p>
          )}
        </div>

        {/* CSV Format Instructions */}
        <div className="p-4 bg-gray-50 border rounded-lg">
          <h4 className="font-medium mb-2">CSV Format Requirements:</h4>
          <ul className="text-sm text-muted-foreground space-y-1">
            <li>
              • <strong>title</strong>: Website title (required)
            </li>
            <li>
              • <strong>url</strong>: Website URL (required, must start with http:// or https://)
            </li>
            <li>
              • <strong>imageUrl</strong>: Image URL (optional)
            </li>
            <li>
              • <strong>supportWebCrawl</strong>: true/false (default: true)
            </li>
            <li>
              • <strong>supportImageCrawl</strong>: true/false (default: false)
            </li>
            <li>
              • <strong>ownerConfirmed</strong>: true/false (default: false)
            </li>
            <li>
              • <strong>notes</strong>: Additional notes (optional)
            </li>
          </ul>
        </div>
      </div>

      {/* Error Display */}
      {error && (
        <Alert className="border-red-200 bg-red-50">
          <AlertDescription className="text-red-700">{error}</AlertDescription>
        </Alert>
      )}

      {/* Upload Result */}
      {uploadResult && (
        <Alert className={uploadResult.failed > 0 ? "border-yellow-200 bg-yellow-50" : "border-green-200 bg-green-50"}>
          <AlertDescription className={uploadResult.failed > 0 ? "text-yellow-700" : "text-green-700"}>
            <div className="space-y-2">
              <p>
                Upload completed: {uploadResult.success} successful, {uploadResult.failed} failed
              </p>
              {uploadResult.errors.length > 0 && (
                <div>
                  <p className="font-medium">Errors:</p>
                  <ul className="list-disc list-inside space-y-1">
                    {uploadResult.errors.slice(0, 5).map((error, index) => (
                      <li key={index} className="text-sm">
                        {error}
                      </li>
                    ))}
                    {uploadResult.errors.length > 5 && (
                      <li className="text-sm">... and {uploadResult.errors.length - 5} more errors</li>
                    )}
                  </ul>
                </div>
              )}
            </div>
          </AlertDescription>
        </Alert>
      )}

      {/* Action Buttons */}
      <div className="flex justify-end space-x-2 pt-4">
        <Button type="button" variant="outline" onClick={onCancel} disabled={isUploading}>
          Cancel
        </Button>
        {uploadResult && uploadResult.success > 0 ? (
          <Button onClick={resetForm} variant="outline">
            Upload Another File
          </Button>
        ) : (
          <Button onClick={handleUpload} disabled={!file || isUploading}>
            {isUploading ? "Uploading..." : "Upload CSV"}
          </Button>
        )}
      </div>
    </div>
  )
}
