"use client"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { Switch } from "@/components/ui/switch"
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { WebResultsTable } from "@/components/admin/web-results-table"
import { Plus, Upload, Download } from "lucide-react"
import type { WebSearchResult } from "@/lib/typesense"

export default function WebManagementPage() {
  const [results, setResults] = useState<WebSearchResult[]>([])
  const [loading, setLoading] = useState(true)
  const [searchQuery, setSearchQuery] = useState("")
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false)
  const [isBulkUploadOpen, setIsBulkUploadOpen] = useState(false)
  const [addForm, setAddForm] = useState({
    title: "",
    description: "",
    url: "",
    keywords: "",
    safesearch: true,
    maindomain: false,
  })
  const [csvFile, setCsvFile] = useState<File | null>(null)
  const [uploading, setUploading] = useState(false)

  const fetchResults = async () => {
    try {
      const response = await fetch(`/api/admin/web-results?search=${encodeURIComponent(searchQuery)}`)
      if (response.ok) {
        const data = await response.json()
        setResults(data.results || [])
      }
    } catch (error) {
      console.error("Failed to fetch web results:", error)
    } finally {
      setLoading(false)
    }
  }

  useEffect(() => {
    fetchResults()
  }, [searchQuery])

  const handleAdd = async () => {
    try {
      const response = await fetch("/api/admin/web-results", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          ...addForm,
          keywords: addForm.keywords
            .split(",")
            .map((k) => k.trim())
            .filter(Boolean),
        }),
      })

      if (response.ok) {
        setIsAddDialogOpen(false)
        setAddForm({
          title: "",
          description: "",
          url: "",
          keywords: "",
          safesearch: true,
          maindomain: false,
        })
        fetchResults()
      }
    } catch (error) {
      console.error("Failed to add web result:", error)
    }
  }

  const handleBulkUpload = async () => {
    if (!csvFile) return

    setUploading(true)
    const formData = new FormData()
    formData.append("file", csvFile)

    try {
      const response = await fetch("/api/admin/web-results/bulk-upload", {
        method: "POST",
        body: formData,
      })

      if (response.ok) {
        setIsBulkUploadOpen(false)
        setCsvFile(null)
        fetchResults()
      }
    } catch (error) {
      console.error("Failed to upload CSV:", error)
    } finally {
      setUploading(false)
    }
  }

  const downloadTemplate = () => {
    const csvContent =
      "title,description,url,keywords,safesearch,maindomain\n" +
      'Example Title,Example description,https://example.com,"keyword1,keyword2",true,false'

    const blob = new Blob([csvContent], { type: "text/csv" })
    const url = window.URL.createObjectURL(blob)
    const a = document.createElement("a")
    a.href = url
    a.download = "web-results-template.csv"
    a.click()
    window.URL.revokeObjectURL(url)
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold font-playfair text-foreground">Web Results</h1>
          <p className="text-muted-foreground mt-2">Manage web search results and content</p>
        </div>
        <div className="flex items-center space-x-2">
          <Dialog open={isBulkUploadOpen} onOpenChange={setIsBulkUploadOpen}>
            <DialogTrigger asChild>
              <Button variant="outline">
                <Upload className="h-4 w-4 mr-2" />
                Bulk Upload
              </Button>
            </DialogTrigger>
            <DialogContent>
              <DialogHeader>
                <DialogTitle>Bulk Upload Web Results</DialogTitle>
              </DialogHeader>
              <div className="space-y-4">
                <div>
                  <Label htmlFor="csv-file">CSV File</Label>
                  <Input
                    id="csv-file"
                    type="file"
                    accept=".csv"
                    onChange={(e) => setCsvFile(e.target.files?.[0] || null)}
                  />
                  <p className="text-sm text-muted-foreground mt-1">
                    Upload a CSV file with columns: title, description, url, keywords, safesearch, maindomain
                  </p>
                </div>
                <Button variant="outline" onClick={downloadTemplate} className="w-full bg-transparent">
                  <Download className="h-4 w-4 mr-2" />
                  Download Template
                </Button>
                <div className="flex justify-end space-x-2">
                  <Button variant="outline" onClick={() => setIsBulkUploadOpen(false)}>
                    Cancel
                  </Button>
                  <Button onClick={handleBulkUpload} disabled={!csvFile || uploading}>
                    {uploading ? "Uploading..." : "Upload"}
                  </Button>
                </div>
              </div>
            </DialogContent>
          </Dialog>
          <Dialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen}>
            <DialogTrigger asChild>
              <Button>
                <Plus className="h-4 w-4 mr-2" />
                Add Result
              </Button>
            </DialogTrigger>
            <DialogContent className="max-w-2xl">
              <DialogHeader>
                <DialogTitle>Add Web Result</DialogTitle>
              </DialogHeader>
              <div className="space-y-4">
                <div>
                  <Label htmlFor="title">Title</Label>
                  <Input
                    id="title"
                    value={addForm.title}
                    onChange={(e) => setAddForm({ ...addForm, title: e.target.value })}
                  />
                </div>
                <div>
                  <Label htmlFor="description">Description</Label>
                  <Textarea
                    id="description"
                    value={addForm.description}
                    onChange={(e) => setAddForm({ ...addForm, description: e.target.value })}
                    rows={3}
                  />
                </div>
                <div>
                  <Label htmlFor="url">URL</Label>
                  <Input
                    id="url"
                    type="url"
                    value={addForm.url}
                    onChange={(e) => setAddForm({ ...addForm, url: e.target.value })}
                  />
                </div>
                <div>
                  <Label htmlFor="keywords">Keywords (comma-separated)</Label>
                  <Input
                    id="keywords"
                    value={addForm.keywords}
                    onChange={(e) => setAddForm({ ...addForm, keywords: e.target.value })}
                  />
                </div>
                <div className="flex items-center space-x-4">
                  <div className="flex items-center space-x-2">
                    <Switch
                      id="safesearch"
                      checked={addForm.safesearch}
                      onCheckedChange={(checked) => setAddForm({ ...addForm, safesearch: checked })}
                    />
                    <Label htmlFor="safesearch">Safe Search</Label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <Switch
                      id="maindomain"
                      checked={addForm.maindomain}
                      onCheckedChange={(checked) => setAddForm({ ...addForm, maindomain: checked })}
                    />
                    <Label htmlFor="maindomain">Main Domain</Label>
                  </div>
                </div>
                <div className="flex justify-end space-x-2">
                  <Button variant="outline" onClick={() => setIsAddDialogOpen(false)}>
                    Cancel
                  </Button>
                  <Button onClick={handleAdd}>Add Result</Button>
                </div>
              </div>
            </DialogContent>
          </Dialog>
        </div>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Search Results</CardTitle>
          <CardDescription>
            <div className="flex items-center space-x-2">
              <Input
                placeholder="Search web results..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="max-w-sm"
              />
            </div>
          </CardDescription>
        </CardHeader>
        <CardContent>
          <WebResultsTable
            results={results}
            loading={loading}
            onResultUpdated={fetchResults}
            onResultDeleted={fetchResults}
          />
        </CardContent>
      </Card>
    </div>
  )
}
