"use client"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Input } from "@/components/ui/input"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog"
import { Label } from "@/components/ui/label"
import { Search, Filter, Eye, Pause, Play, Trash2, TrendingUp, MousePointer, Wallet, Users } from "lucide-react"

interface UserAd {
  id: string
  userId: string
  username: string
  title: string
  description: string
  imageUrl?: string
  targetUrl: string
  keywords: string[]
  dailyLimit: number
  monthlyLimit: number
  costPerClick: number
  costPerImpression: number
  totalSpent: number
  clicks: number
  impressions: number
  isActive: boolean
  createdAt: string
  walletBalance: number
}

export default function AdminUserAdsPage() {
  const [userAds, setUserAds] = useState<UserAd[]>([])
  const [loading, setLoading] = useState(true)
  const [searchQuery, setSearchQuery] = useState("")
  const [statusFilter, setStatusFilter] = useState("all")
  const [selectedAd, setSelectedAd] = useState<UserAd | null>(null)
  const [editDialogOpen, setEditDialogOpen] = useState(false)

  useEffect(() => {
    fetchUserAds()
  }, [])

  const fetchUserAds = async () => {
    try {
      const response = await fetch("/api/admin/user-ads")
      if (response.ok) {
        const data = await response.json()
        setUserAds(data.ads || [])
      }
    } catch (error) {
      console.error("Error fetching user ads:", error)
    } finally {
      setLoading(false)
    }
  }

  const handleToggleAdStatus = async (adId: string, currentStatus: boolean) => {
    try {
      const response = await fetch(`/api/admin/user-ads/${adId}`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ isActive: !currentStatus }),
      })

      if (response.ok) {
        setUserAds(userAds.map((ad) => (ad.id === adId ? { ...ad, isActive: !currentStatus } : ad)))
      }
    } catch (error) {
      console.error("Error toggling ad status:", error)
    }
  }

  const handleDeleteAd = async (adId: string) => {
    if (!confirm("Are you sure you want to delete this user ad?")) return

    try {
      const response = await fetch(`/api/admin/user-ads/${adId}`, {
        method: "DELETE",
      })

      if (response.ok) {
        setUserAds(userAds.filter((ad) => ad.id !== adId))
      }
    } catch (error) {
      console.error("Error deleting ad:", error)
    }
  }

  const filteredAds = userAds.filter((ad) => {
    const matchesSearch =
      ad.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
      ad.description.toLowerCase().includes(searchQuery.toLowerCase()) ||
      ad.username.toLowerCase().includes(searchQuery.toLowerCase())

    const matchesStatus =
      statusFilter === "all" ||
      (statusFilter === "active" && ad.isActive) ||
      (statusFilter === "paused" && !ad.isActive) ||
      (statusFilter === "low-balance" && ad.walletBalance < Math.max(ad.costPerClick, ad.costPerImpression))

    return matchesSearch && matchesStatus
  })

  const totalUserAds = userAds.length
  const activeUserAds = userAds.filter((ad) => ad.isActive).length
  const totalRevenue = userAds.reduce((sum, ad) => sum + ad.totalSpent, 0)
  const totalClicks = userAds.reduce((sum, ad) => sum + ad.clicks, 0)

  return (
    <div className="space-y-6">
      {/* Header */}
      <div>
        <h1 className="text-3xl font-bold text-foreground">User Ads Management</h1>
        <p className="text-muted-foreground">Monitor and manage user-created advertisements</p>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total User Ads</CardTitle>
            <Users className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-primary">{totalUserAds}</div>
            <p className="text-xs text-muted-foreground">{activeUserAds} active</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Revenue</CardTitle>
            <Wallet className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-primary">₹{totalRevenue.toFixed(2)}</div>
            <p className="text-xs text-muted-foreground">from user ads</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Clicks</CardTitle>
            <MousePointer className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-primary">{totalClicks.toLocaleString()}</div>
            <p className="text-xs text-muted-foreground">across all user ads</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Active Users</CardTitle>
            <TrendingUp className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-primary">
              {new Set(userAds.filter((ad) => ad.isActive).map((ad) => ad.userId)).size}
            </div>
            <p className="text-xs text-muted-foreground">with active ads</p>
          </CardContent>
        </Card>
      </div>

      {/* Search and Filters */}
      <div className="flex flex-col sm:flex-row gap-4">
        <div className="relative flex-1">
          <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground h-4 w-4" />
          <Input
            placeholder="Search by title, description, or username..."
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="pl-10"
          />
        </div>
        <Select value={statusFilter} onValueChange={setStatusFilter}>
          <SelectTrigger className="w-full sm:w-48">
            <Filter className="w-4 h-4 mr-2" />
            <SelectValue placeholder="Filter by status" />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="all">All Ads</SelectItem>
            <SelectItem value="active">Active</SelectItem>
            <SelectItem value="paused">Paused</SelectItem>
            <SelectItem value="low-balance">Low Balance</SelectItem>
          </SelectContent>
        </Select>
        <Button onClick={fetchUserAds} variant="outline">
          Refresh
        </Button>
      </div>

      {/* User Ads List */}
      <Card>
        <CardHeader>
          <CardTitle>User Advertisements</CardTitle>
          <CardDescription>
            Manage user-created ads and monitor performance. Showing {filteredAds.length} of {totalUserAds} ads
          </CardDescription>
        </CardHeader>
        <CardContent>
          {loading ? (
            <div className="text-center py-12">
              <p className="text-muted-foreground">Loading user ads...</p>
            </div>
          ) : filteredAds.length === 0 ? (
            <div className="text-center py-12">
              <p className="text-muted-foreground">No user ads found</p>
            </div>
          ) : (
            <div className="space-y-4">
              {filteredAds.map((ad) => (
                <div key={ad.id} className="border border-border rounded-lg p-4 hover:bg-muted/50 transition-colors">
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      <div className="flex items-center gap-2 mb-2">
                        <h3 className="font-semibold text-foreground">{ad.title}</h3>
                        <Badge variant={ad.isActive ? "default" : "secondary"}>
                          {ad.isActive ? "Active" : "Paused"}
                        </Badge>
                        {ad.walletBalance < Math.max(ad.costPerClick, ad.costPerImpression) && (
                          <Badge variant="destructive">Low Balance</Badge>
                        )}
                      </div>
                      <p className="text-sm text-muted-foreground mb-2">{ad.description}</p>
                      <div className="flex items-center gap-4 text-xs text-muted-foreground">
                        <span>User: {ad.username}</span>
                        <span>Clicks: {ad.clicks}</span>
                        <span>Impressions: {ad.impressions}</span>
                        <span>Spent: ₹{ad.totalSpent.toFixed(2)}</span>
                        <span>Balance: ₹{ad.walletBalance.toFixed(2)}</span>
                      </div>
                    </div>
                    <div className="flex items-center gap-2 ml-4">
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => {
                          setSelectedAd(ad)
                          setEditDialogOpen(true)
                        }}
                      >
                        <Eye className="w-4 h-4" />
                      </Button>
                      <Button variant="outline" size="sm" onClick={() => handleToggleAdStatus(ad.id, ad.isActive)}>
                        {ad.isActive ? <Pause className="w-4 h-4" /> : <Play className="w-4 h-4" />}
                      </Button>
                      <Button variant="outline" size="sm" onClick={() => handleDeleteAd(ad.id)}>
                        <Trash2 className="w-4 h-4" />
                      </Button>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          )}
        </CardContent>
      </Card>

      {/* View/Edit Dialog */}
      <Dialog open={editDialogOpen} onOpenChange={setEditDialogOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>User Ad Details</DialogTitle>
            <DialogDescription>View and manage user advertisement</DialogDescription>
          </DialogHeader>
          {selectedAd && (
            <div className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label>Title</Label>
                  <p className="text-sm font-medium">{selectedAd.title}</p>
                </div>
                <div>
                  <Label>User</Label>
                  <p className="text-sm font-medium">{selectedAd.username}</p>
                </div>
              </div>
              <div>
                <Label>Description</Label>
                <p className="text-sm">{selectedAd.description}</p>
              </div>
              <div>
                <Label>Target URL</Label>
                <p className="text-sm text-blue-600">{selectedAd.targetUrl}</p>
              </div>
              <div>
                <Label>Keywords</Label>
                <div className="flex flex-wrap gap-1 mt-1">
                  {selectedAd.keywords.map((keyword, index) => (
                    <Badge key={index} variant="secondary" className="text-xs">
                      {keyword}
                    </Badge>
                  ))}
                </div>
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label>Cost per Click</Label>
                  <p className="text-sm font-medium">₹{selectedAd.costPerClick.toFixed(2)}</p>
                </div>
                <div>
                  <Label>Cost per Impression</Label>
                  <p className="text-sm font-medium">₹{selectedAd.costPerImpression.toFixed(2)}</p>
                </div>
              </div>
              <div className="grid grid-cols-3 gap-4">
                <div>
                  <Label>Clicks</Label>
                  <p className="text-sm font-medium">{selectedAd.clicks}</p>
                </div>
                <div>
                  <Label>Impressions</Label>
                  <p className="text-sm font-medium">{selectedAd.impressions}</p>
                </div>
                <div>
                  <Label>Total Spent</Label>
                  <p className="text-sm font-medium text-primary">₹{selectedAd.totalSpent.toFixed(2)}</p>
                </div>
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>
    </div>
  )
}
