"use client"

import { useState, useEffect } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Switch } from "@/components/ui/switch"
import { Badge } from "@/components/ui/badge"
import { Separator } from "@/components/ui/separator"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Textarea } from "@/components/ui/textarea"
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from "@/components/ui/alert-dialog"
import {
  Settings,
  Database,
  Key,
  Trash2,
  RefreshCw,
  CheckCircle,
  XCircle,
  Home,
  Plus,
  X,
  Palette,
  Globe,
  FileText,
} from "lucide-react"
import { FileUpload } from "@/components/admin/file-upload"
import { useToast } from "@/hooks/use-toast"

interface SystemInfo {
  collections: Array<{
    name: string
    numDocuments: number
    createdAt: string
  }>
  typesenseStatus: "connected" | "disconnected"
  youtubeStatus: "configured" | "not-configured"
}

interface SiteSettings {
  siteName: string
  siteTitle: string
  siteDescription: string
  siteKeywords: string
  themeColor: string
  favicon: string
  logo: string
  welcomeMessage: string
  popularSearches: string[]
  showSubmitWebsite: boolean
}

export default function SettingsPage() {
  const [systemInfo, setSystemInfo] = useState<SystemInfo | null>(null)
  const [siteSettings, setSiteSettings] = useState<SiteSettings>({
    siteName: "",
    siteTitle: "",
    siteDescription: "",
    siteKeywords: "",
    themeColor: "#10b981",
    favicon: "/favicon.ico",
    logo: "/logo.png",
    welcomeMessage: "",
    popularSearches: [],
    showSubmitWebsite: true,
  })
  const [homepageSettings, setHomepageSettings] = useState({
    popularSearches: [],
    welcomeMessage: "",
    showSubmitWebsite: true,
  })
  const [loading, setLoading] = useState(true)
  const [testingConnection, setTestingConnection] = useState(false)
  const [deletingCollection, setDeletingCollection] = useState<string | null>(null)
  const [savingHomepage, setSavingHomepage] = useState(false)
  const [savingSiteSettings, setSavingSiteSettings] = useState(false)
  const [newSearch, setNewSearch] = useState("")
  const [adminId, setAdminId] = useState("")
  const { toast } = useToast()

  const fetchSystemInfo = async () => {
    setLoading(true)
    try {
      const response = await fetch("/api/admin/system-info")
      const data = await response.json()
      setSystemInfo(data)
    } catch (error) {
      console.error("Failed to fetch system info:", error)
    } finally {
      setLoading(false)
    }
  }

  const fetchHomepageSettings = async () => {
    try {
      const response = await fetch("/api/admin/homepage-settings")
      if (response.ok) {
        const data = await response.json()
        setHomepageSettings(data)
      }
    } catch (error) {
      console.error("Failed to fetch homepage settings:", error)
    }
  }

  const fetchSiteSettings = async () => {
    try {
      const response = await fetch("/api/admin/site-settings")
      if (response.ok) {
        const data = await response.json()
        setSiteSettings(data)
      }
    } catch (error) {
      console.error("Failed to fetch site settings:", error)
    }
  }

  const saveHomepageSettings = async () => {
    setSavingHomepage(true)
    try {
      const response = await fetch("/api/admin/homepage-settings", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(homepageSettings),
      })
      if (response.ok) {
        // Settings saved successfully
      }
    } catch (error) {
      console.error("Failed to save homepage settings:", error)
    } finally {
      setSavingHomepage(false)
    }
  }

  const saveSiteSettings = async () => {
    setSavingSiteSettings(true)
    try {
      const response = await fetch("/api/admin/site-settings", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ ...siteSettings, adminId }),
      })
      if (response.ok) {
        toast({
          title: "Settings saved",
          description: "Site settings have been updated successfully",
        })
      } else {
        throw new Error("Failed to save settings")
      }
    } catch (error) {
      console.error("Failed to save site settings:", error)
      toast({
        title: "Save failed",
        description: "Failed to save site settings",
        variant: "destructive",
      })
    } finally {
      setSavingSiteSettings(false)
    }
  }

  const addPopularSearch = () => {
    if (newSearch.trim() && siteSettings.popularSearches.length < 5) {
      setSiteSettings((prev) => ({
        ...prev,
        popularSearches: [...prev.popularSearches, newSearch.trim()],
      }))
      setNewSearch("")
    }
  }

  const removePopularSearch = (index: number) => {
    setSiteSettings((prev) => ({
      ...prev,
      popularSearches: prev.popularSearches.filter((_, i) => i !== index),
    }))
  }

  useEffect(() => {
    fetchSystemInfo()
    fetchHomepageSettings()
    fetchSiteSettings()

    // Get admin ID from session/cookie
    const getAdminId = async () => {
      try {
        const response = await fetch("/api/admin/auth/validate")
        if (response.ok) {
          const data = await response.json()
          setAdminId(data.admin?.id || "")
        }
      } catch (error) {
        console.error("Failed to get admin ID:", error)
      }
    }
    getAdminId()
  }, [])

  const testConnection = async () => {
    setTestingConnection(true)
    try {
      const response = await fetch("/api/admin/test-connection")
      const result = await response.json()
      if (result.success) {
        await fetchSystemInfo()
      }
    } catch (error) {
      console.error("Connection test failed:", error)
    } finally {
      setTestingConnection(false)
    }
  }

  const deleteCollection = async (collectionName: string) => {
    setDeletingCollection(collectionName)
    try {
      const response = await fetch(`/api/admin/collections/${collectionName}`, {
        method: "DELETE",
      })
      if (response.ok) {
        await fetchSystemInfo()
      }
    } catch (error) {
      console.error("Failed to delete collection:", error)
    } finally {
      setDeletingCollection(null)
    }
  }

  const recreateCollections = async () => {
    try {
      const response = await fetch("/api/admin/recreate-collections", {
        method: "POST",
      })
      if (response.ok) {
        await fetchSystemInfo()
      }
    } catch (error) {
      console.error("Failed to recreate collections:", error)
    }
  }

  return (
    <div className="space-y-6">
      <div>
        <h1 className="text-3xl font-bold font-playfair text-foreground">Settings</h1>
        <p className="text-muted-foreground mt-2">Manage your IndusSearch configuration and system settings</p>
      </div>

      <Tabs defaultValue="general" className="space-y-6">
        <TabsList>
          <TabsTrigger value="general">General</TabsTrigger>
          <TabsTrigger value="appearance">Appearance</TabsTrigger>
          <TabsTrigger value="seo">SEO</TabsTrigger>
          <TabsTrigger value="homepage">Homepage</TabsTrigger>
          <TabsTrigger value="system">System</TabsTrigger>
          <TabsTrigger value="collections">Collections</TabsTrigger>
          <TabsTrigger value="integrations">Integrations</TabsTrigger>
        </TabsList>

        <TabsContent value="general" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center space-x-2">
                <Globe className="h-5 w-5" />
                <span>General Settings</span>
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div className="space-y-4">
                  <div>
                    <Label htmlFor="site-name">Site Name</Label>
                    <Input
                      id="site-name"
                      placeholder="IndusSearch"
                      value={siteSettings.siteName}
                      onChange={(e) => setSiteSettings((prev) => ({ ...prev, siteName: e.target.value }))}
                    />
                    <p className="text-xs text-muted-foreground mt-1">The name of your search engine</p>
                  </div>

                  <div>
                    <Label htmlFor="site-title">Site Title</Label>
                    <Input
                      id="site-title"
                      placeholder="IndusSearch - Advanced Search Engine"
                      value={siteSettings.siteTitle}
                      onChange={(e) => setSiteSettings((prev) => ({ ...prev, siteTitle: e.target.value }))}
                    />
                    <p className="text-xs text-muted-foreground mt-1">
                      Appears in browser title bar and search results
                    </p>
                  </div>
                </div>

                <div className="space-y-4">
                  <div>
                    <Label htmlFor="site-description">Site Description</Label>
                    <Textarea
                      id="site-description"
                      placeholder="Discover the web with IndusSearch - your comprehensive search engine..."
                      value={siteSettings.siteDescription}
                      onChange={(e) => setSiteSettings((prev) => ({ ...prev, siteDescription: e.target.value }))}
                      rows={3}
                    />
                    <p className="text-xs text-muted-foreground mt-1">
                      Brief description for search engines and social media
                    </p>
                  </div>

                  <div>
                    <Label htmlFor="site-keywords">Keywords</Label>
                    <Input
                      id="site-keywords"
                      placeholder="search engine, web search, image search"
                      value={siteSettings.siteKeywords}
                      onChange={(e) => setSiteSettings((prev) => ({ ...prev, siteKeywords: e.target.value }))}
                    />
                    <p className="text-xs text-muted-foreground mt-1">Comma-separated keywords for SEO</p>
                  </div>
                </div>
              </div>

              <div className="flex justify-end">
                <Button onClick={saveSiteSettings} disabled={savingSiteSettings}>
                  {savingSiteSettings ? (
                    <RefreshCw className="h-4 w-4 mr-2 animate-spin" />
                  ) : (
                    <CheckCircle className="h-4 w-4 mr-2" />
                  )}
                  Save General Settings
                </Button>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="appearance" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center space-x-2">
                <Palette className="h-5 w-5" />
                <span>Appearance Settings</span>
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div className="space-y-6">
                  <div>
                    <Label htmlFor="theme-color">Theme Color</Label>
                    <div className="flex items-center space-x-3 mt-2">
                      <Input
                        id="theme-color"
                        type="color"
                        value={siteSettings.themeColor}
                        onChange={(e) => setSiteSettings((prev) => ({ ...prev, themeColor: e.target.value }))}
                        className="w-16 h-10 p-1 border rounded"
                      />
                      <Input
                        placeholder="#10b981"
                        value={siteSettings.themeColor}
                        onChange={(e) => setSiteSettings((prev) => ({ ...prev, themeColor: e.target.value }))}
                        className="flex-1"
                      />
                    </div>
                    <p className="text-xs text-muted-foreground mt-1">Primary color used throughout the interface</p>
                  </div>

                  <FileUpload
                    type="favicon"
                    currentPath={siteSettings.favicon}
                    onUploadSuccess={(path) => setSiteSettings((prev) => ({ ...prev, favicon: path }))}
                    adminId={adminId}
                  />
                </div>

                <div>
                  <FileUpload
                    type="logo"
                    currentPath={siteSettings.logo}
                    onUploadSuccess={(path) => setSiteSettings((prev) => ({ ...prev, logo: path }))}
                    adminId={adminId}
                  />
                </div>
              </div>

              <div className="flex justify-end">
                <Button onClick={saveSiteSettings} disabled={savingSiteSettings}>
                  {savingSiteSettings ? (
                    <RefreshCw className="h-4 w-4 mr-2 animate-spin" />
                  ) : (
                    <CheckCircle className="h-4 w-4 mr-2" />
                  )}
                  Save Appearance Settings
                </Button>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="seo" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center space-x-2">
                <FileText className="h-5 w-5" />
                <span>SEO Settings</span>
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="space-y-4">
                <div>
                  <Label htmlFor="meta-title">Meta Title</Label>
                  <Input
                    id="meta-title"
                    placeholder="IndusSearch - Advanced Search Engine"
                    value={siteSettings.siteTitle}
                    onChange={(e) => setSiteSettings((prev) => ({ ...prev, siteTitle: e.target.value }))}
                  />
                  <p className="text-xs text-muted-foreground mt-1">
                    Title tag for search engines (50-60 characters recommended)
                  </p>
                </div>

                <div>
                  <Label htmlFor="meta-description">Meta Description</Label>
                  <Textarea
                    id="meta-description"
                    placeholder="Discover the web with IndusSearch - your comprehensive search engine for web results, images, and more."
                    value={siteSettings.siteDescription}
                    onChange={(e) => setSiteSettings((prev) => ({ ...prev, siteDescription: e.target.value }))}
                    rows={3}
                  />
                  <p className="text-xs text-muted-foreground mt-1">
                    Description for search engine results (150-160 characters recommended)
                  </p>
                </div>

                <div>
                  <Label htmlFor="meta-keywords">Meta Keywords</Label>
                  <Input
                    id="meta-keywords"
                    placeholder="search engine, web search, image search, IndusSearch"
                    value={siteSettings.siteKeywords}
                    onChange={(e) => setSiteSettings((prev) => ({ ...prev, siteKeywords: e.target.value }))}
                  />
                  <p className="text-xs text-muted-foreground mt-1">Comma-separated keywords for search engines</p>
                </div>
              </div>

              <div className="flex justify-end">
                <Button onClick={saveSiteSettings} disabled={savingSiteSettings}>
                  {savingSiteSettings ? (
                    <RefreshCw className="h-4 w-4 mr-2 animate-spin" />
                  ) : (
                    <CheckCircle className="h-4 w-4 mr-2" />
                  )}
                  Save SEO Settings
                </Button>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="homepage" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center space-x-2">
                <Home className="h-5 w-5" />
                <span>Homepage Configuration</span>
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="space-y-4">
                <div>
                  <Label htmlFor="welcome-message">Welcome Message</Label>
                  <Textarea
                    id="welcome-message"
                    placeholder="Search the web with IndusSearch"
                    value={siteSettings.welcomeMessage}
                    onChange={(e) => setSiteSettings((prev) => ({ ...prev, welcomeMessage: e.target.value }))}
                    className="mt-2"
                  />
                  <p className="text-xs text-muted-foreground mt-1">
                    This message appears below the search title on the homepage
                  </p>
                </div>

                <Separator />

                <div>
                  <Label>Popular Searches (Max 5)</Label>
                  <p className="text-xs text-muted-foreground mb-3">
                    Configure the popular search suggestions shown on the homepage
                  </p>

                  <div className="space-y-3">
                    <div className="flex space-x-2">
                      <Input
                        placeholder="Add popular search term..."
                        value={newSearch}
                        onChange={(e) => setNewSearch(e.target.value)}
                        onKeyPress={(e) => e.key === "Enter" && addPopularSearch()}
                        disabled={siteSettings.popularSearches.length >= 5}
                      />
                      <Button
                        onClick={addPopularSearch}
                        disabled={!newSearch.trim() || siteSettings.popularSearches.length >= 5}
                        size="sm"
                      >
                        <Plus className="h-4 w-4" />
                      </Button>
                    </div>

                    <div className="space-y-2">
                      {siteSettings.popularSearches.map((search, index) => (
                        <div key={index} className="flex items-center justify-between p-2 border rounded">
                          <span className="text-sm">{search}</span>
                          <Button onClick={() => removePopularSearch(index)} variant="ghost" size="sm">
                            <X className="h-4 w-4" />
                          </Button>
                        </div>
                      ))}
                    </div>

                    {siteSettings.popularSearches.length === 0 && (
                      <p className="text-sm text-muted-foreground text-center py-4">No popular searches configured</p>
                    )}
                  </div>
                </div>

                <Separator />

                <div className="flex items-center justify-between">
                  <div>
                    <Label htmlFor="show-submit-website">Show Submit Website Button</Label>
                    <p className="text-xs text-muted-foreground">
                      Display the submit website button for logged-in users
                    </p>
                  </div>
                  <Switch
                    id="show-submit-website"
                    checked={siteSettings.showSubmitWebsite}
                    onCheckedChange={(checked) => setSiteSettings((prev) => ({ ...prev, showSubmitWebsite: checked }))}
                  />
                </div>
              </div>

              <div className="flex justify-end">
                <Button onClick={saveSiteSettings} disabled={savingSiteSettings}>
                  {savingSiteSettings ? (
                    <RefreshCw className="h-4 w-4 mr-2 animate-spin" />
                  ) : (
                    <CheckCircle className="h-4 w-4 mr-2" />
                  )}
                  Save Homepage Settings
                </Button>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="system" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center space-x-2">
                <Settings className="h-5 w-5" />
                <span>System Status</span>
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              {loading ? (
                <div className="space-y-3">
                  {[...Array(3)].map((_, i) => (
                    <div key={i} className="flex items-center justify-between animate-pulse">
                      <div className="h-4 bg-muted rounded w-32" />
                      <div className="h-6 bg-muted rounded w-20" />
                    </div>
                  ))}
                </div>
              ) : (
                <div className="space-y-3">
                  <div className="flex items-center justify-between">
                    <span className="text-sm font-medium">Typesense Connection</span>
                    <Badge variant={systemInfo?.typesenseStatus === "connected" ? "default" : "destructive"}>
                      {systemInfo?.typesenseStatus === "connected" ? (
                        <CheckCircle className="h-3 w-3 mr-1" />
                      ) : (
                        <XCircle className="h-3 w-3 mr-1" />
                      )}
                      {systemInfo?.typesenseStatus === "connected" ? "Connected" : "Disconnected"}
                    </Badge>
                  </div>
                  <div className="flex items-center justify-between">
                    <span className="text-sm font-medium">YouTube API</span>
                    <Badge variant={systemInfo?.youtubeStatus === "configured" ? "default" : "secondary"}>
                      {systemInfo?.youtubeStatus === "configured" ? (
                        <CheckCircle className="h-3 w-3 mr-1" />
                      ) : (
                        <XCircle className="h-3 w-3 mr-1" />
                      )}
                      {systemInfo?.youtubeStatus === "configured" ? "Configured" : "Not Configured"}
                    </Badge>
                  </div>
                  <div className="flex items-center justify-between">
                    <span className="text-sm font-medium">Total Collections</span>
                    <Badge variant="outline">{systemInfo?.collections.length || 0}</Badge>
                  </div>
                </div>
              )}
              <Separator />
              <div className="flex space-x-2">
                <Button onClick={testConnection} disabled={testingConnection} variant="outline">
                  <RefreshCw className={`h-4 w-4 mr-2 ${testingConnection ? "animate-spin" : ""}`} />
                  Test Connection
                </Button>
                <Button onClick={fetchSystemInfo} variant="outline">
                  <RefreshCw className="h-4 w-4 mr-2" />
                  Refresh Status
                </Button>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="collections" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center space-x-2">
                <Database className="h-5 w-5" />
                <span>Collection Management</span>
              </CardTitle>
            </CardHeader>
            <CardContent>
              {loading ? (
                <div className="space-y-4">
                  {[...Array(3)].map((_, i) => (
                    <div key={i} className="flex items-center justify-between p-4 border rounded animate-pulse">
                      <div className="space-y-2">
                        <div className="h-4 bg-muted rounded w-32" />
                        <div className="h-3 bg-muted rounded w-24" />
                      </div>
                      <div className="h-8 bg-muted rounded w-20" />
                    </div>
                  ))}
                </div>
              ) : systemInfo?.collections.length === 0 ? (
                <div className="text-center py-8">
                  <Database className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
                  <p className="text-muted-foreground">No collections found</p>
                  <Button onClick={recreateCollections} className="mt-4">
                    Create Default Collections
                  </Button>
                </div>
              ) : (
                <div className="space-y-4">
                  {systemInfo?.collections.map((collection) => (
                    <div key={collection.name} className="flex items-center justify-between p-4 border rounded">
                      <div>
                        <h4 className="font-medium">{collection.name}</h4>
                        <p className="text-sm text-muted-foreground">
                          {collection.numDocuments.toLocaleString()} documents
                        </p>
                      </div>
                      <div className="flex items-center space-x-2">
                        <Badge variant="outline">{new Date(collection.createdAt).toLocaleDateString()}</Badge>
                        <AlertDialog>
                          <AlertDialogTrigger asChild>
                            <Button variant="outline" size="sm" disabled={deletingCollection === collection.name}>
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          </AlertDialogTrigger>
                          <AlertDialogContent>
                            <AlertDialogHeader>
                              <AlertDialogTitle>Delete Collection</AlertDialogTitle>
                              <AlertDialogDescription>
                                Are you sure you want to delete the "{collection.name}" collection? This will
                                permanently delete all {collection.numDocuments} documents. This action cannot be
                                undone.
                              </AlertDialogDescription>
                            </AlertDialogHeader>
                            <AlertDialogFooter>
                              <AlertDialogCancel>Cancel</AlertDialogCancel>
                              <AlertDialogAction onClick={() => deleteCollection(collection.name)}>
                                Delete Collection
                              </AlertDialogAction>
                            </AlertDialogFooter>
                          </AlertDialogContent>
                        </AlertDialog>
                      </div>
                    </div>
                  ))}
                  <Separator />
                  <div className="flex space-x-2">
                    <Button onClick={recreateCollections} variant="outline">
                      <RefreshCw className="h-4 w-4 mr-2" />
                      Recreate Collections
                    </Button>
                  </div>
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="integrations" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center space-x-2">
                <Key className="h-5 w-5" />
                <span>API Configuration</span>
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="space-y-4">
                <div>
                  <h4 className="font-medium mb-2">Typesense Configuration</h4>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <Label htmlFor="typesense-host">Host</Label>
                      <Input
                        id="typesense-host"
                        placeholder="localhost"
                        value={process.env.NEXT_PUBLIC_TYPESENSE_HOST || ""}
                        disabled
                      />
                    </div>
                    <div>
                      <Label htmlFor="typesense-port">Port</Label>
                      <Input
                        id="typesense-port"
                        placeholder="8108"
                        value={process.env.NEXT_PUBLIC_TYPESENSE_PORT || ""}
                        disabled
                      />
                    </div>
                  </div>
                  <p className="text-xs text-muted-foreground mt-2">
                    Configure these values in your environment variables
                  </p>
                </div>

                <Separator />

                <div>
                  <h4 className="font-medium mb-2">YouTube API Configuration</h4>
                  <div>
                    <Label htmlFor="youtube-key">API Key</Label>
                    <Input
                      id="youtube-key"
                      type="password"
                      placeholder="Your YouTube API Key"
                      value={systemInfo?.youtubeStatus === "configured" ? "••••••••••••••••" : ""}
                      disabled
                    />
                  </div>
                  <p className="text-xs text-muted-foreground mt-2">
                    Add YOUTUBE_API_KEY to your environment variables to enable video search
                  </p>
                </div>

                <Separator />

                <div>
                  <h4 className="font-medium mb-2">Search Configuration</h4>
                  <div className="space-y-3">
                    <div className="flex items-center justify-between">
                      <div>
                        <Label htmlFor="safe-search-default">Default Safe Search</Label>
                        <p className="text-xs text-muted-foreground">Enable safe search by default for new results</p>
                      </div>
                      <Switch id="safe-search-default" defaultChecked />
                    </div>
                    <div className="flex items-center justify-between">
                      <div>
                        <Label htmlFor="auto-favicon">Auto-generate Favicons</Label>
                        <p className="text-xs text-muted-foreground">Automatically fetch favicons for new URLs</p>
                      </div>
                      <Switch id="auto-favicon" defaultChecked />
                    </div>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  )
}
