"use client"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Switch } from "@/components/ui/switch"
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { ImageResultsTable } from "@/components/admin/image-results-table"
import { Plus, Upload, Download } from "lucide-react"
import type { ImageSearchResult } from "@/lib/typesense"

export default function ImageManagementPage() {
  const [results, setResults] = useState<ImageSearchResult[]>([])
  const [loading, setLoading] = useState(true)
  const [searchQuery, setSearchQuery] = useState("")
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false)
  const [isBulkUploadOpen, setIsBulkUploadOpen] = useState(false)
  const [addForm, setAddForm] = useState({
    title: "",
    url: "",
    imgurl: "",
    keywords: "",
    safesearch: true,
  })
  const [csvFile, setCsvFile] = useState<File | null>(null)
  const [uploading, setUploading] = useState(false)

  const fetchResults = async () => {
    try {
      const response = await fetch(`/api/admin/image-results?search=${encodeURIComponent(searchQuery)}`)
      if (response.ok) {
        const data = await response.json()
        setResults(data.results || [])
      }
    } catch (error) {
      console.error("Failed to fetch image results:", error)
    } finally {
      setLoading(false)
    }
  }

  useEffect(() => {
    fetchResults()
  }, [searchQuery])

  const handleAdd = async () => {
    try {
      const response = await fetch("/api/admin/image-results", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          ...addForm,
          keywords: addForm.keywords
            .split(",")
            .map((k) => k.trim())
            .filter(Boolean),
        }),
      })

      if (response.ok) {
        setIsAddDialogOpen(false)
        setAddForm({
          title: "",
          url: "",
          imgurl: "",
          keywords: "",
          safesearch: true,
        })
        fetchResults()
      }
    } catch (error) {
      console.error("Failed to add image result:", error)
    }
  }

  const handleBulkUpload = async () => {
    if (!csvFile) return

    setUploading(true)
    const formData = new FormData()
    formData.append("file", csvFile)

    try {
      const response = await fetch("/api/admin/image-results/bulk-upload", {
        method: "POST",
        body: formData,
      })

      if (response.ok) {
        setIsBulkUploadOpen(false)
        setCsvFile(null)
        fetchResults()
      }
    } catch (error) {
      console.error("Failed to upload CSV:", error)
    } finally {
      setUploading(false)
    }
  }

  const downloadTemplate = () => {
    const csvContent =
      "title,url,imgurl,keywords,safesearch\n" +
      'Example Image,https://example.com,https://example.com/image.jpg,"keyword1,keyword2",true'

    const blob = new Blob([csvContent], { type: "text/csv" })
    const url = window.URL.createObjectURL(blob)
    const a = document.createElement("a")
    a.href = url
    a.download = "image-results-template.csv"
    a.click()
    window.URL.revokeObjectURL(url)
  }

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold font-playfair text-foreground">Image Results</h1>
          <p className="text-muted-foreground mt-2">Manage image search results and content</p>
        </div>
        <div className="flex items-center space-x-2">
          <Dialog open={isBulkUploadOpen} onOpenChange={setIsBulkUploadOpen}>
            <DialogTrigger asChild>
              <Button variant="outline">
                <Upload className="h-4 w-4 mr-2" />
                Bulk Upload
              </Button>
            </DialogTrigger>
            <DialogContent>
              <DialogHeader>
                <DialogTitle>Bulk Upload Image Results</DialogTitle>
              </DialogHeader>
              <div className="space-y-4">
                <div>
                  <Label htmlFor="csv-file">CSV File</Label>
                  <Input
                    id="csv-file"
                    type="file"
                    accept=".csv"
                    onChange={(e) => setCsvFile(e.target.files?.[0] || null)}
                  />
                  <p className="text-sm text-muted-foreground mt-1">
                    Upload a CSV file with columns: title, url, imgurl, keywords, safesearch
                  </p>
                </div>
                <Button variant="outline" onClick={downloadTemplate} className="w-full bg-transparent">
                  <Download className="h-4 w-4 mr-2" />
                  Download Template
                </Button>
                <div className="flex justify-end space-x-2">
                  <Button variant="outline" onClick={() => setIsBulkUploadOpen(false)}>
                    Cancel
                  </Button>
                  <Button onClick={handleBulkUpload} disabled={!csvFile || uploading}>
                    {uploading ? "Uploading..." : "Upload"}
                  </Button>
                </div>
              </div>
            </DialogContent>
          </Dialog>
          <Dialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen}>
            <DialogTrigger asChild>
              <Button>
                <Plus className="h-4 w-4 mr-2" />
                Add Result
              </Button>
            </DialogTrigger>
            <DialogContent className="max-w-2xl">
              <DialogHeader>
                <DialogTitle>Add Image Result</DialogTitle>
              </DialogHeader>
              <div className="space-y-4">
                <div>
                  <Label htmlFor="title">Title</Label>
                  <Input
                    id="title"
                    value={addForm.title}
                    onChange={(e) => setAddForm({ ...addForm, title: e.target.value })}
                  />
                </div>
                <div>
                  <Label htmlFor="url">Source URL</Label>
                  <Input
                    id="url"
                    type="url"
                    value={addForm.url}
                    onChange={(e) => setAddForm({ ...addForm, url: e.target.value })}
                  />
                </div>
                <div>
                  <Label htmlFor="imgurl">Image URL</Label>
                  <Input
                    id="imgurl"
                    type="url"
                    value={addForm.imgurl}
                    onChange={(e) => setAddForm({ ...addForm, imgurl: e.target.value })}
                  />
                </div>
                <div>
                  <Label htmlFor="keywords">Keywords (comma-separated)</Label>
                  <Input
                    id="keywords"
                    value={addForm.keywords}
                    onChange={(e) => setAddForm({ ...addForm, keywords: e.target.value })}
                  />
                </div>
                <div className="flex items-center space-x-2">
                  <Switch
                    id="safesearch"
                    checked={addForm.safesearch}
                    onCheckedChange={(checked) => setAddForm({ ...addForm, safesearch: checked })}
                  />
                  <Label htmlFor="safesearch">Safe Search</Label>
                </div>
                <div className="flex justify-end space-x-2">
                  <Button variant="outline" onClick={() => setIsAddDialogOpen(false)}>
                    Cancel
                  </Button>
                  <Button onClick={handleAdd}>Add Result</Button>
                </div>
              </div>
            </DialogContent>
          </Dialog>
        </div>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Image Results</CardTitle>
          <CardDescription>
            <div className="flex items-center space-x-2">
              <Input
                placeholder="Search image results..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="max-w-sm"
              />
            </div>
          </CardDescription>
        </CardHeader>
        <CardContent>
          <ImageResultsTable
            results={results}
            loading={loading}
            onResultUpdated={fetchResults}
            onResultDeleted={fetchResults}
          />
        </CardContent>
      </Card>
    </div>
  )
}
