"use client"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Search, Plus, Edit, Trash2, Eye, EyeOff, Upload, AlertTriangle } from "lucide-react"
import { AddAdDialog } from "@/components/admin/add-ad-dialog"
import { EditAdDialog } from "@/components/admin/edit-ad-dialog"
import { BulkUploadDialog } from "@/components/admin/bulk-upload-dialog"
import { toast } from "sonner" // or your toast library
import type { AdResult } from "@/lib/typesense"

export default function AdsManagementPage() {
  const [ads, setAds] = useState<AdResult[]>([])
  const [loading, setLoading] = useState(true)
  const [searchTerm, setSearchTerm] = useState("")
  const [showAddDialog, setShowAddDialog] = useState(false)
  const [showEditDialog, setShowEditDialog] = useState(false)
  const [showBulkUpload, setShowBulkUpload] = useState(false)
  const [editingAd, setEditingAd] = useState<AdResult | null>(null)
  const [sortBy, setSortBy] = useState<string>("all")

  useEffect(() => {
    fetchAds()
  }, [])

  const fetchAds = async () => {
    try {
      setLoading(true)
      const response = await fetch("/api/admin/ads-results")
      if (response.ok) {
        const data = await response.json()
        setAds(data.ads || [])
      }
    } catch (error) {
      console.error("Error fetching ads:", error)
      toast.error("Failed to fetch ads")
    } finally {
      setLoading(false)
    }
  }

  const handleSearch = async () => {
    if (!searchTerm.trim()) {
      fetchAds()
      return
    }

    try {
      setLoading(true)
      const response = await fetch(`/api/admin/ads-results?search=${encodeURIComponent(searchTerm)}`)
      if (response.ok) {
        const data = await response.json()
        setAds(data.ads || [])
      }
    } catch (error) {
      console.error("Error searching ads:", error)
      toast.error("Failed to search ads")
    } finally {
      setLoading(false)
    }
  }

  const handleDelete = async (id: string) => {
    if (!confirm("Are you sure you want to delete this ad?")) return

    try {
      const response = await fetch(`/api/admin/ads-results/${id}`, {
        method: "DELETE",
      })
      if (response.ok) {
        setAds(ads.filter((ad) => ad.id !== id))
        toast.success("Ad deleted successfully")
      } else {
        toast.error("Failed to delete ad")
      }
    } catch (error) {
      console.error("Error deleting ad:", error)
      toast.error("Failed to delete ad")
    }
  }

  const handleToggleActive = async (id: string, active: boolean) => {
    try {
      const response = await fetch(`/api/admin/ads-results/${id}`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ active: !active }),
      })
      if (response.ok) {
        setAds(ads.map((ad) => (ad.id === id ? { ...ad, active: !active } : ad)))
        toast.success(`Ad ${!active ? "activated" : "deactivated"} successfully`)
      } else {
        toast.error("Failed to update ad status")
      }
    } catch (error) {
      console.error("Error updating ad:", error)
      toast.error("Failed to update ad status")
    }
  }

  const handleEditClick = (ad: AdResult) => {
    setEditingAd(ad)
    setShowEditDialog(true)
  }

  const handleEditComplete = () => {
    setShowEditDialog(false)
    setEditingAd(null)
    fetchAds() // Refresh the ads list
  }

  const getAdStatus = (ad: AdResult) => {
    const dailyLimitReached = (ad.dailyShown || 0) >= ad.dailyLimit
    const monthlyLimitReached = (ad.monthlyShown || 0) >= ad.monthlyLimit

    if (!ad.active) return "inactive"
    if (dailyLimitReached || monthlyLimitReached) return "limit_exceeded"
    return "active"
  }

  const filteredAndSortedAds = ads
    .filter(
      (ad) =>
        ad.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
        ad.description.toLowerCase().includes(searchTerm.toLowerCase()) ||
        ad.keywords.some((keyword) => keyword.toLowerCase().includes(searchTerm.toLowerCase())),
    )
    .sort((a, b) => {
      const aStatus = getAdStatus(a)
      const bStatus = getAdStatus(b)

      switch (sortBy) {
        case "active":
          if (aStatus === "active" && bStatus !== "active") return -1
          if (bStatus === "active" && aStatus !== "active") return 1
          break
        case "inactive":
          if (aStatus === "inactive" && bStatus !== "inactive") return -1
          if (bStatus === "inactive" && aStatus !== "inactive") return 1
          break
        case "limit_exceeded":
          if (aStatus === "limit_exceeded" && bStatus !== "limit_exceeded") return -1
          if (bStatus === "limit_exceeded" && aStatus !== "limit_exceeded") return 1
          break
        case "newest":
          return new Date(b.createdAt || 0).getTime() - new Date(a.createdAt || 0).getTime()
        case "oldest":
          return new Date(a.createdAt || 0).getTime() - new Date(b.createdAt || 0).getTime()
        default:
          return 0
      }
      return 0
    })

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold font-playfair text-foreground">Ads Management</h1>
          <p className="text-muted-foreground">Manage your search engine advertisements</p>
        </div>
        <div className="flex gap-2">
          <Button onClick={() => setShowBulkUpload(true)} variant="outline" size="sm">
            <Upload className="w-4 h-4 mr-2" />
            Bulk Upload
          </Button>
          <Button onClick={() => setShowAddDialog(true)} size="sm">
            <Plus className="w-4 h-4 mr-2" />
            Add Ad
          </Button>
        </div>
      </div>

      {/* Search and Sort */}
      <div className="flex gap-2">
        <div className="relative flex-1">
          <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
          <Input
            placeholder="Search ads by title, description, or keywords..."
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
            onKeyDown={(e) => e.key === "Enter" && handleSearch()}
            className="pl-10"
          />
        </div>
        <Select value={sortBy} onValueChange={setSortBy}>
          <SelectTrigger className="w-48">
            <SelectValue placeholder="Sort by..." />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="all">All Ads</SelectItem>
            <SelectItem value="active">Active First</SelectItem>
            <SelectItem value="inactive">Inactive First</SelectItem>
            <SelectItem value="limit_exceeded">Limit Exceeded First</SelectItem>
            <SelectItem value="newest">Newest First</SelectItem>
            <SelectItem value="oldest">Oldest First</SelectItem>
          </SelectContent>
        </Select>
        <Button onClick={handleSearch} variant="outline">
          Search
        </Button>
        <Button
          onClick={() => {
            setSearchTerm("")
            setSortBy("all")
            fetchAds()
          }}
          variant="outline"
        >
          Reset
        </Button>
      </div>

      {/* Stats */}
      <div className="grid grid-cols-1 md:grid-cols-5 gap-4">
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium">Total Ads</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{ads.length}</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium">Active Ads</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">{ads.filter((ad) => ad.active).length}</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium">Inactive Ads</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-red-600">{ads.filter((ad) => !ad.active).length}</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium">Total Impressions</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{ads.reduce((sum, ad) => sum + (ad.monthlyShown || 0), 0)}</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium">Limit Exceeded</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-orange-600">
              {
                ads.filter((ad) => {
                  const dailyLimitReached = (ad.dailyShown || 0) >= ad.dailyLimit
                  const monthlyLimitReached = (ad.monthlyShown || 0) >= ad.monthlyLimit
                  return dailyLimitReached || monthlyLimitReached
                }).length
              }
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Ads Table */}
      <Card>
        <CardHeader>
          <CardTitle>Advertisements</CardTitle>
          <CardDescription>
            Manage your search engine ads with keyword targeting and limits. Showing {filteredAndSortedAds.length} of{" "}
            {ads.length} ads
          </CardDescription>
        </CardHeader>
        <CardContent>
          {loading ? (
            <div className="flex justify-center py-8">
              <div className="text-muted-foreground">Loading ads...</div>
            </div>
          ) : filteredAndSortedAds.length === 0 ? (
            <div className="text-center py-8">
              <div className="text-muted-foreground">No ads found</div>
              <Button onClick={() => setShowAddDialog(true)} className="mt-4">
                Create your first ad
              </Button>
            </div>
          ) : (
            <div className="space-y-4">
              {filteredAndSortedAds.map((ad) => {
                const dailyLimitReached = (ad.dailyShown || 0) >= ad.dailyLimit
                const monthlyLimitReached = (ad.monthlyShown || 0) >= ad.monthlyLimit
                const hasLimitIssue = dailyLimitReached || monthlyLimitReached

                return (
                  <div
                    key={ad.id}
                    className={`border rounded-lg p-4 space-y-3 ${hasLimitIssue ? "border-orange-200 bg-orange-50/50 dark:border-orange-800 dark:bg-orange-950/20" : ""}`}
                  >
                    <div className="flex items-start justify-between">
                      <div className="flex-1 space-y-2">
                        <div className="flex items-center gap-2">
                          <h3 className="font-semibold text-lg">{ad.title}</h3>
                          <Badge variant={ad.active ? "default" : "secondary"}>
                            {ad.active ? "Active" : "Inactive"}
                          </Badge>
                          {hasLimitIssue && (
                            <Badge
                              variant="destructive"
                              className="bg-orange-100 text-orange-800 border-orange-300 dark:bg-orange-900 dark:text-orange-200"
                            >
                              <AlertTriangle className="w-3 h-3 mr-1" />
                              Limit Exceeded
                            </Badge>
                          )}
                        </div>
                        <p className="text-muted-foreground">{ad.description}</p>
                        <div className="flex flex-wrap gap-1">
                          {ad.keywords.map((keyword, index) => (
                            <Badge key={index} variant="outline" className="text-xs">
                              {keyword}
                            </Badge>
                          ))}
                        </div>
                        <div className="flex items-center gap-4 text-sm">
                          <span className={dailyLimitReached ? "text-orange-600 font-medium" : "text-muted-foreground"}>
                            Daily: {ad.dailyShown || 0}/{ad.dailyLimit}
                            {dailyLimitReached && " ⚠️"}
                          </span>
                          <span
                            className={monthlyLimitReached ? "text-orange-600 font-medium" : "text-muted-foreground"}
                          >
                            Monthly: {ad.monthlyShown || 0}/{ad.monthlyLimit}
                            {monthlyLimitReached && " ⚠️"}
                          </span>
                          <span className="text-muted-foreground">Target: {ad.targetUrl}</span>
                        </div>
                      </div>
                      <div className="flex items-center gap-2 ml-4">
                        {ad.imageUrl && (
                          <img
                            src={ad.imageUrl || "/placeholder.svg"}
                            alt={ad.title}
                            className="w-16 h-16 object-cover rounded border"
                            onError={(e) => {
                              e.currentTarget.src = "/broken-image.png"
                            }}
                          />
                        )}
                        <div className="flex flex-col gap-1">
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={() => handleToggleActive(ad.id, ad.active)}
                            title={ad.active ? "Deactivate ad" : "Activate ad"}
                          >
                            {ad.active ? <EyeOff className="w-4 h-4" /> : <Eye className="w-4 h-4" />}
                          </Button>
                          <Button size="sm" variant="outline" onClick={() => handleEditClick(ad)} title="Edit ad">
                            <Edit className="w-4 h-4" />
                          </Button>
                          <Button size="sm" variant="outline" onClick={() => handleDelete(ad.id)} title="Delete ad">
                            <Trash2 className="w-4 h-4" />
                          </Button>
                        </div>
                      </div>
                    </div>
                  </div>
                )
              })}
            </div>
          )}
        </CardContent>
      </Card>

      {/* Dialogs */}
      <AddAdDialog open={showAddDialog} onOpenChange={setShowAddDialog} onAdAdded={fetchAds} />

      <EditAdDialog
        open={showEditDialog}
        onOpenChange={setShowEditDialog}
        ad={editingAd}
        onAdUpdated={handleEditComplete}
      />

      <BulkUploadDialog open={showBulkUpload} onOpenChange={setShowBulkUpload} type="ads" onUploadComplete={fetchAds} />
    </div>
  )
}
