"use client"

import { useState, useEffect } from "react"
import { useRouter, useSearchParams } from "next/navigation"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { CheckCircle, AlertCircle, ArrowLeft, Mail, Loader2 } from "lucide-react"
import { Alert, AlertDescription } from "@/components/ui/alert"

export default function VerifyEmailPage() {
  const router = useRouter()
  const searchParams = useSearchParams()
  const token = searchParams.get("token")

  const [status, setStatus] = useState<"loading" | "success" | "error">("loading")
  const [message, setMessage] = useState("")

  useEffect(() => {
    if (!token) {
      setStatus("error")
      setMessage("Invalid verification link")
      return
    }

    const verifyEmail = async () => {
      try {
        const response = await fetch(`/api/auth/verify-email?token=${token}`)
        const data = await response.json()

        if (response.ok) {
          setStatus("success")
          setMessage("Your email has been verified successfully!")
        } else {
          setStatus("error")
          setMessage(data.error || "Verification failed")
        }
      } catch (error) {
        setStatus("error")
        setMessage("An error occurred during verification")
      }
    }

    verifyEmail()
  }, [token])

  return (
    <div className="min-h-screen bg-background bg-grid-white flex items-center justify-center p-4 relative overflow-hidden">
      {/* Background Glow Effect */}
      <div className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-[500px] h-[500px] bg-primary/10 rounded-full blur-[120px] pointer-events-none" />

      <div className="w-full max-w-md animate-reveal">
        <Card className="relative border-border bg-card/50 backdrop-blur-xl shadow-2xl overflow-hidden">
          {/* Subtle Top Shine Line */}
          <div className="absolute top-0 left-0 right-0 h-px bg-gradient-to-r from-transparent via-primary/50 to-transparent" />

          <CardHeader className="text-center pt-8">
            <div className="mx-auto mb-6 w-16 h-16 rounded-2xl bg-secondary flex items-center justify-center border border-border shadow-inner rotate-3 hover:rotate-0 transition-transform duration-500">
              {status === "loading" && <Loader2 className="w-8 h-8  animate-spin" />}
              {status === "success" && <CheckCircle className="w-8 h-8 text-green-500" />}
              {status === "error" && <AlertCircle className="w-8 h-8 text-destructive" />}
            </div>

            <CardTitle className="text-2xl font-bold tracking-tight mb-2 ">
              {status === "loading" && "Verifying Security Token"}
              {status === "success" && "Identity Confirmed"}
              {status === "error" && "Verification Failed"}
            </CardTitle>

            <CardDescription className="text-muted-foreground">
              {status === "loading" && "Securing your connection to the dashboard..."}
              {status === "success" && "Your email has been successfully authorized."}
              {status === "error" && "The link may be expired or invalid."}
            </CardDescription>
          </CardHeader>

          <CardContent className="space-y-6 pb-8">
            {message && (
              <div className={`p-4 rounded-lg border text-sm transition-all duration-500 ${status === "success"
                  ? "bg-green-500/10 border-green-500/20 text-green-400"
                  : "bg-destructive/10 border-destructive/20 text-destructive-foreground"
                }`}>
                <p className="flex items-center gap-2">
                  {status === "success" ? <CheckCircle className="w-4 h-4" /> : <AlertCircle className="w-4 h-4" />}
                  {message}
                </p>
              </div>
            )}


            <div className="space-y-3">
              {status === "success" && (
                <Button
                  onClick={() => router.push("/ads-center/auth")}
                  className="relative w-full overflow-hidden bg-primary px-8 py-6 text-primary-foreground transition-all duration-300 hover:ring-2 hover:ring-primary/50 hover:ring-offset-2 hover:ring-offset-background active:scale-[0.98]"
                >
                  {/* Shimmer Effect */}
                  <div className="absolute inset-0 -translate-x-full animate-[shimmer_2s_infinite] bg-gradient-to-r from-transparent via-white/20 to-transparent" />
                  <span className="relative font-semibold tracking-wide">
                    Enter Dashboard
      </span>
                </Button>
              )}

              {status === "error" && (
                <Button
                  onClick={() => window.location.reload()}
                  className="w-full border border-border bg-secondary/50 py-6 backdrop-blur-sm transition-all hover:bg-secondary hover:border-primary/50 active:scale-[0.98]"
                  variant="outline"
                >
                  <span className="font-medium">Request New Link</span>
                </Button>
              )}

              <Button
                variant="ghost"
                onClick={() => router.push("/")}
                className="w-full py-6 text-muted-foreground hover:bg-transparent hover:text-white transition-colors group"
              >
                <ArrowLeft className="w-4 h-4 mr-2 group-hover:-translate-x-1 transition-transform" />
                <span className="text-xs uppercase tracking-widest font-medium">Return to Landing</span>
              </Button>
            </div>

          </CardContent>

          {/* Bottom Accent Decor */}
          <div className="h-1.5 w-full bg-gradient-to-r from-transparent via-primary/20 to-transparent opacity-50" />
        </Card>
      </div>
    </div>
  )
}

// "use client"

// import { useState, useEffect } from "react"
// import { useRouter, useSearchParams } from "next/navigation"
// import { Button } from "@/components/ui/button"
// import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
// import { CheckCircle, AlertCircle, ArrowLeft, Mail } from "lucide-react"
// import { Alert, AlertDescription } from "@/components/ui/alert"

// export default function VerifyEmailPage() {
//   const router = useRouter()
//   const searchParams = useSearchParams()
//   const token = searchParams.get("token")

//   const [status, setStatus] = useState<"loading" | "success" | "error">("loading")
//   const [message, setMessage] = useState("")

//   useEffect(() => {
//     if (!token) {
//       setStatus("error")
//       setMessage("Invalid verification link")
//       return
//     }

//     const verifyEmail = async () => {
//       try {
//         const response = await fetch(`/api/auth/verify-email?token=${token}`)
//         const data = await response.json()

//         if (response.ok) {
//           setStatus("success")
//           setMessage("Your email has been verified successfully!")
//         } else {
//           setStatus("error")
//           setMessage(data.error || "Verification failed")
//         }
//       } catch (error) {
//         setStatus("error")
//         setMessage("An error occurred during verification")
//       }
//     }

//     verifyEmail()
//   }, [token])

//   return (
//     <div className="min-h-screen bg-background flex items-center justify-center p-4">
//       <div className="w-full max-w-md">
//         <Card>
//           <CardHeader className="text-center">
//             <div className="mx-auto mb-4 w-12 h-12 rounded-full bg-primary/10 flex items-center justify-center">
//               {status === "loading" && <Mail className="w-6 h-6 text-primary animate-pulse" />}
//               {status === "success" && <CheckCircle className="w-6 h-6 text-green-600" />}
//               {status === "error" && <AlertCircle className="w-6 h-6 text-red-600" />}
//             </div>
//             <CardTitle>
//               {status === "loading" && "Verifying Email..."}
//               {status === "success" && "Email Verified!"}
//               {status === "error" && "Verification Failed"}
//             </CardTitle>
//             <CardDescription>
//               {status === "loading" && "Please wait while we verify your email address"}
//               {status === "success" && "Your account is now active and ready to use"}
//               {status === "error" && "There was a problem verifying your email"}
//             </CardDescription>
//           </CardHeader>

//           <CardContent className="space-y-4">
//             {message && (
//               <Alert className={status === "success" ? "border-green-200 bg-green-50" : "border-red-200 bg-red-50"}>
//                 <AlertDescription className={status === "success" ? "text-green-800" : "text-red-800"}>
//                   {message}
//                 </AlertDescription>
//               </Alert>
//             )}

//             <div className="space-y-2">
//               {status === "success" && (
//                 <Button onClick={() => router.push("/ads-center/auth")} className="w-full">
//                   Sign In to Your Account
//                 </Button>
//               )}

//               {status === "error" && (
//                 <Button onClick={() => router.push("/ads-center/auth")} variant="outline" className="w-full">
//                   Try Again
//                 </Button>
//               )}

//               <Button variant="ghost" onClick={() => router.push("/")} className="w-full flex items-center gap-2">
//                 <ArrowLeft className="w-4 h-4" />
//                 Back to Home
//               </Button>
//             </div>
//           </CardContent>
//         </Card>
//       </div>
//     </div>
//   )
// }
