"use client"

import type React from "react"
import { useState } from "react"
import { useRouter } from "next/navigation"
import { ArrowLeft, Upload, Check, AlertCircle, Loader2, ShieldCheck, Zap } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Checkbox } from "@/components/ui/checkbox"
import { Card, CardContent } from "@/components/ui/card"
import { Alert, AlertDescription } from "@/components/ui/alert"
import { LogoWithText } from "@/components/logo"
import { useAuth } from "@/components/auth-provider"

export default function SubmitWebsitePage() {
  const router = useRouter()
  const { user, isLoading } = useAuth()
  const [isSubmitting, setIsSubmitting] = useState(false)
  const [submitted, setSubmitted] = useState(false)
  const [error, setError] = useState("")

  const [formData, setFormData] = useState({
    title: "",
    url: "",
    imageUrl: "",
    supportImageCrawl: false,
    supportWebCrawl: true,
    ownerConfirmed: false,
  })

  // Validations
  const validateForm = () => {
    if (!formData.title.trim() || formData.title.length < 2) {
      setError("Please enter a valid website name.")
      return false
    }
    try {
      const url = new URL(formData.url)
      if (!["http:", "https:"].includes(url.protocol)) throw new Error()
    } catch {
      setError("Enter a valid URL (e.g., https://example.com)")
      return false
    }
    if (!formData.ownerConfirmed) {
      setError("Confirmation of ownership is required.")
      return false
    }
    return true
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    if (!validateForm()) return

    setIsSubmitting(true)
    setError("")

    try {
      const response = await fetch("/api/submit-website", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(formData),
      })
      if (!response.ok) throw new Error()
      setSubmitted(true)
    } catch {
      setError("Something went wrong. Please try again later.")
    } finally {
      setIsSubmitting(false)
    }
  }

  if (isLoading) return (
    <div className="min-h-screen bg-background flex items-center justify-center">
      <Loader2 className="w-6 h-6 animate-spin text-primary" />
    </div>
  )

  return (
    <div className="min-h-screen bg-background text-foreground transition-colors duration-500">
      {/* Navbar */}
      <nav className="flex items-center justify-between px-6 py-4 border-b border-border/40 backdrop-blur-md sticky top-0 z-50">
        <div className="flex items-center gap-4">
          <Button 
            variant="ghost" 
            size="icon" 
            onClick={() => router.push("/")}
            className="rounded-full hover:bg-secondary transition-all"
          >
            <ArrowLeft className="w-5 h-5" />
          </Button>
          <LogoWithText size="sm" />
        </div>
        <div className="hidden md:block text-xs font-medium text-muted-foreground tracking-widest uppercase">
          Publisher Dashboard
        </div>
      </nav>

      <main className="max-w-5xl mx-auto px-6 py-12 lg:py-20">
        <div className="grid lg:grid-cols-5 gap-12 items-start">
          
          {/* Left Side: Copy */}
          <div className="lg:col-span-2 space-y-6">
            <div className="inline-flex items-center gap-2 px-3 py-1 rounded-full bg-primary/10 border border-primary/20 text-primary text-[10px] font-bold uppercase tracking-tighter animate-pulse">
              <Zap className="w-3 h-3 fill-current" />
              Instant Indexing
            </div>
            <h1 className="text-4xl md:text-5xl font-bold tracking-tight leading-none">
              Grow your <span className="text-muted-foreground">Digital Presence.</span>
            </h1>
            <p className="text-muted-foreground leading-relaxed">
              Submit your project to the IndusSearch index. We prioritize high-quality, developer-focused content.
            </p>
            
            <div className="space-y-4 pt-4">
              <div className="flex items-start gap-3 group">
                <div className="mt-1 p-1 rounded-md bg-secondary border border-border group-hover:border-primary/50 transition-colors">
                  <ShieldCheck className="w-4 h-4 text-muted-foreground" />
                </div>
                <div>
                  <h4 className="text-sm font-semibold">Secure Verification</h4>
                  <p className="text-xs text-muted-foreground">Automatic ownership check via meta tags.</p>
                </div>
              </div>
            </div>
          </div>

          {/* Right Side: Form */}
          <div className="lg:col-span-3">
            <Card className="relative overflow-hidden border-border/60 bg-card/50 shadow-sm transition-all duration-300 hover:shadow-md">
              {/* Subtle top glow line */}
              <div className="absolute top-0 left-0 right-0 h-[1px] bg-gradient-to-r from-transparent via-primary/50 to-transparent" />
              
              <CardContent className="p-8">
                {submitted ? (
                  <div className="py-12 text-center animate-in fade-in zoom-in duration-500">
                    <div className="w-16 h-16 bg-primary text-primary-foreground rounded-2xl flex items-center justify-center mx-auto mb-6 shadow-lg shadow-primary/20">
                      <Check className="w-8 h-8" />
                    </div>
                    <h2 className="text-2xl font-bold mb-2">Success!</h2>
                    <p className="text-muted-foreground mb-8">Your website has been queued for review.</p>
                    <Button onClick={() => router.push("/")} className="w-full">Go Home</Button>
                  </div>
                ) : (
                  <form onSubmit={handleSubmit} className="space-y-6">
                    {error && (
                      <Alert variant="destructive" className="animate-in slide-in-from-top-2 duration-300">
                        <AlertCircle className="w-4 h-4" />
                        <AlertDescription>{error}</AlertDescription>
                      </Alert>
                    )}

                    <div className="grid md:grid-cols-2 gap-4">
                      <div className="space-y-2">
                        <Label htmlFor="title" className="text-[10px] uppercase tracking-widest font-bold text-muted-foreground">Project Name</Label>
                        <Input 
                          id="title" 
                          placeholder="Acme Inc" 
                          className="bg-background border-border/40 focus:border-primary transition-all"
                          value={formData.title}
                          onChange={(e) => setFormData({...formData, title: e.target.value})}
                        />
                      </div>
                      <div className="space-y-2">
                        <Label htmlFor="url" className="text-[10px] uppercase tracking-widest font-bold text-muted-foreground">Website URL</Label>
                        <Input 
                          id="url" 
                          placeholder="https://..." 
                          className="bg-background border-border/40 focus:border-primary transition-all"
                          value={formData.url}
                          onChange={(e) => setFormData({...formData, url: e.target.value})}
                        />
                      </div>
                    </div>

                    <div className="space-y-2">
                      <Label htmlFor="imageUrl" className="text-[10px] uppercase tracking-widest font-bold text-muted-foreground">Preview Image URL (Optional)</Label>
                      <Input 
                        id="imageUrl" 
                        placeholder="https://.../og-image.png" 
                        className="bg-background border-border/40 focus:border-primary transition-all"
                        value={formData.imageUrl}
                        onChange={(e) => setFormData({...formData, imageUrl: e.target.value})}
                      />
                    </div>

                    <div className="pt-4 space-y-4">
                      <p className="text-[10px] uppercase tracking-widest font-bold text-muted-foreground">Permissions</p>
                      <div className="grid gap-3">
                        <label className="flex items-center justify-between p-4 rounded-lg border border-border/40 bg-background/20 hover:bg-secondary/40 transition-all cursor-pointer group">
                          <span className="text-sm font-medium">Allow Image Indexing</span>
                          <Checkbox 
                            checked={formData.supportImageCrawl}
                            onCheckedChange={(val) => setFormData({...formData, supportImageCrawl: !!val})}
                          />
                        </label>
                        <label className="flex items-center justify-between p-4 rounded-lg border border-border/40 bg-background/20 hover:bg-secondary/40 transition-all cursor-pointer group">
                          <span className="text-sm font-medium">Automatic Web Crawl</span>
                          <Checkbox 
                            checked={formData.supportWebCrawl}
                            onCheckedChange={(val) => setFormData({...formData, supportWebCrawl: !!val})}
                          />
                        </label>
                      </div>
                    </div>

                    <div className="flex items-start gap-3 p-3 rounded-md bg-secondary/30 border border-border/20 mt-6">
                      <Checkbox 
                        id="ownerConfirmed" 
                        className="mt-1"
                        onCheckedChange={(val) => setFormData({...formData, ownerConfirmed: !!val})}
                      />
                      <Label htmlFor="ownerConfirmed" className="text-xs leading-relaxed text-muted-foreground select-none">
                        I am the owner of this domain and agree to the Terms of Service.
                      </Label>
                    </div>

                    <Button 
                      type="submit" 
                      disabled={isSubmitting}
                      className="w-full h-12 text-sm font-bold uppercase tracking-widest transition-all active:scale-[0.98] shadow-lg shadow-primary/10 hover:shadow-primary/20"
                    >
                      {isSubmitting ? <Loader2 className="w-4 h-4 animate-spin" /> : "Submit for Indexing"}
                    </Button>
                  </form>
                )}
              </CardContent>
            </Card>
          </div>
        </div>
      </main>
    </div>
  )
}