"use client"

import { useState, useEffect } from "react"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Slider } from "@/components/ui/slider"
import { Switch } from "@/components/ui/switch"
import { Separator } from "@/components/ui/separator"
import { Badge } from "@/components/ui/badge"
import { ArrowLeft, Settings, Palette, Search, Monitor, Moon, Sun, Save } from "lucide-react"
import { useRouter } from "next/navigation"
import { useTheme } from "@/components/theme-provider"
import { LogoWithText } from "@/components/logo"

interface UserSettings {
  theme: "light" | "dark" | "system"
  maxResults: number
  safeSearch: boolean
  autoComplete: boolean
  language: string
  region: string
}

export default function SettingsPage() {
  const router = useRouter()
  const { theme, setTheme } = useTheme()
  const [settings, setSettings] = useState<UserSettings>({
    theme: "system",
    maxResults: 10,
    safeSearch: true,
    autoComplete: true,
    language: "en",
    region: "global",
  })
  const [saving, setSaving] = useState(false)
  const [mounted, setMounted] = useState(false)

  useEffect(() => {
    setMounted(true)
    // Load settings from localStorage
    const savedSettings = localStorage.getItem("indus-search-settings")
    if (savedSettings) {
      const parsed = JSON.parse(savedSettings)
      setSettings(parsed)
    }
  }, [])

  const saveSettings = async () => {
    setSaving(true)
    try {
      // Save to localStorage
      localStorage.setItem("indus-search-settings", JSON.stringify(settings))

      // Apply theme change immediately
      setTheme(settings.theme)

      // Simulate API call delay
      await new Promise((resolve) => setTimeout(resolve, 500))
    } catch (error) {
      console.error("Failed to save settings:", error)
    } finally {
      setSaving(false)
    }
  }

  const resetSettings = () => {
    const defaultSettings: UserSettings = {
      theme: "system",
      maxResults: 10,
      safeSearch: true,
      autoComplete: true,
      language: "en",
      region: "global",
    }
    setSettings(defaultSettings)
    localStorage.removeItem("indus-search-settings")
    setTheme("system")
  }

  const getThemeIcon = (themeValue: string) => {
    switch (themeValue) {
      case "light":
        return <Sun className="h-4 w-4" />
      case "dark":
        return <Moon className="h-4 w-4" />
      default:
        return <Monitor className="h-4 w-4" />
    }
  }

  if (!mounted) {
    return (
      <div className="min-h-screen bg-background flex items-center justify-center">
        <div className="animate-pulse">
          <div className="h-8 w-32 bg-muted rounded mb-4" />
          <div className="h-4 w-48 bg-muted rounded" />
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <header className="border-b border-border bg-background/80 backdrop-blur-sm sticky top-0 z-50">
        <div className="container mx-auto px-4 py-4 flex items-center justify-between">
          <div className="flex items-center gap-4">
            <Button variant="ghost" size="sm" onClick={() => router.push("/")} className="flex items-center gap-2">
              <ArrowLeft className="h-4 w-4" />
              Back to Search
            </Button>
            <Separator orientation="vertical" className="h-6" />
            <LogoWithText size="sm" />
          </div>
          <div className="flex items-center gap-2">
            <Badge variant="outline" className="flex items-center gap-1">
              <Settings className="h-3 w-3" />
              Settings
            </Badge>
          </div>
        </div>
      </header>

      {/* Main Content */}
      <main className="container mx-auto px-4 py-8 max-w-4xl">
        <div className="space-y-6">
          <div>
            <h1 className="text-3xl font-bold text-foreground">Search Settings</h1>
            <p className="text-muted-foreground mt-2">
              Customize your IndusSearch experience with personalized preferences
            </p>
          </div>

          <div className="grid gap-6 md:grid-cols-2">
            {/* Theme Settings */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Palette className="h-5 w-5" />
                  Appearance
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="theme-select">Theme</Label>
                  <Select
                    value={settings.theme}
                    onValueChange={(value: "light" | "dark" | "system") =>
                      setSettings((prev) => ({ ...prev, theme: value }))
                    }
                  >
                    <SelectTrigger className="mt-2">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="light">
                        <div className="flex items-center gap-2">
                          <Sun className="h-4 w-4" />
                          Light
                        </div>
                      </SelectItem>
                      <SelectItem value="dark">
                        <div className="flex items-center gap-2">
                          <Moon className="h-4 w-4" />
                          Dark
                        </div>
                      </SelectItem>
                      <SelectItem value="system">
                        <div className="flex items-center gap-2">
                          <Monitor className="h-4 w-4" />
                          System
                        </div>
                      </SelectItem>
                    </SelectContent>
                  </Select>
                  <p className="text-xs text-muted-foreground mt-1">Choose your preferred color scheme</p>
                </div>

                <div className="flex items-center justify-between">
                  <div>
                    <Label htmlFor="auto-complete">Auto-complete</Label>
                    <p className="text-xs text-muted-foreground">Show search suggestions as you type</p>
                  </div>
                  <Switch
                    id="auto-complete"
                    checked={settings.autoComplete}
                    onCheckedChange={(checked) => setSettings((prev) => ({ ...prev, autoComplete: checked }))}
                  />
                </div>
              </CardContent>
            </Card>

            {/* Search Settings */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Search className="h-5 w-5" />
                  Search Preferences
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="max-results">Results per page: {settings.maxResults}</Label>
                  <Slider
                    id="max-results"
                    min={5}
                    max={50}
                    step={5}
                    value={[settings.maxResults]}
                    onValueChange={(value) => setSettings((prev) => ({ ...prev, maxResults: value[0] }))}
                    className="mt-2"
                  />
                  <p className="text-xs text-muted-foreground mt-1">Number of search results to display per page</p>
                </div>

                <div className="flex items-center justify-between">
                  <div>
                    <Label htmlFor="safe-search">Safe Search</Label>
                    <p className="text-xs text-muted-foreground">Filter explicit content from results</p>
                  </div>
                  <Switch
                    id="safe-search"
                    checked={settings.safeSearch}
                    onCheckedChange={(checked) => setSettings((prev) => ({ ...prev, safeSearch: checked }))}
                  />
                </div>
              </CardContent>
            </Card>

            {/* Language & Region */}
            <Card>
              <CardHeader>
                <CardTitle>Language & Region</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="language">Language</Label>
                  <Select
                    value={settings.language}
                    onValueChange={(value) => setSettings((prev) => ({ ...prev, language: value }))}
                  >
                    <SelectTrigger className="mt-2">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="en">English</SelectItem>
                      <SelectItem value="es">Español</SelectItem>
                      <SelectItem value="fr">Français</SelectItem>
                      <SelectItem value="de">Deutsch</SelectItem>
                      <SelectItem value="hi">हिन्दी</SelectItem>
                      <SelectItem value="zh">中文</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <Label htmlFor="region">Region</Label>
                  <Select
                    value={settings.region}
                    onValueChange={(value) => setSettings((prev) => ({ ...prev, region: value }))}
                  >
                    <SelectTrigger className="mt-2">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="global">Global</SelectItem>
                      <SelectItem value="us">United States</SelectItem>
                      <SelectItem value="uk">United Kingdom</SelectItem>
                      <SelectItem value="in">India</SelectItem>
                      <SelectItem value="ca">Canada</SelectItem>
                      <SelectItem value="au">Australia</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </CardContent>
            </Card>

            {/* Current Settings Preview */}
            <Card>
              <CardHeader>
                <CardTitle>Current Settings</CardTitle>
              </CardHeader>
              <CardContent className="space-y-3">
                <div className="flex items-center justify-between text-sm">
                  <span className="text-muted-foreground">Theme:</span>
                  <div className="flex items-center gap-1">
                    {getThemeIcon(settings.theme)}
                    <span className="capitalize">{settings.theme}</span>
                  </div>
                </div>
                <div className="flex items-center justify-between text-sm">
                  <span className="text-muted-foreground">Results per page:</span>
                  <span>{settings.maxResults}</span>
                </div>
                <div className="flex items-center justify-between text-sm">
                  <span className="text-muted-foreground">Safe Search:</span>
                  <Badge variant={settings.safeSearch ? "default" : "secondary"}>
                    {settings.safeSearch ? "On" : "Off"}
                  </Badge>
                </div>
                <div className="flex items-center justify-between text-sm">
                  <span className="text-muted-foreground">Auto-complete:</span>
                  <Badge variant={settings.autoComplete ? "default" : "secondary"}>
                    {settings.autoComplete ? "On" : "Off"}
                  </Badge>
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Action Buttons */}
          <div className="flex justify-between items-center pt-6 border-t border-border">
            <Button variant="outline" onClick={resetSettings}>
              Reset to Defaults
            </Button>
            <div className="flex gap-2">
              <Button variant="outline" onClick={() => router.push("/")}>
                Cancel
              </Button>
              <Button onClick={saveSettings} disabled={saving}>
                {saving ? (
                  <div className="flex items-center gap-2">
                    <div className="h-4 w-4 animate-spin rounded-full border-2 border-current border-t-transparent" />
                    Saving...
                  </div>
                ) : (
                  <div className="flex items-center gap-2">
                    <Save className="h-4 w-4" />
                    Save Settings
                  </div>
                )}
              </Button>
            </div>
          </div>
        </div>
      </main>
    </div>
  )
}
