"use client";

import { useState, useRef, useEffect } from "react";
import { Search, Command, CornerDownLeft } from "lucide-react";
import { SETTINGS_CONFIG } from "./settings.config";
import { useSettings } from "@/context/SettingsContext";

export default function SettingsSearch() {
  const [query, setQuery] = useState("");
  const [isFocused, setIsFocused] = useState(false);
  const { setTab } = useSettings();
  const containerRef = useRef<HTMLDivElement>(null);

  // Filter logic
  const results = SETTINGS_CONFIG.filter((item) =>
    // `${item.title} ${item.description}`
    //   .toLowerCase()
    //   .includes(query.toLowerCase())
      [
      item.title,
      item.description,
      ...item.keywords,
      item.id,
    ].some((text) => text.toLowerCase().includes(query.toLowerCase()))
  );


  // Close dropdown when clicking outside
  useEffect(() => {
    const handleClickOutside = (e: MouseEvent) => {
      if (containerRef.current && !containerRef.current.contains(e.target as Node)) {
        setIsFocused(false);
      }
    };
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);

  return (
    <div ref={containerRef} className="relative w-full group mb-10  rounded-xl outline-none transition-all"
    >
      {/* Search Input Container */}
      <div className="relative flex items-center w-full">
        <Search className={`absolute left-3 w-4 h-4 transition-colors duration-200 ${isFocused ? 'text-primary' : 'text-muted-foreground'}`} />
        
        <input
          value={query}
          onFocus={() => setIsFocused(true)}
          onChange={(e) => setQuery(e.target.value)}
          placeholder="Search settings..."
          className="w-full pl-10 pr-12 py-2.5 bg-secondary/30 border border-border/50 rounded-xl text-sm transition-all duration-200 outline-none focus:ring-2 focus:ring-ring/20 focus:border-primary focus:bg-background shadow-sm placeholder:text-muted-foreground/50"
        style={{
                backgroundColor: 'var(--input)',
                color: 'var(--foreground)',
                // border: '1px solid var(--border)'
              }}
        
        />

        {/* Command K style Hint */}
        <div className="absolute right-3 hidden sm:flex items-center gap-1 px-1.5 py-0.5 rounded border border-border/50 bg-muted/50 text-[10px] text-muted-foreground font-medium uppercase tracking-widest">
          <Command className="w-2.5 h-2.5" /> K
        </div>
      </div>

      {/* Results Dropdown */}
      {isFocused && query && (
        <div className="absolute mt-2 w-full bg-popover/95 backdrop-blur-xl border border-border shadow-2xl rounded-xl z-50 overflow-hidden animate-in fade-in zoom-in-95 duration-200 origin-top">
          <div className="p-2 max-h-[320px] overflow-y-auto overflow-x-hidden custom-scrollbar">
            
            <div className="px-2 py-1.5 mb-1 text-[10px] font-bold uppercase tracking-[0.2em] text-muted-foreground/70">
              Settings Suggestions
            </div>

            {results.length === 0 ? (
              <div className="px-4 py-8 text-center">
                <p className="text-sm text-muted-foreground">No matches for <span className="text-foreground font-medium">"{query}"</span></p>
              </div>
            ) : (
              results.map((item) => (
                <button
                  key={item.id}
                  onClick={() => {
                    setTab(item.id);
                    setQuery("");
                    setIsFocused(false);
                  }}
                  className="group flex items-center justify-between w-full px-3 py-2.5 rounded-lg text-left transition-all duration-150 hover:bg-accent hover:text-accent-foreground"
                >
                  <div className="flex items-center gap-3">
                    <div className="p-2 rounded-md bg-background border border-border/50 group-hover:border-primary/30 group-hover:text-primary transition-colors">
                      <item.icon className="w-4 h-4" />
                    </div>
                    <div>
                      <div className="text-sm font-medium leading-none">{item.title}</div>
                      <div className="mt-1 text-xs text-muted-foreground line-clamp-1 group-hover:text-accent-foreground/70">
                        {item.description}
                      </div>
                    </div>
                  </div>
                  
                  {/* Subtle Enter Hint */}
                  <CornerDownLeft className="w-3 h-3 text-muted-foreground/0 group-hover:text-muted-foreground/50 transition-opacity mr-2" />
                </button>
              ))
            )}
          </div>

          {/* Footer Info */}
          <div className="px-4 py-2 bg-secondary/20 border-t border-border/40 flex justify-between items-center">
            <span className="text-[10px] text-muted-foreground/60 italic">Search indexed by IndusSearch</span>
            <div className="flex gap-2">
              <span className="text-[10px] text-muted-foreground/60">↑↓ to navigate</span>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}