"use client"
import React, { useState, useEffect } from "react";
import { ShieldCheck, ListOrdered, Sparkles } from "lucide-react";
import "@/styles/settings.css"
interface ToggleProps {
  enabled: boolean;
  setEnabled: (value: boolean) => void;
}

export default function SearchPreferences() {
  const [results, setResults] = useState(50);
  const [safeSearch, setSafeSearch] = useState(true);

  // Load initial values
  useEffect(() => {
    const saved = localStorage.getItem("indus-search-settings");
    if (saved) {
      const parsed = JSON.parse(saved);
      if (parsed.maxResults) setResults(parsed.maxResults);
      if (parsed.safeSearch !== undefined) setSafeSearch(parsed.safeSearch);
    }
  }, []);

  // Sync to localStorage whenever state changes
  useEffect(() => {
    const saved = localStorage.getItem("indus-search-settings");
    const currentSettings = saved ? JSON.parse(saved) : {};

    const updatedSettings = {
      ...currentSettings,
      maxResults: results,
      safeSearch: safeSearch,
    };

    localStorage.setItem("indus-search-settings", JSON.stringify(updatedSettings));
    
    // Crucial: Tell the "CurrentSettings" card to update its UI
    window.dispatchEvent(new Event("settingsUpdated"));
  }, [results, safeSearch]);

  return (
    <div className="max-w-3xl mx-auto animate-in fade-in slide-in-from-bottom-4 duration-700">
      <div
        className="rounded-2xl border transition-all duration-300 overflow-hidden"
        style={{
          backgroundColor: 'var(--card)',
          borderColor: 'var(--border)',
          color: 'var(--card-foreground)'
        }}
      >
        <div className="p-8 border-b" style={{ borderColor: 'var(--border)' }}>
          <h2 className="text-xl font-bold tracking-tight flex items-center gap-2">
            <Sparkles size={20} style={{ color: 'var(--chart-3)' }} />
            Search Preferences
          </h2>
          <p className="text-sm mt-1" style={{ color: 'var(--muted-foreground)' }}>
            Configure how you explore and interact with search results.
          </p>
        </div>

        <div className="p-8 space-y-10">
          {/* Slider Section */}
          <div className="space-y-4">
            <div className="flex justify-between items-end">
              <div className="space-y-1">
                <label className="text-sm font-semibold flex items-center gap-2">
                  <ListOrdered size={16} className="opacity-60" />
                  Results per page
                </label>
                <p className="text-xs opacity-50">Higher counts may slow down loading times.</p>
              </div>
              <span className="text-2xl font-mono font-bold" style={{ color: 'var(--primary)' }}>
                {results}
              </span>
            </div>

            <div className="relative flex items-center group">
              <input
                type="range"
                min={10}
                max={100}
                step={10}
                value={results}
                onChange={(e) => setResults(+e.target.value)}
                className="w-full h-1.5 rounded-lg appearance-none cursor-pointer transition-all focus:outline-none"
                style={{
                  backgroundColor: 'var(--secondary)',
                  backgroundImage: `linear-gradient(to right, var(--primary) ${(results - 10) / 0.9}%, transparent 0%)`
                }}
              />
              {/* Custom Track CSS via style tag below */}
            </div>
            <div className="flex justify-between text-[10px] uppercase tracking-widest font-bold opacity-30">
              <span>10</span>
              <span>100</span>
            </div>
          </div>

          {/* Toggle Section */}
          <div
            className="flex flex-col lg:flex-row lg:items-center p-5 rounded-2xl border transition-all gap-4 lg:gap-6"
            style={{ backgroundColor: 'var(--sidebar)', borderColor: 'var(--border)' }}
          >
            {/* 1. Shield Icon - Centered on mobile, aligned left on desktop */}
            <div className="flex lg:block shrink-0">
              <div className="w-12 h-12 rounded-xl flex items-center justify-center transition-transform hover:rotate-12"
                style={{ backgroundColor: 'var(--accent)', color: 'var(--primary)' }}>
                <ShieldCheck size={24} style={{ color: '#fff' }} />
              </div>
            </div>

            {/* 2. Content Wrapper - Row layout on mobile text/toggle, integrated on desktop */}
            <div className="flex flex-row items-center justify-between lg:justify-start w-full gap-4 lg:gap-8">
              <div className="space-y-0.5 lg:flex-1">
                <p className="text-sm font-bold leading-none">Safe Search</p>
                <p className="text-xs opacity-60 mt-1" style={{ color: 'var(--muted-foreground)' }}>
                  Filter explicit content from your results
      </p>
              </div>

              {/* Toggle - Stays on the same line as text in mobile, end of row in desktop */}
              <div className="flex items-center shrink-0">
                <Toggle enabled={safeSearch} setEnabled={setSafeSearch} />
              </div>
            </div>
          </div>
        </div>
      </div>

      <style jsx>{`
        input[type='range']::-webkit-slider-thumb {
          appearance: none;
          height: 18px;
          width: 18px;
          border-radius: 50%;
          background: var(--primary);
          cursor: pointer;
          border: 3px solid var(--card);
          box-shadow: 0 0 10px rgba(0,0,0,0.5);
          transition: all 0.2s ease-in-out;
        }
        input[type='range']::-webkit-slider-thumb:hover {
          transform: scale(1.2);
          box-shadow: 0 0 15px var(--primary);
        }
      `}</style>
    </div>
  )
}

function Toggle({ enabled, setEnabled }: ToggleProps) {
  return (
    <button
      onClick={() => setEnabled(!enabled)}
      className="relative w-14 h-7 rounded-full transition-all duration-300 focus:outline-none   .custom-toggle"
      style={{
        backgroundColor: enabled ? 'var(--primary)' : 'var(--input)',
        // borderColor: 'var(--border)',
        borderWidth: '1px',
        // ringColor: 'var(--ring)'
      }}
    >
      <div
        className={`absolute top-1 left-1 w-5 h-5 rounded-full transition-transform duration-300 shadow-sm`}
        style={{
          backgroundColor: enabled ? 'var(--primary-foreground)' : 'var(--muted-foreground)',
          transform: enabled ? 'translateX(28px)' : 'translateX(0px)'
        }}
      />
    </button>
  )
}