"use client"
import React, { useState, useEffect, useMemo } from "react";
import { Search, Clock, Trash2, History, RotateCcw, Globe, ExternalLink } from "lucide-react";

import { useDebounce } from "@/hooks/useDebounce.ts";
import { useHistory } from "@/hooks/useHistory"

export default function SearchHistory() {
    const [activeTab, setActiveTab] = useState("recent"); // "recent" or "browser"
    const [searchQuery, setSearchQuery] = useState("");
    const { getSearchHistory, removeSearchHistory, clearSearchHistory } = useHistory()
    
    const [recentSearches, setRecentSearches] = useState([]);
    const [browserHistory, setBrowserHistory] = useState([]);

    const debouncedQuery = useDebounce(searchQuery, 300);

    const loadAllHistory = () => {
        setRecentSearches(getSearchHistory());
        const rawBrowser = localStorage.getItem("__browser_history__");
        if (rawBrowser) {
            try {
                setBrowserHistory(JSON.parse(rawBrowser));
            } catch (e) {
                console.error("Failed to parse browser history");
            }
        }
    };

    useEffect(() => {
        loadAllHistory();
    }, []);

    const filteredRecent = useMemo(() => {
        if (!debouncedQuery.trim()) return recentSearches;
        return recentSearches.filter(item => item.query.toLowerCase().includes(debouncedQuery.toLowerCase()));
    }, [debouncedQuery, recentSearches]);

    const filteredBrowser = useMemo(() => {
        if (!debouncedQuery.trim()) return browserHistory;
        return browserHistory.filter(item => 
            item.title.toLowerCase().includes(debouncedQuery.toLowerCase()) || 
            item.url.toLowerCase().includes(debouncedQuery.toLowerCase())
        );
    }, [debouncedQuery, browserHistory]);

    const deleteRecentItem = (id) => {
        removeSearchHistory(id);
        setRecentSearches(prev => prev.filter(item => item.id !== id));
    };

    const deleteBrowserItem = (id) => {
        const updated = browserHistory.filter(item => item.id !== id);
        setBrowserHistory(updated);
        localStorage.setItem("__browser_history__", JSON.stringify(updated));
    };

    const clearAll = () => {
        if (activeTab === "recent") {
            clearSearchHistory();
            setRecentSearches([]);
        } else {
            setBrowserHistory([]);
            localStorage.removeItem("__browser_history__");
        }
    };

    return (
        <div className="w-full mx-auto animate-in fade-in slide-in-from-bottom-4 duration-700">
            <div
                className="rounded-[2rem] border shadow-2xl overflow-hidden"
                style={{ backgroundColor: 'var(--card)', borderColor: 'var(--border)' }}
            >
                <div className="p-6">
                    <form onSubmit={(e) => e.preventDefault()} className="relative group mb-6">
                        <Search className="absolute left-5 top-1/2 -translate-y-1/2 opacity-30 group-focus-within:opacity-100" size={20} />
                        <input
                            type="text"
                            value={searchQuery}
                            onChange={(e) => setSearchQuery(e.target.value)}
                            placeholder={activeTab === "recent" ? "Search activity..." : "Search history..."}
                            className="w-full py-4 pl-14 pr-6 rounded-2xl border-2 outline-none transition-all text-lg"
                            style={{ backgroundColor: 'var(--input)', borderColor: 'var(--border)', color: 'var(--foreground)' }}
                        />
                    </form>

                    {/* Tab Switcher - */}
                    <div className="flex p-1.5 rounded-2xl bg-white/5 border border-white/5 mb-4">
                        <button 
                            onClick={() => setActiveTab("recent")}
                            className="flex-1 py-2.5 text-[10px] font-bold uppercase tracking-widest rounded-xl transition-all"
                            style={{ 
                                backgroundColor: activeTab === 'recent' ? 'var(--secondary)' : 'transparent',
                                color: activeTab === 'recent' ? 'var(--secondary-foreground)' : 'var(--foreground)',
                                opacity: activeTab === 'recent' ? 1 : 0.4
                            }}
                        >
                            Recent
                        </button>
                        <button 
                            onClick={() => setActiveTab("browser")}
                            className="flex-1 py-2.5 text-[10px] font-bold uppercase tracking-widest rounded-xl transition-all"
                            style={{ 
                                backgroundColor: activeTab === 'browser' ? 'var(--secondary)' : 'transparent',
                                color: activeTab === 'browser' ? 'var(--secondary-foreground)' : 'var(--foreground)',
                                opacity: activeTab === 'browser' ? 1 : 0.4
                            }}
                        >
                            Browser
                        </button>
                    </div>
                </div>

                <div className="px-6 pb-6">
                    <div className="flex items-center justify-between mb-4 px-2">
                        <div className="flex items-center gap-2 opacity-60">
                            {activeTab === "recent" ? <History size={14} /> : <Globe size={14} />}
                            <span className="text-[10px] font-bold uppercase tracking-[0.2em]">
                                {activeTab === "recent" ? "Searches" : "History"}
                            </span>
                        </div>
                        {(activeTab === "recent" ? recentSearches : browserHistory).length > 0 && (
                            <button onClick={clearAll} className="text-[10px] font-bold uppercase tracking-widest text-red-500/80 hover:text-red-500">
                                Clear All
                            </button>
                        )}
                    </div>

                    <div className="space-y-2 search-scroll overflow-y-auto" style={{ height: "calc(100vh - 480px)" }}>
                        {activeTab === "recent" ? (
                            filteredRecent.length === 0 ? <EmptyState /> : (
                                filteredRecent.map((item) => (
                                    <HistoryItem 
                                        key={item.id} 
                                        title={item.query} 
                                        onDelete={() => deleteRecentItem(item.id)}
                                        icon={<RotateCcw size={16} />}
                                    />
                                ))
                            )
                        ) : (
                            filteredBrowser.length === 0 ? <EmptyState /> : (
                                filteredBrowser.map((item) => (
                                    <HistoryItem 
                                        key={item.id} 
                                        title={item.title} 
                                        subtitle={item.url}
                                        onDelete={() => deleteBrowserItem(item.id)}
                                        onVisit={() => window.open(item.url, '_blank')}
                                        icon={<img src={`https://www.google.com/s2/favicons?domain=${new URL(item.url).hostname}&sz=32`} className="w-4 h-4 rounded-sm" alt="" />}
                                    />
                                ))
                            )
                        )}
                    </div>
                </div>

                <div className="p-6 pt-0 opacity-20 text-center">
                    <p className="text-[10px] font-black tracking-widest uppercase">Privacy Protected</p>
                </div>
            </div>
        </div>
    );
}

const HistoryItem = ({ title, subtitle, onDelete, onVisit, icon }) => (
    <div className="group flex items-center justify-between p-3 rounded-2xl border border-white/5 bg-white/[0.02] active:scale-[0.99] transition-all">
        <div className="flex items-center gap-3 min-w-0 flex-1">
            <div className="p-2 rounded-lg bg-white/5 shrink-0">
                {icon}
            </div>
            <div className="min-w-0 flex-1">
                <p className="text-sm font-medium truncate opacity-90">{title}</p>
                {subtitle && <p className="text-[10px] truncate opacity-100 font-mono mt-0.5">{subtitle}</p>}
            </div>
        </div>
        
        {/* Buttons: hidden by default on desktop (hover shows), always visible on small screens */}
        <div className="flex items-center gap-1 ml-2 md:opacity-0 md:group-hover:opacity-100 transition-opacity shrink-0">
            {onVisit && (
                <button onClick={onVisit} className="p-2.5 rounded-xl bg-white/5 md:bg-transparent text-blue-400">
                    <ExternalLink size={16} />
                </button>
            )}
            <button 
                onClick={(e) => { e.stopPropagation(); onDelete(); }} 
                className="p-2.5 rounded-xl bg-white/5 md:bg-transparent text-red-500/70 hover:text-red-500"
            >
                <Trash2 size={16} />
            </button>
        </div>
    </div>
);

const EmptyState = () => (
    <div className="py-12 text-center opacity-30">
        <Clock className="mx-auto mb-3" size={32} />
        <p className="text-sm font-medium">No records found</p>
    </div>
);



// "use client"
// import React, { useState, useEffect, useMemo } from "react";
// import { Search, Clock, Trash2, X, History, RotateCcw } from "lucide-react";

// import { useSearchHistory } from "@/hooks/useSearchHistory"
// import { useDebounce } from "@/hooks/useDebounce.ts";
// import { useHistory } from "@/hooks/useHistory"
// export default function SearchHistory() {
//     const [searchQuery, setSearchQuery] = useState("");
//     const { getSearchHistory, updateSearchState, removeSearchHistory, clearSearchHistory } = useHistory()
//     // const {
//     //     history,
//     //     deleteFromHistory,
//     //     clearHistory
//     // } = useSearchHistory();
//     const history = getSearchHistory()
//     const [searchHistory, setSearchHistory] = useState<string[]>([]);

//     // 1. Create a debounced version of the search query
//     const debouncedQuery = useDebounce(searchQuery, 300);


//     // 3. Filter the history based on the debounced query
//     // We use useMemo so it only recalculates when history or debouncedQuery changes
//     // const filteredHistory = useMemo(() => {
//     //     if (!debouncedQuery.trim()) return history;

//     //     return history.filter((item) =>
//     //         item.query.toLowerCase().includes(debouncedQuery.toLowerCase())
//     //     );
//     // }, [debouncedQuery, history]);

//     const filteredHistory = useMemo(() => {
//         if (!debouncedQuery.trim()) return searchHistory;

//         return searchHistory.filter(item =>
//             item.query.toLowerCase().includes(debouncedQuery.toLowerCase())
//         );
//     }, [debouncedQuery, searchHistory]);



//     // useEffect(() => {
//     //     setSearchHistory(history)
//     // }, []);

//     useEffect(() => {
//         const history = getSearchHistory();
//         setSearchHistory(history);
//     }, []);

//     const saveHistory = (newHistory: string[]) => {
//         // setSearchHistory(newHistory);
//     };

//     const handleSearch = (e: React.FormEvent) => {
//         e.preventDefault();
//         if (!searchQuery.trim()) return;
//         const updated = [searchQuery, ...history.filter(item => item.query !== searchQuery)].slice(0, 10);
//         console.log(updated)
//         setSearchQuery("");
//     };

//     // const deleteItem = (index: number) => {
//     //     // deleteFromHistory(index)
//     // };

//     const deleteItem = (id: string) => {
//         removeSearchHistory(id);
//         setSearchHistory(prev => prev.filter(item => item.id !== id));
//     };

//     // const clearAll = () => {
//     //     // clearHistory()
//     // };
//     const clearAll = () => {
//         clearSearchHistory();
//         setSearchHistory([]);
//     };

//     return (
//         <div className="w-full  mx-auto animate-in fade-in slide-in-from-bottom-4 duration-700">
//             <div
//                 className="rounded-3xl border shadow-2xl overflow-hidden"
//                 style={{ backgroundColor: 'var(--card)', borderColor: 'var(--border)' }}
//             >
//                 {/* Search Input Area */}
//                 <div className="p-6 pb-0">
//                     <form onSubmit={handleSearch} className="relative group">
//                         <Search className="absolute left-5 top-1/2 -translate-y-1/2 opacity-30 group-focus-within:opacity-100 transition-opacity" size={20} />
//                         <input
//                             type="text"
//                             value={searchQuery}
//                             onChange={(e) => setSearchQuery(e.target.value)}
//                             placeholder="Search or type URL"
//                             className="w-full py-4 pl-14 pr-6 rounded-xl border-2 outline-none transition-all text-lg font-small"
//                             style={{
//                                 backgroundColor: 'var(--input)',
//                                 borderColor: 'var(--border)',
//                                 color: 'var(--foreground)'
//                             }}
//                         />
//                     </form>
//                 </div>

//                 {/* History Section */}
//                 <div className="p-6">
//                     <div className="flex items-center justify-between mb-6 px-2">
//                         <div className="flex items-center gap-2 opacity-60">
//                             <History size={16} />
//                             <span className="text-xs font-bold uppercase tracking-[0.2em]">Recent Activity</span>
//                         </div>
//                         {history.length > 0 && (
//                             <button
//                                 onClick={clearAll}
//                                 className="text-[10px] font-bold uppercase tracking-widest px-3 py-1.5 rounded-full hover:bg-white/5 transition-all"
//                                 style={{ color: 'var(--chart-5)' }}
//                             >
//                                 Clear All
//                             </button>
//                         )}
//                     </div>
//                     {/* max-h-[400px]  */}
//                     <div className="space-y-2 search-scroll overflow-y-auto" style={{
//                         height: "calc(100vh - 425px)",
//                     }}>
//                         {history.length === 0 ? (
//                             <div className="py-12 text-center opacity-30">
//                                 <Clock className="mx-auto mb-3" size={32} />
//                                 <p className="text-sm font-medium">No recent searches</p>
//                             </div>
//                         ) : (
//                                 filteredHistory.map((item, index) => (
//                                     <div
//                                         key={`${item.id}`}
//                                         className="history-item-anim group flex items-center justify-between p-4 rounded-2xl border border-transparent hover:border-white/5 active:scale-[0.98] cursor-pointer"
//                                         style={{ backgroundColor: 'rgba(255,255,255,0.02)' }}
//                                     >
//                                         <div className="flex items-center gap-4 min-w-0">
//                                             <RotateCcw size={16} className="opacity-30 group-hover:rotate-[-45deg] transition-transform" />
//                                             <span className="text-sm md:text-base font-medium truncate opacity-80 group-hover:opacity-100 transition-opacity">
//                                                 {item.query}
//                                             </span>
//                                         </div>
//                                         <button
//                                             onClick={(e) => { e.stopPropagation(); deleteItem(item.id); }}
//                                             className="p-2 rounded-xl opacity-100 group-hover:opacity-100 hover:bg-white/5 transition-all"
//                                             style={{ color: 'var(--destructive)' }}
//                                         >
//                                             <Trash2 size={16} />
//                                         </button>
//                                     </div>
//                                 ))
//                             )}
//                     </div>
//                 </div>

//                 {/* Branding Footer */}
//                 <div className="p-6 pt-0 opacity-20 text-center">
//                     <p className="text-[10px] font-black tracking-widest uppercase">Privacy Protected Search</p>
//                 </div>
//             </div>
//         </div>
//     );
// }