import React, { useState, useEffect } from 'react';
import { X, UserPlus, User, Mail, Lock, ShieldCheck, Loader2, LogIn, RefreshCw, CheckCircle2 } from 'lucide-react';

const RegisterModal = ({ onClose, onSwitchToLogin }) => {
    // 1. Core Logic States
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState("");
    const [registerData, setRegisterData] = useState({
        username: "",
        email: "",
        password: "",
        confirmPassword: "",
    });

    // 2. Advanced Auth States (from your logic)
    const [usernameAvailable, setUsernameAvailable] = useState(null);
    const [checkingUsername, setCheckingUsername] = useState(false);
    const [showVerificationMessage, setShowVerificationMessage] = useState(false);
    const [verificationEmail, setVerificationEmail] = useState("");
    const [resendingVerification, setResendingVerification] = useState(false);

    // 3. Username Availability Check
    const checkUsernameAvailability = async (username) => {
        if (username.length < 3) {
            setUsernameAvailable(null);
            return;
        }
        setCheckingUsername(true);
        try {
            const response = await fetch(`/api/auth/check-username?username=${encodeURIComponent(username)}`);
            const data = await response.json();
            setUsernameAvailable(data.available);
        } catch (error) {
            console.error("Error checking username:", error);
            setUsernameAvailable(null);
        } finally {
            setCheckingUsername(false);
        }
    };

    // Debounce username check
    useEffect(() => {
        const timer = setTimeout(() => {
            checkUsernameAvailability(registerData.username);
        }, 500);
        return () => clearTimeout(timer);
    }, [registerData.username]);

    // 4. Handle Registration
    const handleRegister = async (e) => {
        e.preventDefault();
        setError("");

        if (registerData.password !== registerData.confirmPassword) {
            setError("Passwords do not match");
            return;
        }
        if (registerData.password.length < 6) {
            setError("Password must be at least 6 characters long");
            return;
        }
        if (usernameAvailable === false) {
            setError("Username is not available");
            return;
        }

        setLoading(true);
        try {
            // Using your endpoint logic
            const response = await fetch('/api/auth/register', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    username: registerData.username,
                    email: registerData.email,
                    password: registerData.password
                }),
            });

            const data = await response.json();

            if (!response.ok) {
                throw new Error(data.message || "Registration failed");
            }

            // Logic for Email Verification
            if (data.requiresVerification) {
                setVerificationEmail(registerData.email);
                setShowVerificationMessage(true);
            } else {
                if (data.token) localStorage.setItem('token', data.token);
                onClose();
            }
        } catch (err) {
            setError(err.message);
        } finally {
            setLoading(false);
        }
    };

    // 5. Resend Verification
    const handleResendVerification = async () => {
        setResendingVerification(true);
        try {
            const response = await fetch("/api/auth/resend-verification", {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify({ email: verificationEmail }),
            });
            if (response.ok) {
                setError("Verification email sent successfully!");
                setTimeout(() => setError(""), 3000);
            } else {
                const data = await response.json();
                setError(data.error || "Failed to send verification email");
            }
        } catch (error) {
            setError("Network error occurred");
        } finally {
            setResendingVerification(false);
        }
    };

    // VIEW: Verification Success
    if (showVerificationMessage) {
        return (
            <div className="fixed inset-0 z-50 flex items-center justify-center p-4 backdrop-blur-sm bg-black/20">
                <div className="relative w-full max-w-md p-10 rounded-[2rem] border shadow-2xl text-center" style={{ backgroundColor: 'var(--card)', borderColor: 'var(--border)' }}>
                    <div className="w-16 h-16 bg-blue-500/10 rounded-full flex items-center justify-center mx-auto mb-6">
                        <Mail size={32} className="text-blue-500" />
                    </div>
                    <h2 className="text-2xl font-bold mb-2">Verify your email</h2>
                    <p className="text-sm text-muted-foreground mb-8">Sent to: <b>{verificationEmail}</b></p>
                    <button
                        onClick={handleResendVerification}
                        disabled={resendingVerification}
                        className="w-full py-4 rounded-2xl border font-bold flex items-center justify-center gap-2 hover:bg-white/5 transition-all"
                        style={{ borderColor: 'var(--border)' }}
                    >
                        {resendingVerification ? <Loader2 className="animate-spin" size={20} /> : <RefreshCw size={20} />}
                        Resend Link
                    </button>
                    <button onClick={onClose} className="mt-6 text-xs font-bold uppercase tracking-widest opacity-50">Close</button>
                </div>
            </div>
        );
    }

    return (
        <div className="fixed inset-0 z-50 flex items-center justify-center p-4 sm:p-6 animate-in fade-in duration-300 backdrop-blur-sm bg-black/20">
            <div
                className="relative w-full max-w-md p-6 sm:p-10 rounded-[2rem] border shadow-2xl animate-in zoom-in-95 slide-in-from-bottom-4 duration-300"
                style={{ backgroundColor: 'var(--card)', borderColor: 'var(--border)' }}
            >
                <button onClick={onClose} className="absolute top-5 right-5 p-2 rounded-full hover:bg-white/5"><X size={20} /></button>

                <div className="text-center mb-6 sm:mb-8">
                    <div className="w-12 h-12 rounded-2xl flex items-center justify-center mx-auto mb-4 border shadow-inner" style={{ backgroundColor: 'var(--muted)', borderColor: 'var(--border)' }}>
                        <UserPlus size={24} style={{ color: 'var(--chart-1)' }} />
                    </div>
                    <h2 className="text-xl sm:text-2xl font-bold tracking-tight">Join Indzs</h2>
                </div>

                {error && (
                    <div className="flex items-center gap-2 p-3 mb-4 text-xs font-medium border rounded-xl bg-red-500/10 border-red-500/20 text-red-500">
                        <span>⚠️</span><span>{error}</span>
                    </div>
                )}

                <form className="space-y-4" onSubmit={handleRegister}>
                    <div className="space-y-1.5">
                        <div className="flex justify-between px-1">
                            <label className="text-[10px] font-bold uppercase tracking-widest opacity-50">Username</label>
                            {registerData.username.length >= 3 && (
                                <span className={`text-[10px] font-bold ${usernameAvailable ? 'text-green-500' : 'text-red-500'}`}>
                                    {checkingUsername ? '...' : usernameAvailable ? 'Available' : 'Taken'}
                                </span>
                            )}
                        </div>
                        <div className="relative group">
                            <User className="absolute left-4 top-1/2 -translate-y-1/2 opacity-40 group-focus-within:opacity-100" size={18} />
                            <input
                                type="text"
                                placeholder="Username"
                                className="w-full rounded-2xl py-3 pl-11 pr-4 text-sm border focus:ring-2 outline-none"
                                style={{ backgroundColor: 'var(--input)', borderColor: 'var(--border)', color: 'var(--foreground)' }}
                                required
                                value={registerData.username}
                                onChange={(e) => setRegisterData({ ...registerData, username: e.target.value })}
                            />
                        </div>
                    </div>

                    <div className="space-y-1.5">
                        <label className="text-[10px] font-bold uppercase tracking-widest opacity-50 ml-1">Email</label>
                        <div className="relative group">
                            <Mail className="absolute left-4 top-1/2 -translate-y-1/2 opacity-40 group-focus-within:opacity-100" size={18} />
                            <input
                                type="email"
                                placeholder="name@example.com"
                                className="w-full rounded-2xl py-3 pl-11 pr-4 text-sm border focus:ring-2 outline-none"
                                style={{ backgroundColor: 'var(--input)', borderColor: 'var(--border)', color: 'var(--foreground)' }}
                                required
                                value={registerData.email}
                                onChange={(e) => setRegisterData({ ...registerData, email: e.target.value })}
                            />
                        </div>
                    </div>

                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                        <div className="space-y-1.5">
                            <label className="text-[10px] font-bold uppercase tracking-widest opacity-50 ml-1">Password</label>
                            <input
                                type="password"
                                className="w-full rounded-2xl py-3 px-4 text-sm border outline-none"
                                style={{ backgroundColor: 'var(--input)', borderColor: 'var(--border)', color: 'var(--foreground)' }}
                                required
                                value={registerData.password}
                                onChange={(e) => setRegisterData({ ...registerData, password: e.target.value })}
                            />
                        </div>
                        <div className="space-y-1.5">
                            <label className="text-[10px] font-bold uppercase tracking-widest opacity-50 ml-1">Confirm</label>
                            <input
                                type="password"
                                className="w-full rounded-2xl py-3 px-4 text-sm border outline-none"
                                style={{ backgroundColor: 'var(--input)', borderColor: 'var(--border)', color: 'var(--foreground)' }}
                                required
                                value={registerData.confirmPassword}
                                onChange={(e) => setRegisterData({ ...registerData, confirmPassword: e.target.value })}
                            />
                        </div>
                    </div>

                    <button
                        type="submit"
                        disabled={loading || !usernameAvailable}
                        className="w-full py-4 font-bold rounded-2xl transition-all hover:brightness-110 active:scale-[0.97] mt-4 shadow-lg text-sm flex items-center justify-center gap-2"
                        style={{ backgroundColor: 'var(--primary)', color: 'var(--primary-foreground)', opacity: (loading || !usernameAvailable) ? '0.7' : '1' }}
                    >
                        {loading ? <Loader2 className="animate-spin" size={20} /> : 'Create Account'}
                    </button>
                </form>

                <div className="mt-8 pt-6 border-t flex flex-col items-center gap-4" style={{ borderColor: 'var(--border)' }}>
                    <p className="text-xs sm:text-sm text-muted-foreground">Already have an account?</p>
                    <button
                        onClick={onSwitchToLogin}
                        className="w-full py-3 rounded-2xl border text-sm font-medium flex items-center justify-center gap-2 hover:bg-white/5 transition-all"
                        style={{ borderColor: 'var(--border)', color: 'var(--foreground)' }}
                    >
                        <LogIn size={16} /> Sign in instead
                    </button>
                </div>
            </div>
        </div>
    );
};

export default RegisterModal;
// import React, { useState } from 'react';
// import { X, UserPlus, User, Mail, Lock, ShieldCheck, Loader2, LogIn } from 'lucide-react';

// const RegisterModal = ({ onClose, onSwitchToLogin }) => {
//     // 1. Logic State
//     const [loading, setLoading] = useState(false);
//     const [error, setError] = useState("");
//     const [regData, setRegData] = useState({
//         name: "",
//         email: "",
//         password: "",
//         confirmPassword: ""
//     });

//     // 2. API Handler
//     const handleRegister = async (e) => {
//         e.preventDefault();
//         setError("");

//         // Basic Validation: Check if passwords match
//         if (regData.password !== regData.confirmPassword) {
//             setError("Passwords do not match. Please try again.");
//             return;
//         }

//         // Basic Validation: Password length
//         if (regData.password.length < 6) {
//             setError("Password must be at least 6 characters long.");
//             return;
//         }

//         setLoading(true);

//         try {
//             const response = await fetch('https://api.indzs.com/v1/auth/register', {
//                 method: 'POST',
//                 headers: {
//                     'Content-Type': 'application/json',
//                 },
//                 body: JSON.stringify({
//                     full_name: regData.name,
//                     email: regData.email,
//                     password: regData.password,
//                 }),
//             });

//             const data = await response.json();

//             if (!response.ok) {
//                 // If the server sends an error message (e.g., "User already exists")
//                 throw new Error(data.message || "Something went wrong during registration.");
//             }

//             // SUCCESS: Save token and close modal
//             console.log("Registration Successful:", data);
//             if (data.token) localStorage.setItem('token', data.token);
            
//             onClose(); // Close modal on success

//         } catch (err) {
//             setError(err.message);
//         } finally {
//             setLoading(false);
//         }
//     };

//     return (
//         <div className="fixed inset-0 z-50 flex items-center justify-center p-4 sm:p-6 animate-in fade-in duration-300 backdrop-blur-sm bg-black/20">
//             <div
//                 className="relative w-full max-w-md p-6 sm:p-10 rounded-[2rem] border shadow-2xl animate-in zoom-in-95 slide-in-from-bottom-4 duration-300"
//                 style={{ backgroundColor: 'var(--card)', borderColor: 'var(--border)' }}
//             >
//                 {/* Close Button */}
//                 <button
//                     onClick={onClose}
//                     className="absolute top-5 right-5 p-2 rounded-full hover:bg-white/5 transition-colors focus:ring-2 focus:ring-offset-2 focus:ring-offset-black"
//                     style={{ outlineColor: 'var(--ring)' }}
//                 >
//                     <X size={20} style={{ color: 'var(--muted-foreground)' }} />
//                 </button>

//                 {/* Header */}
//                 <div className="text-center mb-6 sm:mb-8">
//                     <div className="w-12 h-12 sm:w-14 sm:h-14 rounded-2xl flex items-center justify-center mx-auto mb-4 border shadow-inner"
//                         style={{ backgroundColor: 'var(--muted)', borderColor: 'var(--border)' }}>
//                         <UserPlus size={24} className="sm:size-7" style={{ color: 'var(--chart-1)' }} />
//                     </div>
//                     <h2 className="text-xl sm:text-2xl font-bold tracking-tight" style={{ color: 'var(--foreground)' }}>
//                         Join Indzs
//                     </h2>
//                     <p className="text-xs sm:text-sm mt-1" style={{ color: 'var(--muted-foreground)' }}>
//                         Start your journey with a new account
//                     </p>
//                 </div>

//                 {/* Alert Section */}
//                 {error && (
//                     <div className="flex items-center gap-2 p-3 mb-4 text-xs font-medium border rounded-xl bg-red-500/10 border-red-500/20 text-red-500 animate-in shake-in">
//                         <span>⚠️</span>
//                         <span>{error}</span>
//                     </div>
//                 )}

//                 {/* Form */}
//                 <form className="space-y-4" onSubmit={handleRegister}>

//                     {/* Full Name Field */}
//                     <div className="space-y-1.5">
//                         <label className="text-[10px] font-bold uppercase tracking-widest opacity-50 ml-1">Full Name</label>
//                         <div className="relative group">
//                             <User className="absolute left-4 top-1/2 -translate-y-1/2 opacity-40 group-focus-within:opacity-100 transition-opacity" size={18} />
//                             <input
//                                 type="text"
//                                 placeholder="John Doe"
//                                 className="w-full rounded-2xl py-3 sm:py-3.5 pl-11 pr-4 text-sm sm:text-base outline-none transition-all border focus:ring-2"
//                                 style={{ backgroundColor: 'var(--input)', borderColor: 'var(--border)', color: 'var(--foreground)' }}
//                                 required
//                                 value={regData.name}
//                                 onChange={(e) => setRegData({ ...regData, name: e.target.value })}
//                             />
//                         </div>
//                     </div>

//                     {/* Email Field */}
//                     <div className="space-y-1.5">
//                         <label className="text-[10px] font-bold uppercase tracking-widest opacity-50 ml-1">Email Address</label>
//                         <div className="relative group">
//                             <Mail className="absolute left-4 top-1/2 -translate-y-1/2 opacity-40 group-focus-within:opacity-100 transition-opacity" size={18} />
//                             <input
//                                 type="email"
//                                 placeholder="name@example.com"
//                                 className="w-full rounded-2xl py-3 sm:py-3.5 pl-11 pr-4 text-sm sm:text-base outline-none transition-all border focus:ring-2"
//                                 style={{ backgroundColor: 'var(--input)', borderColor: 'var(--border)', color: 'var(--foreground)' }}
//                                 required
//                                 value={regData.email}
//                                 onChange={(e) => setRegData({ ...regData, email: e.target.value })}
//                             />
//                         </div>
//                     </div>

//                     {/* Password Fields Row */}
//                     <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
//                         <div className="space-y-1.5">
//                             <label className="text-[10px] font-bold uppercase tracking-widest opacity-50 ml-1">Password</label>
//                             <div className="relative group">
//                                 <Lock className="absolute left-4 top-1/2 -translate-y-1/2 opacity-40 group-focus-within:opacity-100 transition-opacity" size={18} />
//                                 <input
//                                     type="password"
//                                     placeholder="••••••••"
//                                     className="w-full rounded-2xl py-3 sm:py-3.5 pl-11 pr-4 text-sm sm:text-base outline-none transition-all border focus:ring-2"
//                                     style={{ backgroundColor: 'var(--input)', borderColor: 'var(--border)', color: 'var(--foreground)' }}
//                                     required
//                                     value={regData.password}
//                                     onChange={(e) => setRegData({ ...regData, password: e.target.value })}
//                                 />
//                             </div>
//                         </div>
//                         <div className="space-y-1.5">
//                             <label className="text-[10px] font-bold uppercase tracking-widest opacity-50 ml-1">Confirm</label>
//                             <div className="relative group">
//                                 <ShieldCheck className="absolute left-4 top-1/2 -translate-y-1/2 opacity-40 group-focus-within:opacity-100 transition-opacity" size={18} />
//                                 <input
//                                     type="password"
//                                     placeholder="••••••••"
//                                     className="w-full rounded-2xl py-3 sm:py-3.5 pl-11 pr-4 text-sm sm:text-base outline-none transition-all border focus:ring-2"
//                                     style={{ backgroundColor: 'var(--input)', borderColor: 'var(--border)', color: 'var(--foreground)' }}
//                                     required
//                                     value={regData.confirmPassword}
//                                     onChange={(e) => setRegData({ ...regData, confirmPassword: e.target.value })}
//                                 />
//                             </div>
//                         </div>
//                     </div>

//                     {/* Submit Button */}
//                     <button
//                         type="submit"
//                         disabled={loading}
//                         className="w-full py-3.5 sm:py-4 font-bold rounded-2xl transition-all hover:brightness-110 active:scale-[0.97] mt-4 shadow-lg text-sm sm:text-base flex items-center justify-center gap-2"
//                         style={{
//                             backgroundColor: 'var(--primary)',
//                             color: 'var(--primary-foreground)',
//                             opacity: loading ? '0.7' : '1',
//                             cursor: loading ? 'not-allowed' : 'pointer'
//                         }}
//                     >
//                         {loading ? (
//                             <>
//                                 <Loader2 className="animate-spin" size={20} /> Processing...
//                             </>
//                         ) : (
//                             'Create Account'
//                         )}
//                     </button>
//                 </form>

//                 {/* Footer */}
//                 <div className="mt-6 sm:mt-8 pt-6 border-t flex flex-col items-center gap-3 sm:gap-4" style={{ borderColor: 'var(--border)' }}>
//                     <p className="text-xs sm:text-sm" style={{ color: 'var(--muted-foreground)' }}>Already have an account?</p>
//                     <button
//                         type="button"
//                         onClick={onSwitchToLogin}
//                         className="w-full py-3 rounded-2xl border text-xs sm:text-sm font-medium flex items-center justify-center gap-2 hover:bg-white/5 transition-all"
//                         style={{ borderColor: 'var(--border)', color: 'var(--foreground)' }}
//                     >
//                         <LogIn size={16} />
//                         Sign in instead
//                     </button>
//                 </div>
//             </div>
//         </div>
//     );
// };

// export default RegisterModal;