"use client"
import React, { useState, useRef, useEffect } from "react"
import { Globe, Languages, MapPin, ChevronDown, Check } from "lucide-react"

// --- Custom Hook for clicking outside to close ---
function useOutsideClick(ref: React.RefObject<HTMLDivElement>, callback: () => void) {
  useEffect(() => {
    function handleClickOutside(event: MouseEvent) {
      if (ref.current && !ref.current.contains(event.target as Node)) {
        callback();
      }
    }
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, [ref, callback]);
}



export default function LanguageRegion() {
  const [language, setLanguage] = useState("English")
  const [region, setRegion] = useState("Global")

  // 1. Load initial values from localStorage
  useEffect(() => {
    const saved = localStorage.getItem("indus-search-settings");
    if (saved) {
      const parsed = JSON.parse(saved);
      if (parsed.language) setLanguage(parsed.language);
      if (parsed.region) setRegion(parsed.region);
    }
  }, []);

  // 2. Sync to localStorage and notify other components
  useEffect(() => {
    const saved = localStorage.getItem("indus-search-settings");
    const currentSettings = saved ? JSON.parse(saved) : {};

    const updatedSettings = {
      ...currentSettings,
      language: language,
      region: region,
    };

    localStorage.setItem("indus-search-settings", JSON.stringify(updatedSettings));

    // Trigger the update for the "CurrentSettings" display card
    window.dispatchEvent(new Event("settingsUpdated"));
  }, [language, region]);

  return (
    <div className="max-w-3xl mx-auto animate-in fade-in slide-in-from-bottom-4 duration-700">
      <div
        className="rounded-2xl border transition-all duration-300 overflow-hidden"
        style={{
          backgroundColor: 'var(--card)',
          borderColor: 'var(--border)',
          color: 'var(--card-foreground)'
        }}
      >
        {/* Header */}
        <div className="p-6 border-b" style={{ borderColor: 'var(--border)' }}>
          <h2 className="text-xl font-bold tracking-tight flex items-center gap-2">
            <Globe size={20} style={{ color: 'var(--secondary)' }} />
            Language & Region
          </h2>
          <p className="text-sm mt-1" style={{ color: 'var(--muted-foreground)' }}>
            Personalize your experience by setting your preferred language and local area.
          </p>
        </div>

        {/* Form Content */}
        <div className="p-6 space-y-8">

          {/* Language Selection */}
          <div className="grid grid-cols-1 lg gap-4 items-start">
            <div className="flex items-center gap-3 lg:pt-3">
              <div className="p-2 rounded-lg" style={{ backgroundColor: 'var(--secondary)' }}>
                <Languages size={18} style={{ color: '#fff' }} />
              </div>
              <label className="text-sm font-semibold opacity-80">Display Language</label>
            </div>

            {/* <div className="lg:col-span-2 relative group">
              <select
                value={language}
                onChange={(e) => setLanguage(e.target.value)}
                className="w-full appearance-none rounded-xl px-4 py-3 outline-none border transition-all cursor-pointer font-medium text-sm"
                style={{ 
                  backgroundColor: 'var(--input)', 
                  borderColor: 'var(--border)',
                  color: 'var(--foreground)' 
                }}
              >
                <option value="English">English</option>
                <option value="Hindi">Hindi</option>
                <option value="Bengali">Bengali</option>
              </select>
              <ChevronDown 
                size={16} 
                className="absolute right-4 top-1/2 -translate-y-1/2 pointer-events-none opacity-40 group-hover:opacity-100 transition-opacity" 
              />
            </div> */}

            <div className="space-y-2">
              {/* <label className="text-xs font-bold uppercase tracking-widest opacity-40 ml-1">Display Language</label> */}
              <CustomDropdown
                value={language}
                options={["English", "Hindi", "Bengali"]}
                onChange={setLanguage}
                icon={<Languages size={18} />}
              />
            </div>

          </div>

          {/* Region Selection */}
          <div className="grid grid-cols-1 lg gap-4 items-start">
            <div className="flex items-center gap-3 lg:pt-3">
              <div className="p-2 rounded-lg" style={{ backgroundColor: 'var(--secondary)' }}>
                <MapPin size={18} style={{ color: '#fff' }} />
              </div>
              <label className="text-sm font-semibold opacity-80">Search Region</label>
            </div>

            {/* <div className="lg:col-span-2 relative group">
              <select
                value={region}
                onChange={(e) => setRegion(e.target.value)}
                className="w-full appearance-none rounded-xl px-4 py-3 outline-none border transition-all cursor-pointer font-medium text-sm"
                style={{ 
                  backgroundColor: 'var(--input)', 
                  borderColor: 'var(--border)',
                  color: 'var(--foreground)' 
                }}
              >
                <option value="Global">Global</option>
                <option value="India">India</option>
                <option value="USA">USA</option>
              </select>
              <ChevronDown 
                size={16} 
                className="absolute right-4 top-1/2 -translate-y-1/2 pointer-events-none opacity-40 group-hover:opacity-100 transition-opacity" 
              />
            </div> */}

            <div className="space-y-2">
              {/* <label className="text-xs font-bold uppercase tracking-widest opacity-40 ml-1">Search Region</label> */}
              <CustomDropdown
                value={region}
                options={["Global", "India", "USA"]}
                onChange={setRegion}
                icon={<MapPin size={18} />}
              />
            </div>

          </div>

        </div>

        {/* Subtle Footer Note */}
        <div className="px-6 py-4 bg-white/[0.02] border-t" style={{ borderColor: 'var(--border)' }}>
          <p className="text-[11px] uppercase tracking-widest font-bold opacity-30">
            Changes are saved automatically
          </p>
        </div>
      </div>
    </div>
  )
}


// --- Custom Select Component ---
interface CustomDropdownProps {
  value: string;
  options: string[];
  onChange: (val: string) => void;
  icon: React.ReactNode;
}

const CustomDropdown = ({ value, options, onChange, icon }: CustomDropdownProps) => {
  const [isOpen, setIsOpen] = useState(false);
  const containerRef = useRef<HTMLDivElement>(null);
  useOutsideClick(containerRef, () => setIsOpen(false));

  return (
    <div className="relative w-full" ref={containerRef}>
      <button
        onClick={() => setIsOpen(!isOpen)}
        className="w-full flex items-center justify-between px-4 py-3 rounded-xl border transition-all duration-200 outline-none hover:bg-white/[0.02]"
        style={{
          backgroundColor: 'var(--input)',
          borderColor: isOpen ? 'var(--primary)' : 'var(--border)',
          color: 'var(--foreground)'
        }}
      >
        <div className="flex items-center gap-3">
          <span className="opacity-60 ">{icon}</span>
          <span className="text-sm font-medium">{value}</span>
        </div>
        <ChevronDown
          size={16}
          className={`transition-transform duration-300 ${isOpen ? "rotate-180" : ""}`}
        />
      </button>

      {isOpen && (
        <div
          className="fixed z-50 w-full mt-2 rounded-xl border shadow-2xl animate-in fade-in zoom-in-95 duration-200 overflow-hidden"
          style={{
            backgroundColor: 'var(--popover)',
            borderColor: 'var(--border)',
            boxShadow: '0 10px 40px -10px rgba(0,0,0,0.5)',
            width: "-webkit-fill-available",
            marginRight: "24px",
          }}
        >
          <div className="p-1">
            {options.map((option) => (
              <button
                key={option}
                onClick={() => {
                  onChange(option);
                  setIsOpen(false);
                }}
                className="w-full flex items-center justify-between px-3 py-2.5 rounded-lg text-sm transition-colors hover:bg-white/5 group"
              >
                <span className={value === option ? "font-bold transition-colors" : "opacity-70 group-hover:opacity-100"}>
                  {option}
                </span>
                {value === option && <Check size={14} className="transition-colors" />}
              </button>
            ))}
          </div>
        </div>
      )}
    </div>
  );
};
