import React, { useState } from 'react';
import { X, KeyRound, Mail, ArrowLeft, Loader2, CheckCircle2 } from 'lucide-react';

const ForgotPasswordModal = ({ onClose, onBackToLogin }) => {
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState("");
    const [submitted, setSubmitted] = useState(false);
    const [email, setEmail] = useState("");

    const handleResetRequest = async (e) => {
        e.preventDefault();
        setError("");
        setLoading(true);

        try {
            const response = await fetch('/api/auth/forgot-password', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ email }),
            });

            const data = await response.json();

            if (!response.ok) {
                throw new Error(data.message || "Could not process request.");
            }

            // Success state
            setSubmitted(true);
        } catch (err) {
            setError(err.message);
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className="fixed inset-0 z-50 flex items-center justify-center p-4 sm:p-6 animate-in fade-in duration-300 backdrop-blur-sm bg-black/20">
            <div
                className="relative w-full max-w-md p-6 sm:p-10 rounded-[2rem] border shadow-2xl animate-in zoom-in-95 slide-in-from-bottom-4 duration-300"
                style={{ backgroundColor: 'var(--card)', borderColor: 'var(--border)' }}
            >
                {/* Close Button */}
                <button
                    onClick={onClose}
                    className="absolute top-5 right-5 p-2 rounded-full hover:bg-white/5 transition-colors"
                >
                    <X size={20} style={{ color: 'var(--muted-foreground)' }} />
                </button>

                {!submitted ? (
                    <>
                        {/* Header */}
                        <div className="text-center mb-6 sm:mb-8">
                            <div className="w-12 h-12 sm:w-14 sm:h-14 rounded-2xl flex items-center justify-center mx-auto mb-4 border shadow-inner"
                                style={{ backgroundColor: 'var(--muted)', borderColor: 'var(--border)' }}>
                                <KeyRound size={24} className="sm:size-7" style={{ color: 'var(--chart-1)' }} />
                            </div>
                            <h2 className="text-xl sm:text-2xl font-bold tracking-tight" style={{ color: 'var(--foreground)' }}>
                                Forgot Password?
                            </h2>
                            <p className="text-xs sm:text-sm mt-1 px-4" style={{ color: 'var(--muted-foreground)' }}>
                                Enter your email and we'll send you a link to reset your password.
                            </p>
                        </div>

                        {/* Error Alert */}
                        {error && (
                            <div className="flex items-center gap-2 p-3 mb-4 text-xs font-medium border rounded-xl bg-red-500/10 border-red-500/20 text-red-500">
                                <span>⚠️</span>
                                <span>{error}</span>
                            </div>
                        )}

                        {/* Form */}
                        <form className="space-y-4" onSubmit={handleResetRequest}>
                            <div className="space-y-1.5">
                                <label className="text-[10px] font-bold uppercase tracking-widest opacity-50 ml-1">Email Address</label>
                                <div className="relative group">
                                    <Mail className="absolute left-4 top-1/2 -translate-y-1/2 opacity-40 group-focus-within:opacity-100 transition-opacity" size={18} />
                                    <input
                                        type="email"
                                        placeholder="name@example.com"
                                        className="w-full rounded-2xl py-3 sm:py-3.5 pl-11 pr-4 text-sm sm:text-base outline-none transition-all border focus:ring-2"
                                        style={{ backgroundColor: 'var(--input)', borderColor: 'var(--border)', color: 'var(--foreground)' }}
                                        required
                                        value={email}
                                        onChange={(e) => setEmail(e.target.value)}
                                    />
                                </div>
                            </div>

                            <button
                                type="submit"
                                disabled={loading}
                                className="w-full py-3.5 sm:py-4 font-bold rounded-2xl transition-all hover:brightness-110 active:scale-[0.97] mt-2 shadow-lg text-sm sm:text-base flex items-center justify-center gap-2"
                                style={{
                                    backgroundColor: 'var(--primary)',
                                    color: 'var(--primary-foreground)',
                                    opacity: loading ? '0.7' : '1'
                                }}
                            >
                                {loading ? <Loader2 className="animate-spin" size={20} /> : 'Send Reset Link'}
                            </button>
                        </form>
                    </>
                ) : (
                    /* Success State */
                    <div className="text-center py-4 animate-in zoom-in-95 duration-500">
                        <div className="w-16 h-16 bg-green-500/10 border border-green-500/20 rounded-full flex items-center justify-center mx-auto mb-6">
                            <CheckCircle2 size={32} className="text-green-500" />
                        </div>
                        <h2 className="text-xl font-bold mb-2" style={{ color: 'var(--foreground)' }}>Check your email</h2>
                        <p className="text-sm mb-8" style={{ color: 'var(--muted-foreground)' }}>
                            We've sent a password reset link to <br />
                            <span className="font-semibold" style={{ color: 'var(--foreground)' }}>{email}</span>
                        </p>
                        <button
                            onClick={onBackToLogin}
                            className="w-full py-3.5 font-bold rounded-2xl transition-all border hover:bg-white/5"
                            style={{ borderColor: 'var(--border)', color: 'var(--foreground)' }}
                        >
                            Back to Sign In
                        </button>
                    </div>
                )}

                {/* Footer (Only shown if not submitted) */}
                {!submitted && (
                    <div className="mt-8 pt-6 border-t text-center" style={{ borderColor: 'var(--border)' }}>
                        <button
                            onClick={onBackToLogin}
                            className="flex items-center justify-center gap-2 text-xs sm:text-sm font-medium mx-auto hover:opacity-70 transition-opacity"
                            style={{ color: 'var(--muted-foreground)' }}
                        >
                            <ArrowLeft size={16} />
                            Back to Sign In
                        </button>
                    </div>
                )}
            </div>
        </div>
    );
};

export default ForgotPasswordModal;