import { useEffect, useState } from "react";
import {LayoutDashboard } from "lucide-react";
export default function CurrentSettings() {
  const [settings, setSettings] = useState({
    theme: "system",
    maxResults: 10,
    safeSearch: true,
    autoComplete: true,
  });

  const updateSettingsFromStorage = () => {
    const saved = localStorage.getItem("indus-search-settings");
    if (saved) {
      setSettings(JSON.parse(saved));
    }
  };

  useEffect(() => {
    // 1. Initial Load
    updateSettingsFromStorage();

    // 2. Listen for changes made in OTHER tabs/windows
    window.addEventListener("storage", updateSettingsFromStorage);

    // 3. Custom Event listener for changes in the SAME tab
    // (Since 'storage' event only fires for other tabs)
    window.addEventListener("settingsUpdated", updateSettingsFromStorage);

    return () => {
      window.removeEventListener("storage", updateSettingsFromStorage);
      window.removeEventListener("settingsUpdated", updateSettingsFromStorage);
    };
  }, []);

  return (
    <div 
      className="rounded-2xl overflow-hidden transition-all duration-500 animate-in fade-in slide-in-from-bottom-4"
      style={{ 
        backgroundColor: 'var(--card)', 
        border: '1px solid var(--border)',
      }}
    >
      
      <div className="p-8 border-b" style={{ borderColor: 'var(--border)', color: 'var(--card-foreground)'}}>
          <h2 className="text-xl font-bold tracking-tight flex items-center gap-2">
            <LayoutDashboard size={20} style={{ color: 'var(--chart-3)' }} />
            System Overview
          </h2>
          <p className="text-sm mt-1" style={{ color: 'var(--muted-foreground)' }}>
           View system status and key information.
          </p>
        </div>

      <div className="p-6">
        <h2 className="text-xs font-bold uppercase tracking-[0.2em] mb-4" 
            style={{ color: 'var(--muted-foreground)' }}>
          System Overview
        </h2>

        <div className="flex flex-col">
          <Row 
            label="Theme" 
            value={settings.theme.charAt(0).toUpperCase() + settings.theme.slice(1)} 
          />
          <Row 
            label="Results per page" 
            value={settings.maxResults} 
          />
          <Row 
            label="Safe Search" 
            value={settings.safeSearch ? "Enabled" : "Disabled"} 
          />
          <Row 
            label="Auto-complete" 
            value={settings.autoComplete ? "Active" : "Inactive"} 
            isLast 
          />
        </div>
      </div>
    </div>
  );
}

// Row component remains the same...
function Row({ label, value, isLast }: { label: string; value: string | number, isLast?: boolean }) {
  return (
    <div className={`group flex justify-between items-center py-4 px-4 -mx-4 transition-all duration-300 ease-out hover:translate-x-1`}
      style={{ borderBottom: isLast ? 'none' : '1px solid var(--border)' }}>
      <span className="text-sm font-medium" style={{ color: 'var(--muted-foreground)' }}>{label}</span>
      <span key={value} className="text-sm font-semibold animate-in fade-in zoom-in-95 duration-300" style={{ color: 'var(--secondary)' }}>
        {value}
      </span>
    </div>
  );
}