"use client"

import type React from "react"

import { useState, useEffect } from "react"
import { useRouter, useSearchParams } from "next/navigation"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Alert, AlertDescription } from "@/components/ui/alert"
import { Logo } from "@/components/logo"
import { Eye, EyeOff, CheckCircle,ShieldCheck, XCircle,KeyRound,Loader2 , ArrowLeft} from "lucide-react"


const PageWrapper = ({ children }) => (
    <div className="min-h-screen flex items-center justify-center p-4 relative overflow-hidden" style={{ backgroundColor: 'var(--background)' }}>
      {/* Decorative background blobs */}
      <div className="absolute top-[-10%] left-[-10%] w-[40%] h-[40%] rounded-full opacity-10 blur-[120px]" style={{ backgroundColor: 'var(--chart-1)' }} />
      <div className="absolute bottom-[-10%] right-[-10%] w-[40%] h-[40%] rounded-full opacity-10 blur-[120px]" style={{ backgroundColor: 'var(--primary)' }} />
      
      <div className="w-full max-w-md z-10 animate-in fade-in zoom-in-95 duration-500">
        {children}
      </div>
    </div>
  );


export default function ResetPasswordPage() {
  const router = useRouter()
  const searchParams = useSearchParams()
  const token = searchParams.get("token")

  const [password, setPassword] = useState("")
  const [confirmPassword, setConfirmPassword] = useState("")
  const [showPassword, setShowPassword] = useState(false)
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState("")
  const [success, setSuccess] = useState(false)
  const [validatingToken, setValidatingToken] = useState(true)
  const [tokenValid, setTokenValid] = useState(false)

  useEffect(() => {
    if (!token) {
      setError("Invalid reset link")
      setValidatingToken(false)
      return
    }

    // Validate token
    const validateToken = async () => {
      try {
        const response = await fetch(`/api/auth/validate-reset-token?token=${token}`)
        if (response.ok) {
          setTokenValid(true)
        } else {
          setError("Invalid or expired reset link")
        }
      } catch (error) {
        setError("Failed to validate reset link")
      } finally {
        setValidatingToken(false)
      }
    }

    validateToken()
  }, [token])

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setLoading(true)
    setError("")

    if (password !== confirmPassword) {
      setError("Passwords do not match")
      setLoading(false)
      return
    }

    if (password.length < 6) {
      setError("Password must be at least 6 characters long")
      setLoading(false)
      return
    }

    try {
      const response = await fetch("/api/auth/reset-password-confirm", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ token, password }),
      })

      const data = await response.json()

      if (response.ok) {
        setSuccess(true)
      } else {
        setError(data.error || "Failed to reset password")
      }
    } catch (error) {
      setError("Network error occurred")
    } finally {
      setLoading(false)
    }
  }


  if (validatingToken) {
    return (
      <PageWrapper>
        <div className="text-center">
          <div className="relative w-20 h-20 mx-auto mb-6">
            <div className="absolute inset-0 rounded-3xl border-4 border-dashed animate-spin opacity-20" style={{ borderColor: 'var(--primary)' }} />
            <div className="absolute inset-0 flex items-center justify-center">
              <ShieldCheck size={32} className="animate-pulse" style={{ color: 'var(--primary)' }} />
            </div>
          </div>
          <h2 className="text-xl font-bold tracking-tight">Securing your request</h2>
          <p className="text-sm opacity-60 mt-2">Validating reset link with our servers...</p>
        </div>
      </PageWrapper>
    )
  }

  if (success) {
    return (
      <PageWrapper>
        <div className="p-10 rounded-[2.5rem] border shadow-2xl text-center" style={{ backgroundColor: 'var(--card)', borderColor: 'var(--border)' }}>
          <div className="w-16 h-16 bg-green-500/10 border border-green-500/20 rounded-2xl flex items-center justify-center mx-auto mb-6">
            <CheckCircle size={32} className="text-green-500" />
          </div>
          <h1 className="text-2xl font-bold mb-3">All Set!</h1>
          <p className="text-sm text-muted-foreground mb-8">
            Your password has been updated. You can now securely sign in to your dashboard.
          </p>
          <button 
            // onClick={() => router.push("/ads-center/auth")}
            onClick={() => router.push("/settings")}
            
            className="w-full py-4 font-bold rounded-2xl shadow-lg transition-all active:scale-[0.98]"
            style={{ backgroundColor: 'var(--primary)', color: 'var(--primary-foreground)' }}
          >
            Back to Sign In
          </button>
        </div>
      </PageWrapper>
    );
  }

  // 3. INVALID TOKEN STATE
  if (!tokenValid) {
    return (
      <PageWrapper>
        <div className="p-10 rounded-[2.5rem] border shadow-2xl text-center" style={{ backgroundColor: 'var(--card)', borderColor: 'var(--border)' }}>
          <div className="w-16 h-16 bg-red-500/10 border border-red-500/20 rounded-2xl flex items-center justify-center mx-auto mb-6">
            <XCircle size={32} className="text-red-500" />
          </div>
          <h1 className="text-2xl font-bold mb-3">Expired Link</h1>
          <p className="text-sm text-muted-foreground mb-8">
            {error || "This password reset link is no longer valid or has already been used."}
          </p>
          <button 
            // onClick={() => router.push("/ads-center/auth")}
            onClick={()=> router.push("/settings")}

            className="w-full py-4 font-bold rounded-2xl border transition-all hover:bg-white/5"
            style={{ borderColor: 'var(--border)', color: 'var(--foreground)' }}
          >
            Request New Link
          </button>
        </div>
      </PageWrapper>
    )
  }

  // if (!tokenValid) {
  //   return (
  //     <div className="min-h-screen bg-background flex items-center justify-center p-4">
  //       <div className="w-full max-w-md">
  //         <div className="text-center mb-8">
  //           <Logo size="md" className="mx-auto mb-4" />
  //           <h1 className="text-2xl font-semibold text-foreground">Invalid Reset Link</h1>
  //         </div>

  //         <Card>
  //           <CardContent className="pt-6">
  //             <Alert variant="destructive">
  //               <AlertDescription>{error}</AlertDescription>
  //             </Alert>
  //             <Button onClick={() => router.push("/ads-center/auth")} className="w-full mt-4">
  //               Back to Sign In
  //             </Button>
  //           </CardContent>
  //         </Card>
  //       </div>
  //     </div>
  //   )
  // }

  return (
    <PageWrapper>
      <div className="p-8 sm:p-10 rounded-[2.5rem] border shadow-2xl" style={{ backgroundColor: 'var(--card)', borderColor: 'var(--border)' }}>
        
        {/* Header */}
        <div className="text-center mb-8">
          <div className="w-14 h-14 rounded-2xl flex items-center justify-center mx-auto mb-4 border shadow-inner"
              style={{ backgroundColor: 'var(--muted)', borderColor: 'var(--border)' }}>
              <KeyRound size={28} style={{ color: 'var(--chart-1)' }} />
          </div>
          <h1 className="text-2xl font-bold tracking-tight">Create New Password</h1>
          <p className="text-sm mt-2 opacity-60 px-4">Ensure your new password is at least 6 characters long.</p>
        </div>

        {/* Error Alert */}
        {error && (
            <div className="flex items-center gap-2 p-3 mb-6 text-xs font-medium border rounded-xl bg-red-500/10 border-red-500/20 text-red-500 animate-in shake-2">
                <span>⚠️</span>
                <span>{error}</span>
            </div>
        )}

        <form onSubmit={handleSubmit} className="space-y-5">
          {/* New Password */}
          <div className="space-y-1.5">
            <label className="text-[10px] font-bold uppercase tracking-widest opacity-50 ml-1">New Password</label>
            <div className="relative group">
              <input
                type={showPassword ? "text" : "password"}
                value={password}
                onChange={(e) => setPassword(e.target.value)}
                placeholder="••••••••"
                className="w-full rounded-2xl py-4 px-5 text-sm outline-none transition-all border focus:ring-2"
                style={{ backgroundColor: 'var(--input)', borderColor: 'var(--border)', color: 'var(--foreground)' }}
                required
              />
              <button
                type="button"
                onClick={() => setShowPassword(!showPassword)}
                className="absolute right-4 top-1/2 -translate-y-1/2 opacity-40 hover:opacity-100 transition-opacity"
              >
                {showPassword ? <EyeOff size={18} /> : <Eye size={18} />}
              </button>
            </div>
          </div>

          {/* Confirm Password */}
          <div className="space-y-1.5">
            <label className="text-[10px] font-bold uppercase tracking-widest opacity-50 ml-1">Confirm Password</label>
            <input
              type={showPassword ? "text" : "password"}
              value={confirmPassword}
              onChange={(e) => setConfirmPassword(e.target.value)}
              placeholder="••••••••"
              className="w-full rounded-2xl py-4 px-5 text-sm outline-none transition-all border focus:ring-2"
              style={{ backgroundColor: 'var(--input)', borderColor: 'var(--border)', color: 'var(--foreground)' }}
              required
            />
          </div>

          <button
            type="submit"
            disabled={loading}
            className="w-full py-4 font-bold rounded-2xl transition-all hover:brightness-110 active:scale-[0.97] mt-4 shadow-lg flex items-center justify-center gap-2"
            style={{
                backgroundColor: 'var(--primary)',
                color: 'var(--primary-foreground)',
                opacity: loading ? '0.7' : '1'
            }}
          >
            {loading ? <Loader2 className="animate-spin" size={20} /> : 'Update Password'}
          </button>
        </form>

        <div className="mt-8 pt-6 border-t text-center" style={{ borderColor: 'var(--border)' }}>
          <button 
            // onClick={() => router.push("/ads-center/auth")}
            onClick={()=> router.push("/settings")}
            className="text-xs font-bold uppercase tracking-widest opacity-50 hover:opacity-100 flex items-center justify-center gap-2 mx-auto transition-all"
          >
            <ArrowLeft size={14} /> Back to Sign In
          </button>
        </div>
      </div>
    </PageWrapper>
  )
}
